package com.ai.osmos.ads.renderer

import com.ai.osmos.ads.fetcher.AdFetcherSDK
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.TargetingParams
import com.ai.osmos.utils.common.JSONParsing
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.StandardErrorHandler

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: AdRenderer
 */

/**
 * The [AdRenderer] class is responsible for fetching and preparing ad data for rendering.
 * It acts as a bridge between the ad fetcher module and UI-level ad views.
 */
internal class AdRenderer(private val config: Config) : AdRendererInterface {
    private val adFetcherSDK = AdFetcherSDK(config)

    /**
     * Common method to fetch display ads with ad units.
     * Reduces code duplication across banner, carousel, and multi-ad carousel methods.
     */
    private suspend fun fetchDisplayAdsWithAuCommon(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        productCount: Int = 1,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?,
        isMultiAd: Boolean = false
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adUnits = arrayListOf(adUnit)
            val adData = adFetcherSDK.fetchDisplayAdsWithAu(
                cliUbid = cliUbid,
                pageType = pageType,
                productCount = productCount,
                adUnits = adUnits,
                targetingParams = targetingParams,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            getAdDataMap(adData.toMutableMap(), cliUbid, isMultiAd)
        }
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams(optional): Filter params that you want to apply
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     * Example:
     * ```
     * val adData = adRender.fetchBannerAdsWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnit = "banner_ads",
     * ```
     */
    override suspend fun fetchBannerAdsWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return fetchDisplayAdsWithAuCommon(
            cliUbid = cliUbid,
            pageType = pageType,
            adUnit = adUnit,
            productCount = 1,
            targetingParams = targetingParams,
            errorCallback = errorCallback,
            isMultiAd = false
        )
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams(optional): Filter params that you want to apply
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *Example:
     * ```
     * val adData = adRender.fetchCarouselAdWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnit = "carousel_ads",
     * ```
     */
    override suspend fun fetchCarouselAdWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return fetchDisplayAdsWithAuCommon(
            cliUbid = cliUbid,
            pageType = pageType,
            adUnit = adUnit,
            productCount = 1,
            targetingParams = targetingParams,
            errorCallback = errorCallback,
            isMultiAd = false
        )
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param productCount: No of ads responded. MAX 10.
     * @param targetingParams(optional): Filter params that you want to apply
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     * Example:
     * ```
     * val adData = adRender.fetchMultiAdCarouselWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnit = "all_in_one",
     *                 productCount = 3,
     * ```
     */
    override suspend fun fetchMultiAdCarouselWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        productCount: Int,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return fetchDisplayAdsWithAuCommon(
            cliUbid = cliUbid,
            pageType = pageType,
            adUnit = adUnit,
            productCount = productCount,
            targetingParams = targetingParams,
            errorCallback = errorCallback,
            isMultiAd = true
        )
    }

    private fun getAdDataMap(
        adData: MutableMap<String, Any>,
        cli_ubid: String,
        isMultiAd: Boolean
    ): Map<String, Any> {
        if (adData.isEmpty()) return emptyMap()

        return StandardErrorHandler.executeWithParsingErrorHandling {
            val adMap = JSONParsing.parseAdData(adData, cli_ubid)

            if (!isMultiAd) {
                val firstAd = adMap.values.firstOrNull()?.firstOrNull()
                if (firstAd != null) {
                    mapOf("ad" to firstAd)
                } else {
                    emptyMap()
                }
            } else {
                val allAds = adMap.values.flatten()
                if (allAds.isNotEmpty()) {
                    mapOf("ad" to allAds)
                } else {
                    emptyMap()
                }
            }
        }
    }


}