package com.ai.osmos.ads.renderer.loaders

import android.app.Dialog
import android.content.Context
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.ImageButton
import android.widget.LinearLayout
import com.ai.osmos.ads.views.Coordinates
import com.ai.osmos.models.ads.AdType
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: InterstitialLoader
 */

/**
 * Responsible for rendering and displaying popup-style ads (image/video).
 *
 * @property coroutineScope Coroutine scope for async tasks like loading content or delaying close button.
 * @property adTracker Used to track impressions or clicks for the ad.
 */
class InterstitialLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface,
) {
    private val imageLoader = ImageLoader(coroutineScope, adTracker)
    private val videoLoader = VideoLoader(coroutineScope, adTracker)
    private var currentDialog: Dialog? = null
    private val jobs = mutableListOf<Job>()
    private var errorCallback: ErrorCallback? = null

    /**
     * Set an error callback to handle errors that occur during interstitial ad loading or playback.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
        imageLoader.setErrorCallback(callback)
        videoLoader.setErrorCallback(callback)
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
        imageLoader.clearErrorCallback()
        videoLoader.clearErrorCallback()
    }

    /**
     * Logs errors consistently across the InterstitialLoader functionality.
     * Creates an ExceptionHandler with structured error information and propagates
     * the error through the errorCallback if available.
     *
     * @param message The error message to be logged, describing the issue encountered.
     * @param errorType The specific error type from the OsmosError enum.
     * @param throwable The original throwable that caused the error (optional).
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Displays a popup dialog containing the given ad content.
     *
     * This function is responsible for showing a popup (image or video ad) in the given context.
     * The ad can be positioned either by gravity (e.g., "CENTER", "BOTTOM") or using absolute (x, y) coordinates.
     *
     * @param context The Android context used for creating views and dialogs. Typically, this would be an Activity or Application context.
     * @param ad The [BaseAd] to be rendered in the popup. This can be an [ImageAd] or [VideoAd], based on the type of ad passed.
     * @param cliUbid The cli_ubid used for tracking impressions and ad clicks.
     * @param height Optional height for the popup content. This is mostly used for video ads to specify their size. Ignored if width and height aren't provided.
     * @param width Optional width for the popup content. Similar to height, it’s used for video ads to define the ad's dimensions.
     * @param alignment Optional alignment string ("CENTER" or "BOTTOM") used to position the popup within the available space.
     *                 If specific (x, y) coordinates are set, this alignment is ignored.
     * @param coordinates Optional object containing the X and Y coordinates for absolute positioning of the popup.
     * @param adClickListener Optional listener that is triggered when the ad is clicked.
     * @param closeButtonSec Optional duration (in seconds) for when the close button will be visible.
     */
    fun showPopup(
        context: Context,
        ad: BaseAd,
        cliUbid: String,
        height: Int,
        width: Int,
        alignment: Int? = Gravity.CENTER,
        coordinates: Coordinates? = Coordinates(0, 0),
        isClamped: Boolean,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null,
        closeButtonSec: Int? = null
    ) {
        try {
            val finalWidth = if (isClamped) width else ViewUtils.dpToPx(context, width)
            val finalHeight = if (isClamped) height else ViewUtils.dpToPx(context, height)

            val dialog = Dialog(context).apply {
                window?.apply {
                    setLayout(finalWidth, finalHeight)
                    setBackgroundDrawableResource(android.R.color.transparent)

                    val params = attributes
                    if (coordinates != null) {
                        if (coordinates.x != 0 && coordinates.y != 0) {
                            params.gravity = Gravity.TOP or Gravity.START
                            params.x = coordinates.x
                            params.y = coordinates.y
                        } else {
                            setGravity(if (alignment == Gravity.BOTTOM || alignment == Gravity.CENTER) alignment else Gravity.CENTER)
                        }
                    }
                    attributes = params
                }
                setCancelable(false)
            }

            val layout = LinearLayout(context).apply {
                orientation = LinearLayout.VERTICAL
                gravity = Gravity.CENTER
            }
            layout.layoutParams = LinearLayout.LayoutParams(finalWidth, finalHeight)

            val closeButton = ImageButton(context).apply {
                setImageResource(android.R.drawable.ic_menu_close_clear_cancel)
                background = null
                visibility = View.INVISIBLE
                layoutParams = LinearLayout.LayoutParams(
                    ViewGroup.LayoutParams.WRAP_CONTENT,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                ).apply {
                    gravity = Gravity.END
                }
                setOnClickListener {
                    dialog.dismiss()
                }
            }

            layout.addView(closeButton)

            val job = coroutineScope.launch {
                delay(
                    (closeButtonSec ?: Constants.CLOSE_BUTTON_SEC_DEFAULT) * 1000L
                ) // delay expects milliseconds as Long
                closeButton.visibility = View.VISIBLE
            }
            jobs.add(job)

            when (ad.elementsType) {

                AdType.IMAGE -> {
                    val imageAd = ad as ImageAd
                    val imageView = imageLoader.createImageView(
                        context,
                        width,
                        height,
                        isClamped = isClamped
                    )

                    layout.addView(imageView)
                    dialog.setContentView(layout)

                    imageLoader.loadImage(imageAd, imageView, cliUbid, false) {
                        adClickListener?.invoke(imageAd.toMap())
                    }
                }

                AdType.VIDEO -> {
                    val videoAd = ad as VideoAd
                    val videoView = videoLoader.createVideoView(
                        context,
                        videoAd,
                        isClamped,
                        width,
                        height,
                        cliUbid
                    ) {
                        adClickListener?.invoke(videoAd.toMap()) // Pass adMetadata when clicked
                    }
                    layout.addView(videoView)
                    dialog.setContentView(layout)
                }

                else -> {
                    errorLog(
                        "Unsupported AdType: ${ad.elementsType}",
                        OsmosError.CONFIGURATION_ERROR
                    )
                    // Cancel the close button job since we're returning early
                    job.cancel()
                    jobs.remove(job)
                    return
                }
            }
            currentDialog = dialog

            // Add dialog dismiss listener to clean up resources
            dialog.setOnDismissListener {
                currentDialog = null
                // Clean up video resources when dialog is dismissed (without dismissing dialog again)
                cleanupResources()
            }

            val activity = (context as? android.app.Activity)
            if (activity != null && !activity.isFinishing && !activity.isDestroyed) {
                dialog.show()
            } else {
                // Cancel the close button job since dialog won't be shown
                job.cancel()
                jobs.remove(job)
            }
        } catch (e: Exception) {
            errorLog("Failed to show interstitial popup: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    private fun cleanupResources() {
        try {
            // Cancel all running jobs
            jobs.forEach { it.cancel() }
            jobs.clear()

            // Release video players with dead thread protection
            videoLoader.safeReleaseAllPlayers()

            // Clear image loader listener
            imageLoader.setViewLoadListener { _, _ -> }

        } catch (e: Exception) {
            errorLog("Error during resource cleanup: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    fun cleanup() {
        try {
            currentDialog?.dismiss()
            currentDialog = null
            cleanupResources()

        } catch (e: Exception) {
            errorLog("Error during cleanup: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }
}
