package com.ai.osmos.ads.renderer.loaders

import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.drawable.GradientDrawable
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.FrameLayout
import android.widget.ImageButton
import android.widget.ImageView
import androidx.lifecycle.LifecycleOwner
import androidx.media3.common.MediaItem
import androidx.media3.common.Player
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.ui.PlayerView
import com.ai.osmos.ads.renderer.VideoPlayerLifecycleObserver
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.models.events.VideoActionType
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.isVisibleInScreen
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicBoolean

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: VideoLoader
 */
class VideoLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface
) {
    private var errorCallback: ErrorCallback? = null
    private lateinit var mButton: ImageButton
    private var isViewVisible = false
    private var wasPlaying = false

    // Thread-safe maps with synchronized access
    private val playerMap: MutableMap<PlayerView, ExoPlayer> = ConcurrentHashMap()
    private val muteStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val visibilityStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val wasPlayingStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val progressJobMap: MutableMap<PlayerView, Job> = ConcurrentHashMap()
    private val lifecycleObserverMap: MutableMap<ExoPlayer, VideoPlayerLifecycleObserver> =
        ConcurrentHashMap()

    // Synchronization locks for atomic operations
    private val stateLock = Any()
    private val playerLock = Any()

    private var onViewLoad: ((adMetadata: Map<String, Any>, String) -> Unit)? = null
    private val isReleased = AtomicBoolean(false)

    /**
     * Sets a listener to be notified when the ad view is fully loaded.
     *
     * @param listener Callback receiving the loaded [ImageAd] and its tracking ID.
     */
    fun setViewLoadListener(listener: (adMetadata: Map<String, Any>, String) -> Unit) {
        this.onViewLoad = { adMetadata, cliUbid ->
            listener(adMetadata, cliUbid)
        }
    }

    /**
     * Set an error callback to handle errors that occur during video loading or playback.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
    }

    /**
     * Logs an error message using structured error handling.
     *
     * This function uses the ExceptionHandler pattern to log errors with proper error codes
     * and calls the error callback if one is set.
     *
     * @param message The error message to be logged, describing the issue encountered.
     * @param errorType The specific error type from the OsmosError enum.
     * @param throwable The original throwable that caused the error (optional).
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Thread-safe cleanup of player state across all maps
     */
    private fun cleanupPlayerState(exoPlayer: ExoPlayer) {
        muteStateMap.remove(exoPlayer)
        visibilityStateMap.remove(exoPlayer)
        wasPlayingStateMap.remove(exoPlayer)
        lifecycleObserverMap.remove(exoPlayer)?.clearErrorCallback()
    }

    internal fun createVideoView(
        context: Context,
        videoAd: VideoAd,
        isClamped: Boolean,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null

    ): FrameLayout {
        val widthPx = if (isClamped) widthDp else ViewUtils.dpToPx(context, widthDp)
        val heightPx = if (isClamped) heightDp else ViewUtils.dpToPx(context, heightDp)
        val exoPlayer: ExoPlayer = ExoPlayer.Builder(context).build()
        val playerView = buildPlayerView(context, widthPx, heightPx)

        // Initialize states for this specific player
        muteStateMap[exoPlayer] = true  // default: muted
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing

        val muteButton = ImageButton(context).apply {
            val currentMuteState = muteStateMap[exoPlayer] ?: false
            setImageResource(
                getDrawableResource(
                    context,
                    if (currentMuteState) "mute" else "unmute"
                )
            )
            setColorFilter(Color.BLACK, PorterDuff.Mode.SRC_IN)
            scaleType = ImageView.ScaleType.FIT_CENTER
            adjustViewBounds = true
            background = GradientDrawable().apply {
                shape = GradientDrawable.OVAL
                setStroke(3, Color.BLACK)
                setColor(Color.WHITE)
            }

            layoutParams = FrameLayout.LayoutParams(
                ViewUtils.dpToPx(context, 24),
                ViewUtils.dpToPx(context, 24),
                Gravity.END or Gravity.BOTTOM
            ).apply {
                bottomMargin = ViewUtils.dpToPx(context, 50)

                marginEnd = ViewUtils.dpToPx(context, 16)
                marginStart = ViewUtils.dpToPx(context, 16)
            }
            visibility = View.GONE
            setOnClickListener { toggleMute(this, videoAd, cliUbid, exoPlayer) }
        }

        // Load video asynchronously
        loadVideoUsingUrl(context, videoAd, cliUbid, playerView, exoPlayer, muteButton)

        //Add Listeners
        addListeners(videoAd, cliUbid, playerView, adClickListener, exoPlayer)

        return buildContainer(context, widthPx, heightPx, playerView, muteButton)
    }

    internal fun createPIPVideoView(
        context: Context,
        videoAd: VideoAd,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null,
        exoPlayer: ExoPlayer
    ): PlayerView {

        val playerView = buildPlayerView(context, widthDp, heightDp)

        // Initialize states for this specific PIP player
        muteStateMap[exoPlayer] = true  // default: muted for PIP
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing

        // Load and play video
        loadVideoUsingUrl(context, videoAd, cliUbid, playerView, exoPlayer)

        addListeners(videoAd, cliUbid, playerView, adClickListener, exoPlayer)

        return playerView
    }

    internal fun createCarouselVideoView(
        context: Context,
        videoAd: VideoAd,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
    ): FrameLayout {

        val widthPx = ViewUtils.dpToPx(context, widthDp)
        val heightPx = ViewUtils.dpToPx(context, heightDp)
        val exoPlayer: ExoPlayer = ExoPlayer.Builder(context).build()
        val playerView = buildPlayerView(context, widthPx, heightPx)

        // Initialize states for this specific carousel player
        muteStateMap[exoPlayer] = true  // default: muted
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing

        val buttonSize = ViewUtils.dpToPx(context, 36)
        val currentMuteState = muteStateMap[exoPlayer] ?: false
        mButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = getDrawableResource(context, if (currentMuteState) "mute" else "unmute"),
            size = buttonSize,
            gravity = Gravity.BOTTOM or Gravity.RIGHT,
            marginBottom = ViewUtils.dpToPx(context, 40),
            marginEnd = ViewUtils.dpToPx(context, 10),
            onClick = {
                toggleMute(mButton, videoAd, cliUbid, exoPlayer)
            })
        mButton.visibility = View.GONE

        // Load video asynchronously
        loadVideoUsingUrl(context, videoAd, cliUbid, playerView, exoPlayer, mButton)

        playerView.viewTreeObserver.addOnScrollChangedListener {
            checkVisibilityAndHandlePlayback(playerView, exoPlayer)
        }

        playerView.viewTreeObserver.addOnPreDrawListener(object :
            ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                if (playerView.isVisibleInScreen()) {
                    onViewLoad?.invoke(videoAd.toMap(), cliUbid)
                    trackImpression(videoAd.uclid, cliUbid)
                    playerView.viewTreeObserver.removeOnPreDrawListener(this)
                }
                return true
            }
        })

        return buildContainer(context, widthPx, heightPx, playerView, mButton)
    }

    private fun buildPlayerView(context: Context, widthPx: Int, heightPx: Int): PlayerView {
        return PlayerView(context).apply {
            layoutParams = FrameLayout.LayoutParams(widthPx, heightPx, Gravity.CENTER)
            useController = false
        }
    }

    private fun buildContainer(
        context: Context,
        widthPx: Int,
        heightPx: Int,
        playerView: PlayerView,
        muteButton: View? = null
    ): FrameLayout {
        return FrameLayout(context).apply {
            layoutParams = ViewGroup.LayoutParams(widthPx, heightPx)
            addView(playerView, FrameLayout.LayoutParams(widthPx, heightPx).apply {
                gravity = Gravity.CENTER
            })
            muteButton?.let {
                addView(it, FrameLayout.LayoutParams(90, 90).apply {
                    gravity = Gravity.BOTTOM or Gravity.END
                    bottomMargin = 26
                })
            }
        }
    }


    private fun loadVideoUsingUrl(
        context: Context,
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer,
        muteButton: ImageButton? = null
    ) {
        val loadingJob = coroutineScope.launch(Dispatchers.IO) {  // Fetch video URL in background
            try {
                val videoUrl = videoAd.elements.value
                withContext(Dispatchers.Main) { // Switch to Main thread for ExoPlayer

                    exoPlayer.apply {
                        // Safest approach: Let ExoPlayer handle all format detection automatically
                        val mediaItem = MediaItem.Builder()
                            .setUri(videoUrl)
                            .build()
                        setMediaItem(mediaItem)
                        prepare()

                        addListener(
                            createPlayerListener(
                                videoAd,
                                cliUbid,
                                playerView,
                                exoPlayer,
                                muteButton
                            )
                        )
                        val playerMuteState = muteStateMap[exoPlayer] ?: false
                        try {
                            volume = if (playerMuteState) 0f else 1f
                        } catch (e: IllegalStateException) {
                            errorLog(
                                "Cannot set initial volume on released player: ${e.message}",
                                OsmosError.UNKNOWN,
                                e
                            )
                        }
                    }
                    playerMap[playerView] = exoPlayer
                    playerView.player = exoPlayer
                    exoPlayer.playWhenReady = true

                    // Attach lifecycle observer after player is added to map
                    if (context is LifecycleOwner) {
                        val observer = VideoPlayerLifecycleObserver(exoPlayer, errorCallback)
                        context.lifecycle.addObserver(observer)
                        lifecycleObserverMap[exoPlayer] = observer
                    }
                }

            } catch (e: Exception) {
                errorLog("Error loading video: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }

        // Store loading job for cleanup - using a temporary key until progress tracking starts
        progressJobMap[playerView] = loadingJob

    }

    private fun addListeners(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)?,
        exoPlayer: ExoPlayer
    ) {
        playerView.setOnClickListener {
            trackAdClick(videoAd.uclid, cliUbid)
            adClickListener?.invoke(videoAd.toMap())
        }

        playerView.viewTreeObserver.addOnScrollChangedListener {
            checkVisibilityAndHandlePlayback(playerView, exoPlayer)
        }

        playerView.viewTreeObserver.addOnPreDrawListener(object :
            ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                if (playerView.isVisibleInScreen()) {
                    onViewLoad?.invoke(videoAd.toMap(), cliUbid)
                    trackImpression(videoAd.uclid, cliUbid)
                    playerView.viewTreeObserver.removeOnPreDrawListener(this)
                }
                return true
            }
        })
    }


    private fun createPlayerListener(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer,
        muteButton: ImageButton? = null,
    ) = object : Player.Listener {

        override fun onPlaybackStateChanged(state: Int) {

            when (state) {
                Player.STATE_IDLE -> {}
                Player.STATE_BUFFERING -> {}
                Player.STATE_READY -> {
                    Log.d(
                        "VideoProgress",
                        "Progress: Start"
                    )
                    if (muteButton != null)
                        muteButton.visibility = View.VISIBLE

                    trackVideoProgressWithEvent(videoAd, cliUbid, playerView, exoPlayer)
                }

                Player.STATE_ENDED -> {
                    if (playerView.isVisibleInScreen()) {
                        val currentPosition = exoPlayer.currentPosition ?: 0L  // Current time (ms)
                        val totalDuration = exoPlayer.duration ?: 0L  // Total duration (ms)

                        Log.d(
                            "VideoProgress",
                            "Progress: End "
                        )

                        fireVideoProgressEvent(
                            videoAd.uclid,
                            cliUbid,
                            currentPosition,
                            totalDuration
                        )
                    }
                    exoPlayer?.seekTo(0)
                    exoPlayer?.playWhenReady = true
                }
            }
        }

        override fun onPlayerError(error: androidx.media3.common.PlaybackException) {
            val errorMessage = when (error.errorCode) {
                androidx.media3.common.PlaybackException.ERROR_CODE_IO_NETWORK_CONNECTION_FAILED ->
                    "Network connection failed while loading video"

                androidx.media3.common.PlaybackException.ERROR_CODE_IO_NETWORK_CONNECTION_TIMEOUT ->
                    "Network connection timeout while loading video"

                androidx.media3.common.PlaybackException.ERROR_CODE_PARSING_CONTAINER_MALFORMED ->
                    "Video file is corrupted or malformed"

                androidx.media3.common.PlaybackException.ERROR_CODE_PARSING_MANIFEST_MALFORMED ->
                    "Video manifest is corrupted or malformed"

                androidx.media3.common.PlaybackException.ERROR_CODE_DECODER_INIT_FAILED ->
                    "Video decoder initialization failed"

                androidx.media3.common.PlaybackException.ERROR_CODE_AUDIO_TRACK_INIT_FAILED ->
                    "Audio track initialization failed"

                androidx.media3.common.PlaybackException.ERROR_CODE_DRM_SYSTEM_ERROR ->
                    "DRM system error"

                else -> "Video playback error: ${error.message}"
            }

            val osmosError = when (error.errorCode) {
                androidx.media3.common.PlaybackException.ERROR_CODE_IO_NETWORK_CONNECTION_FAILED,
                androidx.media3.common.PlaybackException.ERROR_CODE_IO_NETWORK_CONNECTION_TIMEOUT ->
                    OsmosError.NETWORK_FAILURE

                androidx.media3.common.PlaybackException.ERROR_CODE_PARSING_CONTAINER_MALFORMED,
                androidx.media3.common.PlaybackException.ERROR_CODE_PARSING_MANIFEST_MALFORMED ->
                    OsmosError.UNSUPPORTED_VIDEO_FORMAT

                else -> OsmosError.UNKNOWN
            }

            errorLog(errorMessage, osmosError, error)
        }
    }


    private fun trackVideoProgressWithEvent(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer
    ) {
        val job = coroutineScope.launch(Dispatchers.Main) {
            val eventProgressTime =
                videoAd.elements.videoProgressSec ?: Constants.VIDEO_PROGRESS_SEC_DEFAULT

            var lastFiredSec = -eventProgressTime

            while (exoPlayer != null) {
                if (exoPlayer.isPlaying && playerView.isVisibleInScreen()) {
                    val currentPositionMs = exoPlayer.currentPosition ?: 0L
                    val currentSec = (currentPositionMs / 1000).toInt()

                    if (currentSec >= lastFiredSec + eventProgressTime) {
                        fireVideoProgressEvent(
                            videoAd.uclid,
                            cliUbid,
                            currentPositionMs,
                            exoPlayer.duration ?: 0L
                        )
                        lastFiredSec = currentSec
                    }
                }
                delay(1000) // Check every second
            }
        }

        // Cancel and replace any existing job (like loadingJob) for this playerView
        progressJobMap.remove(playerView)?.cancel()
        progressJobMap[playerView] = job
    }

    private fun fireVideoProgressEvent(
        uclid: String,
        cliUbid: String,
        currentPosition: Long,
        totalDuration: Long
    ) {
        trackVideoProgress(
            uclid,
            cliUbid,
            (currentPosition / 1000).toFloat(),
            (totalDuration / 1000).toFloat()
        )
    }

    private fun toggleMute(
        button: ImageButton,
        videoAd: VideoAd,
        cliUbid: String,
        exoPlayer: ExoPlayer
    ) {
        // Get current mute state for this specific player, default to false (unmuted)
        val currentMuteState = muteStateMap[exoPlayer] ?: false
        val newMuteState = !currentMuteState

        // Update mute state for this specific player
        muteStateMap[exoPlayer] = newMuteState

        val actionType = if (newMuteState) VideoActionType.MUTE else VideoActionType.UNMUTE
        exoPlayer?.volume = if (newMuteState) 0f else 1f
        button.setImageResource(
            getDrawableResource(
                button.context,
                if (newMuteState) "mute" else "unmute"
            )
        )

        val currentPosition = exoPlayer?.currentPosition ?: 0L

        //Call video action event for mute and unmute
        trackVideoAction(
            videoAd.uclid,
            cliUbid,
            actionType,
            (currentPosition / 1000).toFloat()
        )
    }

    private fun checkVisibilityAndHandlePlayback(playerView: PlayerView, exoPlayer: ExoPlayer) {
        val screenLocation = IntArray(2)
        playerView.getLocationOnScreen(screenLocation)

        val screenHeight = Resources.getSystem().displayMetrics.heightPixels
        val viewTop = screenLocation[1]
        val viewBottom = viewTop + playerView.height

        val isVisibleNow = viewTop < screenHeight && viewBottom > 0

        // Thread-safe state access and modification
        synchronized(stateLock) {
            // Get per-player visibility state
            val wasVisible = visibilityStateMap[exoPlayer] ?: false
            val wasPlayerPlaying = wasPlayingStateMap[exoPlayer] ?: false

            if (isVisibleNow && !wasVisible) {
                // Video became visible
                visibilityStateMap[exoPlayer] = true
                if (wasPlayerPlaying) {
                    val playerMuteState = muteStateMap[exoPlayer] ?: false
                    try {
                        exoPlayer.volume = if (playerMuteState) 0f else 1f
                        exoPlayer.play()
                    } catch (e: IllegalStateException) {
                        errorLog(
                            "Cannot control released player on visible: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                        // Clean up state for released player
                        cleanupPlayerState(exoPlayer)
                    }
                } else if (exoPlayer.playWhenReady && !exoPlayer.isPlaying) {
                    val playerMuteState = muteStateMap[exoPlayer] ?: false
                    try {
                        exoPlayer.volume = if (playerMuteState) 0f else 1f
                        wasPlayingStateMap[exoPlayer] = true
                    } catch (e: IllegalStateException) {
                        errorLog(
                            "Cannot set volume on released player: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                        // Clean up state for released player
                        cleanupPlayerState(exoPlayer)
                    }
                }
            } else if (!isVisibleNow && wasVisible) {
                // Video went out of view
                visibilityStateMap[exoPlayer] = false
                if (exoPlayer.isPlaying == true) {
                    try {
                        exoPlayer.pause()
                        // Don't change volume here - preserve individual mute state
                        // The volume should remain as per the player's mute state
                        wasPlayingStateMap[exoPlayer] = true
                    } catch (e: IllegalStateException) {
                        errorLog(
                            "Cannot pause released player: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                        // Clean up state for released player
                        cleanupPlayerState(exoPlayer)
                    }
                }
            }
        }
    }

    fun safeStopAndReleasePlayer(playerView: PlayerView?) {
        if (isReleased.get()) {
            // Bulk release already happened - don't proceed
            return
        }

        playerView?.let { pv ->
            try {
                pv.player?.apply {
                    playWhenReady = false
                    stop()
                    clearVideoSurface()
                }
            } catch (e: IllegalStateException) {
                // Player already released - continue with cleanup
            }
            releasePlayer(pv)
        }
    }

    fun releasePlayer(playerView: PlayerView?) {
        playerView?.let {
            try {
                progressJobMap.remove(it)?.cancel()

                // Thread-safe player release
                synchronized(playerLock) {
                    val player = playerMap.remove(it)

                    it.player = null
                    player?.apply {
                        playWhenReady = false
                        stop()
                        clearVideoSurface()

                        // Thread-safe state cleanup
                        synchronized(stateLock) {
                            cleanupPlayerState(this)
                        }

                        release()
                    }
                }

            } catch (e: Exception) {
                errorLog("releasePlayer failed: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }
    }


    fun pause(playerView: PlayerView) {
        val player = playerView.player
        player?.pause()
    }

    fun playExclusive(playerView: PlayerView) {
        // Thread-safe exclusive playback
        synchronized(playerLock) {
            // Pause all others
            for ((view, player) in playerMap) {
                if (view != playerView) {
                    try {
                        player.pause()
                    } catch (e: IllegalStateException) {
                        errorLog(
                            "Cannot pause released player during exclusive play: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                    }
                }
            }
            // Play the selected one
            try {
                playerMap[playerView]?.play()
            } catch (e: IllegalStateException) {
                errorLog(
                    "Cannot play released player during exclusive play: ${e.message}",
                    OsmosError.UNKNOWN,
                    e
                )
            }
        }
    }

    /**
     * Clear the view load listener to prevent memory leaks
     */
    private fun clearViewLoadListener() {
        onViewLoad = null
    }

    /**
     * Clear all callbacks to prevent memory leaks
     */
    private fun clearAllCallbacks() {
        onViewLoad = null
        errorCallback = null
    }


    /**
     * Safe version of releaseAllPlayers that handles dead thread scenarios
     */
    fun safeReleaseAllPlayers() {
        if (isReleased.compareAndSet(false, true)) {
            try {
                releaseAllPlayers()
            } catch (e: IllegalStateException) {
                errorLog(
                    "Handler thread is dead, performing minimal cleanup: ${e.message}",
                    OsmosError.UNKNOWN,
                    e
                )
                // Just clear the maps without UI operations - thread-safely
                try {
                    synchronized(playerLock) {
                        progressJobMap.clear()
                        playerMap.clear()
                    }
                    synchronized(stateLock) {
                        muteStateMap.clear()
                        visibilityStateMap.clear()
                        wasPlayingStateMap.clear()
                        // Clear error callbacks from lifecycle observers before clearing the map
                        lifecycleObserverMap.values.forEach { it.clearErrorCallback() }
                        lifecycleObserverMap.clear()
                    }
                    clearAllCallbacks()
                } catch (e2: Exception) {
                    errorLog("Error during minimal cleanup: ${e2.message}", OsmosError.UNKNOWN, e2)
                }
            } catch (e: Exception) {
                errorLog("Error during safe release: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }
    }

    private fun releaseAllPlayers() {
        try {
            // Cancel progress jobs with dead thread protection
            try {
                progressJobMap.values.forEach { it.cancel() }
                progressJobMap.clear()
            } catch (e: IllegalStateException) {
                progressJobMap.clear()
            } catch (e: Exception) {
                progressJobMap.clear()
            }

            clearViewLoadListener()

            // Clear player references with dead thread protection
            try {
                playerMap.keys.forEach { playerView ->
                    try {
                        playerView.player = null
                    } catch (e: IllegalStateException) {
                        // Handler thread might be dead, skip UI operation
                        errorLog(
                            "Handler thread is dead, skipping playerView.player = null: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                    } catch (e: Exception) {
                        errorLog(
                            "Error setting playerView.player = null: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                    }
                }
            } catch (e: IllegalStateException) {
                errorLog(
                    "Handler thread is dead, skipping all playerView operations: ${e.message}",
                    OsmosError.UNKNOWN,
                    e
                )
            } catch (e: Exception) {
                errorLog("Error iterating playerMap: ${e.message}", OsmosError.UNKNOWN, e)
            }

            val playersToRelease = playerMap.toMap()

            playersToRelease.forEach { (playerView, player) ->
                try {
                    playerView.player = null
                    player.playWhenReady = false
                    player.stop()
                    player.clearVideoSurface()
                    player.clearVideoSurface()
                    player.release()
                } catch (e: Exception) {
                    errorLog(
                        "Error releasing individual player: ${e.message}",
                        OsmosError.UNKNOWN,
                        e
                    )
                }
            }
            playerMap.clear()
            muteStateMap.clear()  // Clean up all mute states
            visibilityStateMap.clear()  // Clean up all visibility states
            wasPlayingStateMap.clear()  // Clean up all playing states
            // Clear error callbacks from lifecycle observers before clearing the map
            lifecycleObserverMap.values.forEach { it.clearErrorCallback() }
            lifecycleObserverMap.clear()  // Clean up lifecycle observer references
        } catch (e: Exception) {
            errorLog("Error in releaseAllPlayers: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    /**
     * Helper method to track impressions using the common interface.
     */
    private fun trackImpression(uclid: String, cliUbid: String) {
        adTracker.trackImpression(uclid, cliUbid)
    }

    /**
     * Helper method to track ad clicks using the common interface.
     */
    private fun trackAdClick(uclid: String, cliUbid: String) {
        adTracker.trackAdClick(uclid, cliUbid)
    }

    /**
     * Helper method to track video progress events using the common interface.
     */
    private fun trackVideoProgress(
        uclid: String,
        cliUbid: String,
        videoViewSec: Float,
        videoDurationSec: Float
    ) {
        adTracker.videoProgressEvent(uclid, cliUbid, videoViewSec, videoDurationSec)
    }

    /**
     * Helper method to track video action events using the common interface.
     */
    private fun trackVideoAction(
        uclid: String,
        cliUbid: String,
        actionType: VideoActionType,
        videoViewSec: Float
    ) {
        adTracker.videoActionClick(uclid, cliUbid, actionType, videoViewSec)
    }

    /**
     * Helper function to get drawable resource ID by name for library projects
     */
    private fun getDrawableResource(context: Context, resourceName: String): Int {
        return context.resources.getIdentifier(resourceName, "drawable", context.packageName)
    }
}
