package com.ai.osmos.ads.views.managers

import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.views.Coordinates
import com.ai.osmos.ads.views.InterstitialAdView
import com.ai.osmos.core.Config
import com.ai.osmos.utils.error.ErrorCallback


/**
 * Configuration parameters for interstitial ad creation.
 *
 * @param width Optional width of interstitial ad view in pixels
 * @param height Optional height of interstitial ad view in pixels
 * @param alignment Optional alignment for positioning (e.g., Gravity.CENTER)
 * @param coordinates Optional coordinates for absolute positioning
 * @param attrs Optional attribute set for view styling
 * @param errorCallback Optional callback for error handling
 */
data class InterstitialAdSettings(
    val width: Int? = null,
    val height: Int? = null,
    val alignment: Int? = Gravity.CENTER,
    val coordinates: Coordinates? = null,
    val attrs: AttributeSet? = null,
    val errorCallback: ErrorCallback? = null
)

class InterstitialAdViewManager(private val config: Config) {

    /**
     * Shows an interstitial ad view with optional parameters.
     *
     * @param context Android context for view creation
     * @param ad Map containing the parsed ad data
     * @param adViewSettings InterstitialAdSettings with optional configuration
     * @return View containing the interstitial ad, or null if creation fails
     *
     * Example:
     * ```
     * val interstitialView = interstitialManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = InterstitialAdSettings(
     *         width = 320,
     *         height = 250,
     *         coordinates = Coordinates(x = 50, y = 40)
     *     )
     * )
     * ```
     */
    fun showAd(
        context: Context,
        ad: Map<String, Any>,
        adViewSettings: InterstitialAdSettings
    ): View? {
        return try {
            val interstitialAdView = InterstitialAdView(
                context = context,
                adData = ad,
                width = adViewSettings.width,
                height = adViewSettings.height,
                alignment = adViewSettings.alignment,
                coordinates = adViewSettings.coordinates,
                attrs = adViewSettings.attrs,
                errorCallback = adViewSettings.errorCallback,
                config = config
            )
            interstitialAdView.getView()
        } catch (e: Exception) {
            adViewSettings.errorCallback?.onError(
                "INTERSTITIAL_CREATION_FAILED",
                e.message ?: "Unknown error",
                e
            )
            null
        }
    }
}