package com.ai.osmos.core

import com.ai.osmos.ads.fetcher.AdFetcherSDK
import com.ai.osmos.ads.fetcher.AdFetcherSDKInterface
import com.ai.osmos.ads.renderer.AdRenderer
import com.ai.osmos.ads.renderer.AdRendererInterface
import com.ai.osmos.ads.views.managers.BannerAdViewManager
import com.ai.osmos.ads.views.managers.CarouselAdViewManager
import com.ai.osmos.ads.views.managers.InterstitialAdViewManager
import com.ai.osmos.ads.views.managers.MultiAdCarouselViewManager
import com.ai.osmos.ads.views.managers.PIPAdViewManager
import com.ai.osmos.tracking.events.RegisterEvent
import com.ai.osmos.tracking.events.RegisterEventInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.logging.DebugLogger
import java.lang.ref.WeakReference

/**
 * Main SDK class for OSMOS advertising platform.
 * Provides access to ad managers for different ad formats and core SDK functionality.
 */
public class OsmosSDK(private val config: Config) {

    init {
        // Set global debug flag when SDK instance is created
        DebugLogger.setDebugEnabled(config.debug)
    }

    private val _registerEvent: RegisterEvent by lazy {
        RegisterEvent(config)
    }

    private val _adFetcherSDK: AdFetcherSDK by lazy {
        AdFetcherSDK(config)
    }

    private val _adRenderer: AdRenderer by lazy {
        AdRenderer(config)
    }

    /**
     * Provides access to event registration functionality.
     * @return RegisterEventInterface for tracking events
     */
    fun registerEvent(): RegisterEventInterface = _registerEvent

    /**
     * Provides access to ad fetching functionality.
     * @return AdFetcherSDKInterface for retrieving advertisements
     */
    fun adFetcherSDK(): AdFetcherSDKInterface = _adFetcherSDK

    /**
     * Provides access to ad rendering functionality.
     * @return AdRendererInterface for displaying advertisements
     */
    fun adRenderer(): AdRendererInterface = _adRenderer

    /**
     * Creates a BannerAdViewManager for displaying banner advertisements.
     *
     * @return BannerAdViewManager instance configured with current SDK settings
     *
     * Example:
     * ```
     * val bannerManager = osmosSDK.bannerAdView()
     * val bannerView = bannerManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = BannerAdSettings(width = 320, height = 250)
     * )
     * parentLayout.addView(bannerView)
     * ```
     */
    fun bannerAdView(): BannerAdViewManager {
        return BannerAdViewManager(config)
    }

    /**
     * Creates an InterstitialAdViewManager for displaying full-screen overlay advertisements.
     *
     * @return InterstitialAdViewManager instance configured with current SDK settings
     *
     * Example:
     * ```
     * val interstitialManager = osmosSDK.interstitialAdView()
     * val interstitialView = interstitialManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = InterstitialAdSettings(
     *         alignment = Gravity.CENTER,
     *         coordinates = Coordinates(x = 50, y = 40)
     *     )
     * )
     * parentLayout.addView(interstitialView)
     * ```
     */
    fun interstitialAdView(): InterstitialAdViewManager {
        return InterstitialAdViewManager(config)
    }

    /**
     * Creates a CarouselAdViewManager for displaying swipeable carousel advertisements.
     *
     * @return CarouselAdViewManager instance configured with current SDK settings
     *
     * Example:
     * ```
     * val carouselManager = osmosSDK.carouselAdView()
     * val carouselView = carouselManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = CarouselAdSettings(width = 320, height = 250)
     * )
     * parentLayout.addView(carouselView)
     * ```
     */
    fun carouselAdView(): CarouselAdViewManager {
        return CarouselAdViewManager(config)
    }

    /**
     * Creates a MultiAdCarouselViewManager for displaying multiple advertisements in a carousel format.
     *
     * @return MultiAdCarouselViewManager instance configured with current SDK settings
     *
     * Example:
     * ```
     * val multiCarouselManager = osmosSDK.multiAdCarouselView()
     * val multiCarouselView = multiCarouselManager.showAd(
     *     context = context,
     *     ad = adData,
     *     adViewSettings = MultiAdCarouselSettings(width = 320, height = 250)
     * )
     * parentLayout.addView(multiCarouselView)
     * ```
     */
    fun multiAdCarouselView(): MultiAdCarouselViewManager {
        return MultiAdCarouselViewManager(config)
    }

    /**
     * Creates a PIPAdViewManager for displaying picture-in-picture advertisements.
     *
     * @return PIPAdViewManager instance configured with current SDK settings
     *
     * Example:
     * ```
     * val pipManager = osmosSDK.pipAdView()
     * val pipView = pipManager.showAd(
     *     activity = activity,
     *     ad = adData,
     *     adViewSettings = PIPAdSettings(
     *         coordinates = Coordinates(x = 50, y = 40)
     *     )
     * )
     * parentLayout.addView(pipView)
     * ```
     */
    fun pipAdView(): PIPAdViewManager {
        return PIPAdViewManager(config)
    }

    companion object {
        @Volatile
        private var sdkInstance: OsmosSDK? = null

        /**
         * Sets the client ID for the SDK (required).
         * @param clientId Unique client identifier
         * @return Builder instance for method chaining
         */
        fun clientId(clientId: String): Builder {
            return Builder().clientId(clientId)
        }

        /**
         * Builder class for SDK configuration.
         * Each builder instance maintains isolated configuration state.
         */
        class Builder {
            // Instance-specific builder properties (not static/shared)
            private var builderClientId: String? = null
            private var builderDebug: Boolean = false
            private var builderProductAdsHost: String = "osx-pla.o-s.io"
            private var builderDisplayAdsHost: String = "osx-ba.o-s.io"
            private var builderEventTrackingHost: String = "osx.o-s.io"
            private var builderErrorCallback: WeakReference<ErrorCallback>? = null

            /**
             * Sets the client ID for the SDK (required).
             * @param clientId Unique client identifier
             * @return Builder instance for method chaining
             */
            fun clientId(clientId: String): Builder {
                builderClientId = clientId
                return this
            }
            /**
             * Sets the product ads host (optional).
             * @param host Product ads host URL
             * @return Builder instance for method chaining
             */
            fun productAdsHost(host: String): Builder {
                builderProductAdsHost = host
                return this
            }

            /**
             * Sets the display ads host (optional).
             * @param host Display ads host URL
             * @return Builder instance for method chaining
             */
            fun displayAdsHost(host: String): Builder {
                builderDisplayAdsHost = host
                return this
            }

            /**
             * Sets the event tracking host (optional).
             * @param host Event tracking host URL
             * @return Builder instance for method chaining
             */
            fun eventTrackingHost(host: String): Builder {
                builderEventTrackingHost = host
                return this
            }

            /**
             * Enables or disables debug mode (optional).
             * @param debug True to enable debug logging, false otherwise
             * @return Builder instance for method chaining
             */
            fun debug(debug: Boolean = true): Builder {
                builderDebug = debug
                return this
            }

            /**
             * Sets the error callback for handling initialization errors (optional).
             * @param callback Callback to handle error scenarios
             * @return Builder instance for method chaining
             */
            fun errorCallback(callback: WeakReference<ErrorCallback>): Builder {
                builderErrorCallback = callback
                return this
            }

            /**
             * Validates the configuration and initializes the SDK as singleton.
             * @throws IllegalStateException if required parameters are missing
             *
             * Example:
             * ```
             * OsmosSDK.clientId("your_client_id")
             *     .productAdsHost("product-ads.example.com")  // optional
             *     .displayAdsHost("display-ads.example.com")  // optional
             *     .eventTrackingHost("events.example.com")    // optional
             *     .debug(true)                                 // optional
             *     .buildGlobalInstance()
             * ```
             */
            fun buildGlobalInstance() {
                val finalClientId =
                    builderClientId ?: throw IllegalStateException("Client ID is required")

                val config = Config(
                    clientId = finalClientId,
                    debug = builderDebug,
                    displayAdsHost = "https://$builderDisplayAdsHost",
                    productAdsHost = "https://$builderProductAdsHost",
                    eventTrackingHost = "https://$builderEventTrackingHost"
                )

                initialize(config, builderErrorCallback)
            }

            /**
             * Validates the configuration and creates a new SDK instance.
             * @return New OsmosSDK instance with the configured settings
             * @throws IllegalStateException if required parameters are missing
             *
             * Example:
             * ```
             * val sdk = OsmosSDK.clientId("your_client_id")
             *     .debug(true)
             *     .build()
             * ```
             */
            fun build(): OsmosSDK {
                val finalClientId =
                    builderClientId ?: throw IllegalStateException("Client ID is required")

                val config = Config(
                    clientId = finalClientId,
                    debug = builderDebug,
                    displayAdsHost = "https://$builderDisplayAdsHost",
                    productAdsHost = "https://$builderProductAdsHost",
                    eventTrackingHost = "https://$builderEventTrackingHost"
                )
                return OsmosSDK(config)
            }
        }

        /**
         * Initializes the OSMOS SDK with the provided configuration.
         * This method must be called before using any SDK functionality.
         *
         * @param config SDK configuration containing client credentials and settings
         * @param errorCallback Optional callback for handling initialization errors
         *
         * Example:
         * ```
         * OsmosSDK.initialize(
         *     config = Config(
         *         clientId = "your_client_id",
         *         debug = BuildConfig.DEBUG
         *     )
         * )
         * ```
         */
        private fun initialize(
            config: Config,
            errorCallback: WeakReference<ErrorCallback>? = null
        ) {
            // Create singleton instance with provided config
            if (sdkInstance == null) {
                synchronized(this) {
                    if (sdkInstance == null) {
                        sdkInstance = OsmosSDK(config)
                    }
                }
            }
        }

        /**
         * Singleton instance of OSMOS SDK.
         *
         * @return The initialized OsmosSDK instance
         * @throws IllegalStateException if SDK has not been initialized
         *
         * Example:
         * ```
         * val osmosSDK = OsmosSDK.globalInstance()
         * val bannerManager = osmosSDK.bannerAdView()
         * ```
         */
        fun globalInstance(): OsmosSDK {
            return sdkInstance ?: throw IllegalStateException(
                "OSMOS SDK not initialized. Please initialize OSMOS SDK using `OsmosSDK.buildGlobalInstance()`"
            )
        }

        /**
         * Shuts down the OSMOS SDK and releases all resources.
         * This method should be called when the SDK is no longer needed to prevent memory leaks.
         * After calling this method, initialize() must be called again before using the SDK.
         *
         * Example:
         * ```
         * // When app is shutting down or SDK is no longer needed
         * OsmosSDK.shutdown()
         * ```
         */
        fun shutdown() {
            // Clear singleton instance
            sdkInstance = null
        }
    }

}
