package com.ai.osmos.tracking.events

import com.ai.osmos.core.Config
import com.ai.osmos.models.events.EventName
import com.ai.osmos.models.events.TrackingParams
import com.ai.osmos.models.events.VideoActionType
import com.ai.osmos.network.WebConstants
import com.ai.osmos.network.WebService
import com.ai.osmos.utils.common.ApiEndpoint
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.common.HttpMethod
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.StandardErrorHandler
import com.ai.osmos.utils.logging.DebugLogger

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: RegisterEvent
 */

/**
 * Represents a product with relevant details for tracking events.
 *
 * @property sku_id Unique identifier of the product.
 * @property product_prices Optional: Price details of the product.
 * @property product_quantities Optional: Quantity details of the product.
 * @property merchant_id Optional: Identifier of the merchant selling the product.
 */
data class Product(
    val skuId: String,
    val price: Float,
    val quantity: Int,
    val merchantId: String? = null // Optional
)

/**
 * Handles event tracking for different user actions such as product views,
 * add-to-cart, purchases, ad impressions, and ad clicks.
 */
internal class RegisterEvent(private val config: Config) : RegisterEventInterface {

    // Instance of WebService to handle network requests
    private val webService = WebService()


    /**
     * Processes event tracking by making an API call to the specified URL with event parameters.
     * This function constructs the full API URL using `WebConstants.buildUrl()` and sends
     * a POST/GET request via `webService.callApi()`. If debugging is enabled, the parameters
     * and response are logged for debugging purposes.
     *
     * @param urlPath The endpoint URL path where the event data will be sent.
     * @param params A map containing the event tracking parameters.
     * @return The API response as a Map<String,Any>.
     *
     */
    private suspend fun trackEventsProcessor(
        urlPath: String,
        params: Map<String, Any>,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>? {
        DebugLogger.log("API_URL $urlPath")
        DebugLogger.log("$params")

        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            // Construct the API URL with the given parameters
            val apiUrl = WebConstants.buildUrl(urlPath, params)

            // Make the API call to track the event using exception-throwing method
            webService.callApiWithException(
                apiUrl,
                HttpMethod.POST,
                requestHeaders = null,
                requestBody = null
            )
        }
    }

    /**
     * Registers an ad click event for tracking user interactions with advertisements.
     * This function constructs a query parameter map containing the `uclid` and any additional parameters.
     * It then calls `track_events_processor` to send the event to the ad click tracking endpoint.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the ad click event.
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val res = registerEvent.registerAdClickEvent(
     *             cliUbid = "c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd",
     *             uclid = "2|haie5ibhks4fkd3a3a36oy9zp23i8ton|0.0001|1680792159968|2777510___Astore___Anet|BRAND_AD_TAG|c27b9ad197765dc9ba51e5b7fb9d5c43eb8f1f7b199367af54c74705424558dd|CPM|whitakers|522130|411094|437705|437705||15211|8816|",
     *         )
     * ```
     */
    override suspend fun registerAdClickEvent(
        cliUbid: String,
        uclid: String,
        trackingParams: TrackingParams?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        val baseParams = TrackingParams.builder()
            .uclid(uclid)
            .build()

        val finalTrackingParams = if (trackingParams != null) {
            baseParams.merge(trackingParams.toMap())
        } else {
            baseParams
        }

        val queryParams = mutableMapOf<String, Any>(
            "client_id" to config.clientId,
            "cli_ubid" to cliUbid,
            "event_time" to System.currentTimeMillis(),
            "os_name" to "Android",
            "os_version" to Constants.SDK_VERSION
        )

        // Add tracking params
        finalTrackingParams.toMap().forEach { (key, value) ->
            queryParams[key] = value
        }

        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {

            val baseUrl = config.eventTrackingHost

            trackEventsProcessor(
                "$baseUrl${ApiEndpoint.AD_CLICK.endpoint}",
                queryParams,
                errorCallback
            )
        }
    }

    /**
     * Registers a generic tracking event with type-safe parameters.
     * This function constructs query parameters containing event details and sends them
     * to the tracking endpoint via `trackEventsProcessor`.
     *
     * @param cliUbid Unique identifier
     * @param eventName The name of the event being tracked.
     * @param trackingParams TrackingParams containing event-specific parameters.
     * @return The API response as a Map<String,Any>.
     *
     */
    private suspend fun registerTrackingEvent(
        cliUbid: String,
        eventName: String,
        trackingParams: TrackingParams,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>? {

        val queryParams: MutableMap<String, Any> = mutableMapOf(
            "client_id" to config.clientId,
            "cli_ubid" to cliUbid,
            "event_time" to System.currentTimeMillis(),
            "os_name" to "Android",
            "os_version" to Constants.SDK_VERSION
        )

        if (eventName != "ad_click") {
            queryParams["event_name"] = eventName
        }

        trackingParams.toMap().forEach { (key, value) ->
            queryParams[key] = value
        }

        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {

            val baseUrl = config.eventTrackingHost

            trackEventsProcessor(
                "$baseUrl${ApiEndpoint.AD_TRACKING.endpoint}",
                queryParams,
                errorCallback
            )
        }
    }


    /**
     * Registers a "view product" event for tracking user interactions with type-safe parameters.
     *
     * @param cliUbid Unique identifier
     * @param skuId The unique identifier of the product being viewed
     * @param price Optional: Prices of the product, if applicable
     * @param merchantId Optional: Identifier of the merchant selling the product
     * @param trackingParams Optional: TrackingParams object with additional parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>
     *
     * Example:
     * ```
     * val trackingParams = TrackingParams.builder()
     *     .productQuantities(1)
     *     .storeId("1")
     *     .build()
     * val response = registerEvent.registerViewProductEvent(
     *     cliUbid = "12345",
     *     skuId = "sku_1234",
     *     productPrices = "50",
     *     merchantId = "merchant1",
     *     trackingParams = trackingParams
     * )
     * ```
     */
    override suspend fun registerViewProductEvent(
        cliUbid: String,
        skuId: String,
        price: Float?,
        merchantId: String?,
        trackingParams: TrackingParams?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        val baseParams = TrackingParams.builder()
            .skuId(skuId)
            .productPrices(price?.toString())
            .sellerId(merchantId)
            .productQuantities("1")
            .build()

        val finalTrackingParams = if (trackingParams != null) {
            baseParams.merge(trackingParams.toMap())
        } else {
            baseParams
        }

        return registerTrackingEvent(
            cliUbid,
            EventName.VIEW_PRODUCT.eventName,
            finalTrackingParams,
            errorCallback
        )
    }

    /**
     * Registers an "add to cart" event for tracking user interactions with type-safe parameters.
     *
     * @param cliUbid Unique identifier
     * @param skuId The unique identifier of the product being added to the cart
     * @param price Optional: The price of the product, if available
     * @param merchantId Optional: The identifier of the merchant selling the product
     * @param trackingParams Optional: TrackingParams object with additional parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>
     *
     * Example:
     * ```
     * val trackingParams = TrackingParams()
     *     .productQuantities(1)
     *     .storeId("1")
     * val response = registerEvent.registerAdd2CartEvent(
     *     cliUbid = "12345",
     *     skuId = "sku_1234",
     *     productPrices = "95",
     *     merchantId = "merchant1",
     *     trackingParams = trackingParams
     * )
     * ```
     */
    override suspend fun registerAdd2CartEvent(
        cliUbid: String,
        skuId: String,
        price: Float?,
        merchantId: String?,
        trackingParams: TrackingParams?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        val baseParams = TrackingParams.builder()
            .skuId(skuId)
            .productPrices(price?.toString())
            .sellerId(merchantId)
            .productQuantities("1")
            .build()

        val finalTrackingParams = if (trackingParams != null) {
            baseParams.merge(trackingParams.toMap())
        } else {
            baseParams
        }

        return registerTrackingEvent(
            cliUbid,
            EventName.ADD_TO_CART.eventName,
            finalTrackingParams,
            errorCallback
        )
    }

    /**
     * Registers a "purchase" event for tracking completed sales with type-safe parameters.
     *
     * @param cliUbid Unique identifier
     * @param sales The total number of sales associated with the purchase
     * @param orderId The unique identifier for the order (item identifier)
     * @param products A list of `Product` objects representing purchased items
     * @param trackingParams Optional: TrackingParams object with additional parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>
     *
     * Example:
     * ```
     * val trackingParams = TrackingParams()
     *     .storeId("1")
     *     .source("online")
     * val response = registerEvent.registerPurchaseEvent(
     *     cliUbid = "12345",
     *     sales = 100,
     *     orderId = "11082669",
     *     products = listOf(
     *         RegisterEvent.Product(
     *             product_prices = "50",
     *             product_quantities = "1",
     *             sku_id = "sku_1",
     *             merchant_id = "merchant1"
     *         )
     *     ),
     *     trackingParams = trackingParams
     * )
     * ```
     */
    override suspend fun registerPurchaseEvent(
        cliUbid: String,
        sales: Int,
        orderId: String,
        products: List<Product>,
        trackingParams: TrackingParams?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        // Extract values from the product list
        val productPrices = products.mapNotNull { it.price }.joinToString("|")
        val productQuantities = products.mapNotNull { it.quantity }.joinToString("|")
        val skuIds = products.joinToString("|") { it.skuId }
        val merchantIds = products.mapNotNull { it.merchantId }
            .takeIf { it.isNotEmpty() }?.joinToString("|")

        val baseParamsBuilder = TrackingParams.builder()
            .productPrices(productPrices)
            .productQuantities(productQuantities)
            .skuId(skuIds)
            .add("sales", sales)
            .add("item_identifier", orderId)

        merchantIds?.let { baseParamsBuilder.sellerId(it) }

        val baseParams = baseParamsBuilder.build()

        val finalTrackingParams = if (trackingParams != null) {
            baseParams.merge(trackingParams.toMap())
        } else {
            baseParams
        }

        return registerTrackingEvent(
            cliUbid,
            EventName.SALE_COMPLETE.eventName,
            finalTrackingParams,
            errorCallback
        )
    }

    /**
     * Registers an "ad impression" event for tracking ad views with type-safe parameters.
     *
     * @param cliUbid Unique identifier
     * @param uclid Unique identifier for the ad impression
     * @param position Optional: The position of the ad in the UI
     * @param trackingParams Optional: TrackingParams object with additional parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>
     *
     * Example:
     * ```
     * val trackingParams = TrackingParams()
     * val response = registerEvent.registerAdImpressionEvent(
     *     cliUbid = "12345",
     *     uclid = "ad_impression_123",
     *     position = 1,
     *     trackingParams = trackingParams
     * )
     * ```
     */
    override suspend fun registerAdImpressionEvent(
        cliUbid: String,
        uclid: String,
        position: Int?,
        trackingParams: TrackingParams?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        val baseParams = TrackingParams.builder()
            .uclid(uclid)
            .position(position)
            .build()

        val finalTrackingParams = if (trackingParams != null) {
            baseParams.merge(trackingParams.toMap())
        } else {
            baseParams
        }

        return registerTrackingEvent(
            cliUbid,
            EventName.IMPRESSION.eventName,
            finalTrackingParams,
            errorCallback
        )
    }

    /**
     * Registers a "video progress" event for tracking user engagement with type-safe parameters.
     *
     * @param cliUbid Unique identifier
     * @param uclid Unique identifier for the video being tracked
     * @param videoViewSec Number of seconds the video has been viewed
     * @param videoDurationSec Total duration of the video in seconds
     * @param trackingParams Optional: TrackingParams object with additional parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>
     *
     * Example:
     * ```
     * val trackingParams = TrackingParams()
     * val response = registerEvent.registerVideoProgressEvent(
     *     cliUbid = "12345",
     *     uclid = "video_123",
     *     videoViewSec = 15,
     *     videoDurationSec = 120,
     *     trackingParams = trackingParams
     * )
     * ```
     */
    override suspend fun registerVideoProgressEvent(
        cliUbid: String,
        uclid: String,
        videoViewSec: Float,
        videoDurationSec: Float,
        trackingParams: TrackingParams?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        val baseParams = TrackingParams.builder()
            .uclid(uclid)
            .videoViewSec(videoViewSec)
            .videoDurationSec(videoDurationSec)
            .build()

        val finalTrackingParams = if (trackingParams != null) {
            baseParams.merge(trackingParams.toMap())
        } else {
            baseParams
        }

        return registerTrackingEvent(
            cliUbid,
            EventName.VIDEO_PROGRESS.eventName,
            finalTrackingParams,
            errorCallback
        )
    }

    /**
     * Registers a "video action" event to track user interactions with a video using type-safe parameters.
     *
     * @param cliUbid Unique identifier
     * @param uclid Unique identifier for the video being tracked
     * @param videoViewSec The timestamp (in seconds) at which the action occurred
     * @param actionType The type of action performed on the video from VideoActionType enum
     * @param trackingParams Optional: TrackingParams object with additional parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>
     *
     * Example:
     * ```
     * val trackingParams = TrackingParams()
     * val response = registerEvent.registerVideoActionEvent(
     *     cliUbid = "12345",
     *     uclid = "video_123",
     *     videoViewSec = 15,
     *     actionType = VideoActionType.MUTE,
     *     trackingParams = trackingParams
     * )
     * ```
     */
    override suspend fun registerVideoActionEvent(
        cliUbid: String,
        uclid: String,
        videoViewSec: Float,
        actionType: VideoActionType,
        trackingParams: TrackingParams?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        val baseParams = TrackingParams.builder()
            .uclid(uclid)
            .videoViewSec(videoViewSec)
            .add("action_type", actionType.actionType)
            .build()

        val finalTrackingParams = if (trackingParams != null) {
            baseParams.merge(trackingParams.toMap())
        } else {
            baseParams
        }

        return registerTrackingEvent(
            cliUbid,
            EventName.VIDEO_ACTION.eventName,
            finalTrackingParams,
            errorCallback
        )
    }

}