package com.ai.osmos.tracking.events

import com.ai.osmos.models.events.TrackingParams
import com.ai.osmos.models.events.VideoActionType
import com.ai.osmos.utils.error.ErrorCallback

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: IRegisterEvent
 *
 * Interface for event tracking functionality in the OSMOS SDK.
 * Provides methods for registering various user interaction events.
 */
interface RegisterEventInterface {

    /**
     * Registers an ad click event for tracking user interactions with advertisements.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the ad click event.
     * @param trackingParams Optional: Additional tracking parameters.
     * @param errorCallback Optional: Callback for handling errors.
     * @return The API response as a Map<String,Any>.
     */
    suspend fun registerAdClickEvent(
        cliUbid: String,
        uclid: String,
        trackingParams: TrackingParams? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Registers a "view product" event for tracking user interactions.
     *
     * @param cliUbid Unique identifier.
     * @param skuId The unique identifier of the product being viewed.
     * @param price Optional: Prices of the product, if applicable.
     * @param merchantId Optional: Identifier of the merchant selling the product.
     * @param trackingParams Optional: Additional tracking parameters.
     * @param errorCallback Optional: Callback for handling errors.
     * @return The API response as a Map<String,Any>.
     */
    suspend fun registerViewProductEvent(
        cliUbid: String,
        skuId: String,
        price: Float? = null,
        merchantId: String? = null,
        trackingParams: TrackingParams? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Registers an "add to cart" event for tracking user interactions.
     *
     * @param cliUbid Unique identifier.
     * @param skuId The unique identifier of the product being added to the cart.
     * @param price Optional: The price of the product, if available.
     * @param merchantId Optional: The identifier of the merchant selling the product.
     * @param trackingParams Optional: Additional tracking parameters.
     * @param errorCallback Optional: Callback for handling errors.
     * @return The API response as a Map<String,Any>.
     */
    suspend fun registerAdd2CartEvent(
        cliUbid: String,
        skuId: String,
        price: Float? = null,
        merchantId: String? = null,
        trackingParams: TrackingParams? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Registers a "purchase" event for tracking completed sales.
     *
     * @param cliUbid Unique identifier.
     * @param sales The total number of sales associated with the purchase.
     * @param orderId The unique identifier for the order (item identifier).
     * @param products A list of `Product` objects representing purchased items.
     * @param trackingParams Optional: Additional tracking parameters.
     * @param errorCallback Optional: Callback for handling errors.
     * @return The API response as a Map<String,Any>.
     */
    suspend fun registerPurchaseEvent(
        cliUbid: String,
        sales: Int,
        orderId: String,
        products: List<Product>,
        trackingParams: TrackingParams? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Registers an "ad impression" event for tracking ad views.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the ad impression.
     * @param position Optional: The position of the ad in the UI.
     * @param trackingParams Optional: Additional tracking parameters.
     * @param errorCallback Optional: Callback for handling errors.
     * @return The API response as a Map<String,Any>.
     */
    suspend fun registerAdImpressionEvent(
        cliUbid: String,
        uclid: String,
        position: Int? = null,
        trackingParams: TrackingParams? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Registers a "video progress" event for tracking user engagement.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the video being tracked.
     * @param videoViewSec Number of seconds the video has been viewed.
     * @param videoDurationSec Total duration of the video in seconds.
     * @param trackingParams Optional: Additional tracking parameters.
     * @param errorCallback Optional: Callback for handling errors.
     * @return The API response as a Map<String,Any>.
     */
    suspend fun registerVideoProgressEvent(
        cliUbid: String,
        uclid: String,
        videoViewSec: Float,
        videoDurationSec: Float,
        trackingParams: TrackingParams? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Registers a "video action" event to track user interactions with a video.
     *
     * @param cliUbid Unique identifier.
     * @param uclid Unique identifier for the video being tracked.
     * @param videoViewSec The timestamp (in seconds) at which the action occurred.
     * @param actionType The type of action performed on the video from VideoActionType enum.
     * @param trackingParams Optional: Additional tracking parameters.
     * @param errorCallback Optional: Callback for handling errors.
     * @return The API response as a Map<String,Any>.
     */
    suspend fun registerVideoActionEvent(
        cliUbid: String,
        uclid: String,
        videoViewSec: Float,
        actionType: VideoActionType,
        trackingParams: TrackingParams? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?
}