package com.ai.osmos.utils.error

/**
 * Enum for network error codes and messages.
 * Centralizes all error definitions to avoid hardcoded values.
 */
enum class OsmosError(
    val code: String,
    val message: String
) {
    BAD_URL(
        code = "ERROR_BAD_URL",
        message = "Invalid or malformed URL"
    ),
    INVALID_REQUEST(
        code = "ERROR_INVALID_REQUEST",
        message = "Invalid request format or missing parameters"
    ),
    BAD_RESPONSE(
        code = "ERROR_BAD_RESPONSE",
        message = "Network error or invalid server response"
    ),
    BAD_STATUS(
        code = "ERROR_BAD_STATUS",
        message = "Unsuccessful HTTP status code"
    ),
    FAILED_TO_DECODE_RESPONSE(
        code = "ERROR_FAILED_TO_DECODE_RESPONSE",
        message = "Failed to decode or parse server response"
    ),
    TIMEOUT(
        code = "ERROR_TIMEOUT",
        message = "Request timeout"
    ),
    NETWORK_FAILURE(
        code = "ERROR_NETWORK_FAILURE",
        message = "Network connection failed"
    ),
    PARSING_ERROR(
        code = "ERROR_PARSING_AD_DATA",
        message = "Error parsing ad data"
    ),
    CONFIGURATION_ERROR(
        code = "ERROR_CONFIGURATION",
        message = "SDK configuration error"
    ),
    SDK_NOT_INITIALIZED(
        code = "ERROR_SDK_NOT_INITIALIZED",
        message = "OsmosSDK is not initialized. Call initialize() first."
    ),
    DISPLAY_ADS_NOT_CONFIGURED(
        code = "ERROR_DISPLAY_ADS_NOT_CONFIGURED",
        message = "Display Ads parameters not configured at time of SDK initialization"
    ),
    PLA_ADS_NOT_CONFIGURED(
        code = "ERROR_PLA_ADS_NOT_CONFIGURED",
        message = "PLA Ads parameters not configured at time of SDK initialization."
    ),
    REGISTER_EVENT_NOT_CONFIGURED(
        code = "ERROR_REGISTER_EVENT_NOT_CONFIGURED",
        message = "Register event parameters not configured at time of SDK initialization"
    ),
    UNKNOWN(
        code = "ERROR_UNKNOWN",
        message = "Unknown error occurred"
    ),

    // Media and dependency errors from BaseAdView
    EXOPLAYER_CORE_MISSING(
        code = "ERROR_EXOPLAYER_CORE_MISSING",
        message = "ExoPlayer core is missing. Please include 'media3-exoplayer' dependency in your app."
    ),
    EXOPLAYER_UI_MISSING(
        code = "ERROR_EXOPLAYER_UI_MISSING",
        message = "ExoPlayer UI module is missing. Please include 'media3-ui' dependency in your app."
    ),
    HLS_SUPPORT_MISSING(
        code = "ERROR_HLS_SUPPORT_MISSING",
        message = "HLS support is missing. Please include 'media3-exoplayer-hls' dependency in your app."
    ),
    DASH_SUPPORT_MISSING(
        code = "ERROR_DASH_SUPPORT_MISSING",
        message = "DASH support is missing. Please include 'media3-exoplayer-dash' dependency in your app."
    ),
    UNSUPPORTED_VIDEO_FORMAT(
        code = "ERROR_UNSUPPORTED_VIDEO_FORMAT",
        message = "Unsupported video format"
    ),
    UNSUPPORTED_IMAGE_FORMAT(
        code = "ERROR_UNSUPPORTED_IMAGE_FORMAT",
        message = "Unsupported image format"
    );

    companion object {
        /**
         * Find Error by error code string
         * Public API method that external apps might use for error handling
         */
        fun fromCode(code: String?): OsmosError? {
            return code?.let { OsmosError.entries.find { it.code == code } }
        }
    }
}