package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.graphics.drawable.GradientDrawable
import android.util.AttributeSet
import android.util.Log
import android.view.Gravity
import android.view.MotionEvent
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageButton
import androidx.media3.exoplayer.ExoPlayer
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.models.events.VideoActionType
import com.ai.osmos.tracking.tracker.AdTracker
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.toMap
import kotlin.math.absoluteValue


/**
 * Project Name: OSMOS-Android-SDK
 * File Name: InAppPIPView
 */
@SuppressLint("ClickableViewAccessibility", "ViewConstructor")
class InAppPIPView @JvmOverloads constructor(
    context: Context,
    videoAd: VideoAd,
    cliUbid: String,
    private var exoPlayer: ExoPlayer,
    private val adTracker: Any,
    height: Int,
    width: Int,
    coordinates: Coordinates?,
    adClickListener: ((Map<String, Any>?) -> Unit)? = null,
    private var errorCallback: ErrorCallback? = null,
    attrs: AttributeSet? = null

) : FrameLayout(context, attrs) {

    private lateinit var videoView: View
    private lateinit var closeButton: ImageButton
    private lateinit var muteButton: ImageButton
    private lateinit var fullscreenButton: ImageButton
    private val buttonSize = ViewUtils.dpToPx(context, 36) // or whatever size works for your design

    private var dX = 0f
    private var dY = 0f
    private var lastTouchDownX = 0f
    private var lastTouchDownY = 0f

    private var isMuted: Boolean = true
    private var onCloseClick: (() -> Unit)? = null
    private var onMuteToggleClick: (() -> Unit)? = null
    private var onFullscreenClick: (() -> Unit)? = null

    /**
     * Set an error callback to handle errors that occur during PIP view operations.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
    }

    /**
     * Logs errors consistently across the InAppPIPView functionality.
     * Creates an ExceptionHandler with structured error information and propagates
     * the error through the errorCallback if available.
     *
     * @param message The error message to be logged, describing the issue encountered.
     * @param errorType The specific error type from the OsmosError enum.
     * @param throwable The original throwable that caused the error (optional).
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Helper method to track video actions using either AdTracker or PersistentAdTracker.
     */
    private fun trackVideoAction(
        uclid: String,
        cliUbid: String,
        actionType: VideoActionType,
        videoViewSec: Float
    ) {
        when (adTracker) {
            is PersistentAdTracker -> adTracker.videoActionClick(
                uclid,
                cliUbid,
                actionType,
                videoViewSec
            )

            is AdTracker -> adTracker.videoActionClick(uclid, cliUbid, actionType, videoViewSec)
            else -> Log.w("InAppPIPView", "Unknown tracker type: ${adTracker.javaClass.simpleName}")
        }
    }

    /**
     * Helper method to track ad clicks using either AdTracker or PersistentAdTracker.
     */
    private fun trackAdClick(uclid: String, cliUbid: String) {
        when (adTracker) {
            is PersistentAdTracker -> adTracker.trackAdClick(uclid, cliUbid)
            is AdTracker -> adTracker.trackAdClick(uclid, cliUbid)
            else -> Log.w("InAppPIPView", "Unknown tracker type: ${adTracker.javaClass.simpleName}")
        }
    }

    init {

        layoutParams = LayoutParams(width, height).apply {
            gravity = Gravity.BOTTOM or Gravity.END
            setMargins(
                ViewUtils.dpToPx(context, 16),
                ViewUtils.dpToPx(context, 16),
                ViewUtils.dpToPx(context, 16),
                ViewUtils.dpToPx(context, 16)
            )
        }.apply {
            if (coordinates != null) {
                this@InAppPIPView.x = coordinates.x.toFloat()
                this@InAppPIPView.y = coordinates.y.toFloat()
            }
        }
        background = GradientDrawable().apply {
            cornerRadius = ViewUtils.dpToPx(context, 12).toFloat()
            setColor(Color.BLACK)
        }

        closeButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = android.R.drawable.ic_menu_close_clear_cancel,
            size = buttonSize,
            gravity = Gravity.TOP or Gravity.END,
            marginTop = ViewUtils.dpToPx(context, 4),
            marginEnd = ViewUtils.dpToPx(context, 4),
            onClick = {
                // Don't release ExoPlayer here - let PIPLoader handle it in cleanUp()
                onCloseClick?.invoke()
            }
        )

        muteButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = getDrawableResource(context, if (isMuted) "mute" else "unmute"),
            size = buttonSize,
            gravity = Gravity.BOTTOM or Gravity.END,
            marginEnd = ViewUtils.dpToPx(context, 4),
            marginBottom = ViewUtils.dpToPx(context, 4),
            onClick = {
                isMuted = !isMuted
                val actionType = if (isMuted) VideoActionType.MUTE else VideoActionType.UNMUTE

                val volumeSet = try {
                    exoPlayer.volume = if (isMuted) 0f else 1f
                    true
                } catch (e: IllegalStateException) {
                    errorLog(
                        "Cannot set volume on released player: ${e.message}",
                        OsmosError.UNKNOWN,
                        e
                    )
                    false
                }

                if (volumeSet) {
                    muteButton.setImageResource(
                        getDrawableResource(context, if (isMuted) "mute" else "unmute")
                    )

                    val currentPosition = exoPlayer.currentPosition

                    trackVideoAction(
                        videoAd.uclid,
                        cliUbid,
                        actionType,
                        (currentPosition / 1000).toFloat()
                    )
                    onMuteToggleClick?.invoke()
                }
            }
        )

        fullscreenButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = getDrawableResource(context, "fullscreen"),
            size = buttonSize,
            gravity = Gravity.TOP or Gravity.END,
            marginTop = ViewUtils.dpToPx(context, 4),
            marginEnd = ViewUtils.dpToPx(context, 50),
            onClick = {
                onFullscreenClick?.invoke()
            }
        )

        addView(closeButton)
        addView(muteButton)
        addView(fullscreenButton)

        setOnTouchListener { view, event ->
            val parent = view.parent as? View ?: return@setOnTouchListener false

            when (event.actionMasked) {
                MotionEvent.ACTION_DOWN -> {
                    dX = event.rawX - view.x
                    dY = event.rawY - view.y
                    lastTouchDownX = event.rawX
                    lastTouchDownY = event.rawY
                    true
                }

                MotionEvent.ACTION_MOVE -> {
                    val newX = event.rawX - dX
                    val newY = event.rawY - dY

                    val maxX = parent.width - view.width
                    val maxY = parent.height - view.height

                    view.x = newX.coerceIn(0f, maxX.toFloat())
                    view.y = newY.coerceIn(0f, maxY.toFloat())

                    true
                }

                MotionEvent.ACTION_UP -> {
                    val deltaX = (event.rawX - lastTouchDownX).absoluteValue
                    val deltaY = (event.rawY - lastTouchDownY).absoluteValue
                    val clickThreshold = 10

                    if (deltaX < clickThreshold && deltaY < clickThreshold) {
                        view.performClick() // triggers setOnClickListener
                    }
                    true
                }

                else -> false
            }
        }

        setOnClickListener {
            adClickListener?.invoke(videoAd.toMap())
            trackAdClick(videoAd.uclid, cliUbid)
        }
    }

    fun updateMuteButtonState(isMuted: Boolean) {
        muteButton.setImageResource(getDrawableResource(context, if (isMuted) "mute" else "unmute"))
        try {
            exoPlayer.volume = if (isMuted) 0f else 1f
        } catch (e: IllegalStateException) {
            errorLog(
                "Cannot update mute state on released player: ${e.message}",
                OsmosError.UNKNOWN,
                e
            )
        }
    }

    fun setVideoView(view: View) {
        videoView = view
        videoView.isClickable = false
        videoView.isFocusable = false

        // Enable dragging
        addView(videoView, 0) // Ensure it's below the buttons
    }

    fun setOnCloseClick(listener: () -> Unit) {
        onCloseClick = listener
    }

    fun setOnMuteToggleClick(listener: () -> Unit) {

        onMuteToggleClick = listener
    }

    fun setOnFullscreenClick(listener: () -> Unit) {
        onFullscreenClick = listener
    }

    override fun onDetachedFromWindow() {
        release()
        super.onDetachedFromWindow()
    }

    fun release() {
        try {
            // Clear listener references to prevent memory leaks
            onCloseClick = null
            onMuteToggleClick = null
            onFullscreenClick = null

            // Remove video view if attached
            if (::videoView.isInitialized) {
                removeView(videoView)
            }

        } catch (e: Exception) {
            errorLog("Error releasing PIP view: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    /**
     * Helper function to get drawable resource ID by name for library projects
     */
    private fun getDrawableResource(context: Context, resourceName: String): Int {
        return context.resources.getIdentifier(resourceName, "drawable", context.packageName)
    }

}