package com.ai.osmos.models.ads

/**
 * Immutable Product Listing Ads (PLA) filtering parameters with clean, Kotlin-idiomatic API
 * 
 * This class is thread-safe and immutable, preventing the issues from the old mutable builder pattern.
 */
class ProductAdsFilter private constructor(
    private val parameters: Map<String, Any>
) {
    
    /**
     * Get the filter parameters as an immutable map
     */
    fun toMap(): Map<String, Any> = parameters
    
    companion object {
        
        // MARK: - Category Factory Methods
        
        /**
         * Create filter with category ID
         */
        fun categoryId(categoryId: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("categoryId" to categoryId))
        }
        
        /**
         * Create filter with categories
         */
        fun categories(categories: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("categories" to categories.toList()))
        }
        
        /**
         * Create filter with merchant category IDs
         */
        fun mcategoryIds(mcategoryIds: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("mcategoryIds" to mcategoryIds.toList()))
        }
        
        /**
         * Create filter with SKU IDs
         */
        fun skuIds(skuIds: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("skuIds" to skuIds.toList()))
        }
        
        // MARK: - Price Factory Methods
        
        /**
         * Create filter with price range
         */
        fun priceRange(minPrice: String, maxPrice: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("minPrice" to minPrice, "maxPrice" to maxPrice))
        }
        
        /**
         * Create filter with minimum price
         */
        fun minPrice(minPrice: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("minPrice" to minPrice))
        }
        
        /**
         * Create filter with maximum price
         */
        fun maxPrice(maxPrice: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("maxPrice" to maxPrice))
        }
        
        // MARK: - Attribute Factory Methods
        
        /**
         * Create filter with colors
         */
        fun colors(colors: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("colors" to colors.toList()))
        }
        
        /**
         * Create filter with sizes
         */
        fun sizes(sizes: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("sizes" to sizes.toList()))
        }
        
        // MARK: - Search Factory Methods
        
        /**
         * Create filter with keywords
         */
        fun keywords(keywords: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("keywords" to keywords.toList()))
        }
        
        /**
         * Create filter with single keyword
         */
        fun keyword(keyword: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("keywords" to listOf(keyword)))
        }
        
        // MARK: - Brand Factory Methods
        
        /**
         * Create filter with brands
         */
        fun brands(brands: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("brands" to brands.toList()))
        }
        
        /**
         * Create filter with brand IDs
         */
        fun brandIds(brandIds: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("brandIds" to brandIds.toList()))
        }
        
        // MARK: - Location Factory Methods
        
        /**
         * Create filter with store IDs
         */
        fun storeIds(storeIds: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("storeIds" to storeIds.toList()))
        }
        
        /**
         * Create filter with networks
         */
        fun networks(networks: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("networks" to networks.toList()))
        }
        
        /**
         * Create filter with UDPs
         */
        fun udps(udps: List<String>): ProductAdsFilter {
            return ProductAdsFilter(mapOf("udps" to udps.toList()))
        }
        
        /**
         * Create filter with device
         */
        fun device(device: String): ProductAdsFilter {
            return ProductAdsFilter(mapOf("device" to device))
        }
        
        // MARK: - Custom Label Factory Methods
        
        /**
         * Create filter with custom labels
         */
        fun customLabels(
            label1: List<String>? = null,
            label2: List<String>? = null,
            label3: List<String>? = null,
            label4: List<String>? = null,
            label5: List<String>? = null,
            label6: List<String>? = null,
            label7: List<String>? = null,
            label8: List<String>? = null,
            rtlLabel0: List<String>? = null
        ): ProductAdsFilter {
            val params = mutableMapOf<String, Any>()
            label1?.let { params["customLabels1"] = it.toList() }
            label2?.let { params["customLabels2"] = it.toList() }
            label3?.let { params["customLabels3"] = it.toList() }
            label4?.let { params["customLabels4"] = it.toList() }
            label5?.let { params["customLabels5"] = it.toList() }
            label6?.let { params["customLabels6"] = it.toList() }
            label7?.let { params["customLabels7"] = it.toList() }
            label8?.let { params["customLabels8"] = it.toList() }
            rtlLabel0?.let { params["rtlCustomLabels0"] = it.toList() }
            return ProductAdsFilter(params.toMap())
        }
        
        // MARK: - Combined Factory Methods
        
        /**
         * Create comprehensive filter with multiple criteria
         */
        fun create(
            categoryId: String? = null,
            categories: List<String>? = null,
            skuIds: List<String>? = null,
            brands: List<String>? = null,
            minPrice: String? = null,
            maxPrice: String? = null,
            colors: List<String>? = null,
            sizes: List<String>? = null,
            keywords: List<String>? = null,
            storeIds: List<String>? = null,
            device: String? = null
        ): ProductAdsFilter {
            val params = mutableMapOf<String, Any>()
            categoryId?.let { params["categoryId"] = it }
            categories?.let { params["categories"] = it.toList() }
            skuIds?.let { params["skuIds"] = it.toList() }
            brands?.let { params["brands"] = it.toList() }
            minPrice?.let { params["minPrice"] = it }
            maxPrice?.let { params["maxPrice"] = it }
            colors?.let { params["colors"] = it.toList() }
            sizes?.let { params["sizes"] = it.toList() }
            keywords?.let { params["keywords"] = it.toList() }
            storeIds?.let { params["storeIds"] = it.toList() }
            device?.let { params["device"] = it }
            return ProductAdsFilter(params.toMap())
        }
    }
}