package com.ai.osmos.models.ads

/**
 * Base interface for all targeting parameter classes
 */
interface TargetingParams {
    fun toMap(): Map<String, Any>
}

/**
 * Platform-specific targeting parameters with thread-safe immutable pattern
 */
class PlatformTargeting private constructor(
    private val parameters: Map<String, Any>
) : TargetingParams {
    
    override fun toMap(): Map<String, Any> {
        val dict = mutableMapOf<String, Any>()
        parameters["device"]?.let { dict["f.device"] = it }
        return dict
    }
    
    companion object {
        /**
         * Create platform targeting with device specification
         */
        fun device(device: String): PlatformTargeting {
            return PlatformTargeting(mapOf("device" to device))
        }
    }
}

/**
 * Merchandise-specific targeting parameters with thread-safe immutable pattern
 */
class MerchandiseTargeting private constructor(
    private val parameters: Map<String, Any>
) : TargetingParams {
    
    override fun toMap(): Map<String, Any> {
        val dict = mutableMapOf<String, Any>()
        parameters["skuId"]?.let { dict["f.sku_id"] = it }
        parameters["categoryId"]?.let { dict["f.category_id"] = it }
        (parameters["brands"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.brand"] = it }
        parameters["categoryL1"]?.let { dict["f.category_l1"] = it }
        parameters["categoryL2"]?.let { dict["f.category_l2"] = it }
        parameters["categoryL3"]?.let { dict["f.category_l3"] = it }
        (parameters["udps"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.udp"] = it }
        return dict
    }
    
    companion object {
        /**
         * Create merchandise targeting with SKU ID
         */
        fun skuId(skuId: String): MerchandiseTargeting {
            return MerchandiseTargeting(mapOf("skuId" to skuId))
        }
        
        /**
         * Create merchandise targeting with category ID
         */
        fun categoryId(categoryId: String): MerchandiseTargeting {
            return MerchandiseTargeting(mapOf("categoryId" to categoryId))
        }
        
        /**
         * Create merchandise targeting with brands
         */
        fun brands(brands: List<String>): MerchandiseTargeting {
            return MerchandiseTargeting(mapOf("brands" to brands.toList()))
        }
        
        /**
         * Create merchandise targeting with category levels
         */
        fun categories(l1: String? = null, l2: String? = null, l3: String? = null): MerchandiseTargeting {
            val params = mutableMapOf<String, Any>()
            l1?.let { params["categoryL1"] = it }
            l2?.let { params["categoryL2"] = it }
            l3?.let { params["categoryL3"] = it }
            return MerchandiseTargeting(params.toMap())
        }
        
        /**
         * Create merchandise targeting with UDPs
         */
        fun udps(udps: List<String>): MerchandiseTargeting {
            return MerchandiseTargeting(mapOf("udps" to udps.toList()))
        }
    }
}

/**
 * Geographic targeting parameters with thread-safe immutable pattern
 */
class GeoTargeting private constructor(
    private val parameters: Map<String, Any>
) : TargetingParams {
    
    override fun toMap(): Map<String, Any> {
        val dict = mutableMapOf<String, Any>()
        (parameters["storeIds"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.store_id"] = it }
        (parameters["networks"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.network"] = it }
        parameters["city"]?.let { dict["f.city"] = it }
        parameters["state"]?.let { dict["f.state"] = it }
        parameters["country"]?.let { dict["f.country"] = it }
        return dict
    }
    
    companion object {
        /**
         * Create geo targeting with store IDs
         */
        fun storeIds(storeIds: List<String>): GeoTargeting {
            return GeoTargeting(mapOf("storeIds" to storeIds.toList()))
        }
        
        /**
         * Create geo targeting with networks
         */
        fun networks(networks: List<String>): GeoTargeting {
            return GeoTargeting(mapOf("networks" to networks.toList()))
        }
        
        /**
         * Create geo targeting with city
         */
        fun city(city: String): GeoTargeting {
            return GeoTargeting(mapOf("city" to city))
        }
        
        /**
         * Create geo targeting with state
         */
        fun state(state: String): GeoTargeting {
            return GeoTargeting(mapOf("state" to state))
        }
        
        /**
         * Create geo targeting with country
         */
        fun country(country: String): GeoTargeting {
            return GeoTargeting(mapOf("country" to country))
        }
        
        /**
         * Create geo targeting with location details
         */
        fun location(city: String? = null, state: String? = null, country: String? = null): GeoTargeting {
            val params = mutableMapOf<String, Any>()
            city?.let { params["city"] = it }
            state?.let { params["state"] = it }
            country?.let { params["country"] = it }
            return GeoTargeting(params.toMap())
        }
    }
}

/**
 * Context-specific targeting parameters with thread-safe immutable pattern
 */
class ContextTargeting private constructor(
    private val parameters: Map<String, Any>
) : TargetingParams {
    
    override fun toMap(): Map<String, Any> {
        val dict = mutableMapOf<String, Any>()
        (parameters["keyword"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.keyword"] = it }
        return dict
    }
    
    companion object {
        /**
         * Create context targeting with keywords
         */
        fun keywords(keywords: List<String>): ContextTargeting {
            return ContextTargeting(mapOf("keyword" to keywords.toList()))
        }
        
        /**
         * Create context targeting with single keyword
         */
        fun keyword(keyword: String): ContextTargeting {
            return ContextTargeting(mapOf("keyword" to listOf(keyword)))
        }
    }
}

/**
 * Custom label targeting parameters with thread-safe immutable pattern
 */
class CustomLabelTargeting private constructor(
    private val parameters: Map<String, Any>
) : TargetingParams {
    
    override fun toMap(): Map<String, Any> {
        val dict = mutableMapOf<String, Any>()
        (parameters["customLabel0"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.custom_label_0"] = it }
        (parameters["customLabel1"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.custom_label_1"] = it }
        (parameters["customLabel2"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.custom_label_2"] = it }
        (parameters["customLabel3"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.custom_label_3"] = it }
        (parameters["customLabel4"] as? List<*>)?.let { if (it.isNotEmpty()) dict["f.custom_label_4"] = it }
        return dict
    }
    
    companion object {
        /**
         * Create custom label targeting for specific label index
         */
        fun label(index: Int, values: List<String>): CustomLabelTargeting {
            require(index in 0..4) { "Label index must be between 0 and 4" }
            return CustomLabelTargeting(mapOf("customLabel$index" to values.toList()))
        }
        
        /**
         * Create custom label targeting with multiple labels
         */
        fun labels(
            label0: List<String>? = null,
            label1: List<String>? = null,
            label2: List<String>? = null,
            label3: List<String>? = null,
            label4: List<String>? = null
        ): CustomLabelTargeting {
            val params = mutableMapOf<String, Any>()
            label0?.let { params["customLabel0"] = it.toList() }
            label1?.let { params["customLabel1"] = it.toList() }
            label2?.let { params["customLabel2"] = it.toList() }
            label3?.let { params["customLabel3"] = it.toList() }
            label4?.let { params["customLabel4"] = it.toList() }
            return CustomLabelTargeting(params.toMap())
        }
    }
}