package com.ai.osmos.tracking.tracker.workers

import android.content.Context
import androidx.work.CoroutineWorker
import androidx.work.WorkerParameters
import androidx.work.workDataOf
import com.ai.osmos.tracking.data.VideoProgressData
import com.ai.osmos.tracking.events.RegisterEvent
import com.ai.osmos.tracking.tracker.BaseAdTracker
import com.ai.osmos.utils.logging.DebugLogger
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlin.math.min
import kotlin.math.pow

/**
 * WorkManager worker for persistent video progress tracking.
 *
 * This worker handles video progress tracking requests that need to survive app restarts
 * and network failures. It implements exponential backoff retry logic and ensures
 * video progress events are properly tracked for analytics.
 */
class VideoProgressTrackingWorker(
    context: Context,
    workerParams: WorkerParameters
) : CoroutineWorker(context, workerParams) {

    companion object {
        private const val TAG = "VideoProgressTrackingWorker"
    }


    override suspend fun doWork(): Result = withContext(Dispatchers.IO) {
        try {
            // Deserialize video progress data from input
            val videoProgressData = VideoProgressData.fromWorkData(inputData)
                ?: return@withContext Result.failure(
                    workDataOf("error" to "Invalid video progress data")
                )

            DebugLogger.log("Processing video progress: ${videoProgressData.videoViewSec}s/${videoProgressData.videoDurationSec}s, attempt: ${videoProgressData.attemptCount}")

            // Create RegisterEvent instance for API calls with instance-specific config
            val config = BaseAdTracker.getConfigFromWorkData(inputData)
            val registerEvent = RegisterEvent(config)

            try {
                // Attempt to track the video progress
                val response = registerEvent.registerVideoProgressEvent(
                    cliUbid = videoProgressData.cliUbid,
                    uclid = videoProgressData.uclid,
                    videoViewSec = videoProgressData.videoViewSec,
                    videoDurationSec = videoProgressData.videoDurationSec,
                )

                if (response != null) {
                    DebugLogger.log("Video progress tracked successfully: ${videoProgressData.videoViewSec}s/${videoProgressData.videoDurationSec}s")
                    Result.success(
                        workDataOf(
                            "uclid" to videoProgressData.uclid,
                            "video_view_sec" to videoProgressData.videoViewSec,
                            "video_duration_sec" to videoProgressData.videoDurationSec,
                            "result" to "success",
                            "attempt_count" to videoProgressData.attemptCount
                        )
                    )
                } else {
                    handleRetry(videoProgressData)
                }

            } catch (e: Exception) {
                DebugLogger.log("Video progress tracking failed: ${e.message}")

                // Handle retry with proper attempt count management
                handleRetry(videoProgressData)
            }

        } catch (e: Exception) {
            DebugLogger.log("Worker execution failed: ${e.message}")

            Result.failure(
                workDataOf(
                    "error" to "Worker execution failed: ${e.message}"
                )
            )
        }
    }

    /**
     * Handles retry logic with exponential backoff.
     *
     * @param videoProgressData The video progress data to retry
     * @return Result indicating whether to retry or give up
     */
    private fun handleRetry(videoProgressData: VideoProgressData): Result {
        val updatedData = videoProgressData.incrementAttempt()

        return if (updatedData.shouldRetry()) {
            DebugLogger.log("Scheduling retry ${updatedData.attemptCount}/${updatedData.maxRetries} for video progress")

            // Schedule retry with exponential backoff
            val retryDelay = calculateBackoffDelay(updatedData.attemptCount)
            scheduleRetryWork(updatedData.toWorkData(), retryDelay)

            // Return failure to indicate this attempt failed but retry is scheduled
            Result.failure()
        } else {
            DebugLogger.log("Max retries exceeded for video progress. Giving up.")

            Result.failure(
                workDataOf(
                    "error" to "Max retries exceeded for video progress tracking",
                    "uclid" to videoProgressData.uclid,
                    "video_view_sec" to videoProgressData.videoViewSec,
                    "video_duration_sec" to videoProgressData.videoDurationSec,
                    "attempt_count" to videoProgressData.attemptCount
                )
            )
        }
    }

    /**
     * Calculates exponential backoff delay for retries.
     *
     * Uses 2^attempt formula with a maximum cap to prevent excessive delays.
     * This helps distribute retry load and respects server capacity.
     *
     * @param attemptCount Current attempt number (1-based)
     * @return Delay in seconds before next retry
     */
    private fun calculateBackoffDelay(attemptCount: Int): Long {
        // Exponential backoff: 2^attempt seconds, capped at 5 minutes (300s)
        val delaySeconds = min(
            2.0.pow(attemptCount.toDouble()).toLong(),
            300L
        )
        DebugLogger.log("Calculated backoff delay: ${delaySeconds}s for attempt $attemptCount")

        return delaySeconds
    }

    /**
     * Schedules a retry work request with appropriate constraints and backoff.
     *
     * @param data Serialized video progress data for the retry
     * @param delaySeconds Delay before executing the retry
     */
    private fun scheduleRetryWork(data: androidx.work.Data, delaySeconds: Long) {
        try {
            val retryRequest =
                androidx.work.OneTimeWorkRequestBuilder<VideoProgressTrackingWorker>()
                    .setInputData(data)
                    .setInitialDelay(delaySeconds, java.util.concurrent.TimeUnit.SECONDS)
                    .setConstraints(
                        androidx.work.Constraints.Builder()
                            .setRequiredNetworkType(androidx.work.NetworkType.CONNECTED)
                            .build()
                    )
                    .addTag(VideoProgressData.WORK_TAG_VIDEO_PROGRESS)
                    .build()

            androidx.work.WorkManager.getInstance(applicationContext).enqueue(retryRequest)
            DebugLogger.log("Retry work scheduled with ${delaySeconds}s delay")

        } catch (e: Exception) {
            DebugLogger.log("Failed to schedule retry work: ${e.message}")
        }
    }
}