package com.ai.osmos.utils.error

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: StandardErrorHandler
 *
 * Centralized error handling utility that provides consistent error handling patterns
 * across all SDK classes. This reduces code duplication and ensures uniform error management.
 */
object StandardErrorHandler {

    /**
     * Executes a suspend function with standardized error handling.
     *
     * @param errorCallback Optional callback to handle errors
     * @param operation The suspend function to execute
     * @return The result of the operation or null if an error occurred
     */
    suspend fun <T> executeWithErrorHandling(
        errorCallback: ErrorCallback? = null,
        operation: suspend () -> T
    ): T? {
        return try {
            operation()
        } catch (e: ExceptionHandler) {
            errorCallback?.onError(e.errorCode, e.errorMessage, e)
            null
        } catch (e: Exception) {
            val error = OsmosError.UNKNOWN
            errorCallback?.onError(error.code, e.message ?: error.message, e)
            null
        }
    }

    /**
     * Validates configuration and throws appropriate exception if not configured.
     *
     * @param configCheck Function that checks if configuration exists
     * @param error The error to throw if configuration is missing
     * @throws ExceptionHandler if configuration is invalid
     */
    fun validateConfiguration(
        configCheck: () -> Any?,
        error: OsmosError
    ) {
        if (configCheck() == null) {
            throw ExceptionHandler(error)
        }
    }

    /**
     * Wraps parsing operations with consistent error handling.
     *
     * @param operation The parsing operation to execute
     * @return The parsed result
     * @throws ExceptionHandler if parsing fails
     */
    fun <T> executeWithParsingErrorHandling(operation: () -> T): T {
        return try {
            operation()
        } catch (e: Exception) {
            throw ExceptionHandler(OsmosError.PARSING_ERROR, e)
        }
    }
}