package com.ai.osmos.ads.renderer

import com.ai.osmos.models.ads.TargetingParams
import com.ai.osmos.models.enums.DisplayAdFormat
import com.ai.osmos.utils.error.ErrorCallback

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: AdRendererInterface
 *
 * Interface for ad rendering functionality in the OSMOS SDK.
 * Provides methods for fetching and preparing different types of ads.
 */
public interface AdRendererInterface {

    /**
     * Fetch banner ads with ad units
     */
    suspend fun fetchBannerAdsWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch carousel ads with ad units
     */
    suspend fun fetchCarouselAdWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch multiple ads in carousel format with ad units
     */
    suspend fun fetchMultiAdCarouselWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        productCount: Int,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Parse display ad response data into structured format
     *
     * @param adData Raw ad data from API response
     * @param cliUbid Client unique browser ID for tracking
     * @param adFormat Display ad format type (Banner, Carousel, MultiAdCarousel)
     * @return Parsed ad data map with "ad" key containing the BaseAd object
     *
     * @example
     * ```kotlin
     * // Example usage - similar to MainActivity showBannerAd function
     * val sdk = OsmosSDK.clientId("10088010").debug(true).build()
     * val adRenderer = sdk.adRenderer()
     *
     * // Sample banner ad JSON response from server
     * val bannerAdRaw = """
     *     {
     *       "ads": {
     *         "banner_ads": [
     *           {
     *             "client_id": 123,
     *             "au": "banner_ads",
     *             "rank": 1,
     *             "click_tracking_url": "https://demo.io/click?client_id=123&...",
     *             "elements": {
     *               "type": "IMAGE",
     *               "value": "https://osads.gumlet.io/image/upload/v1747206595/lacy7ljgoxevl9rgmeqo.jpg",
     *               "width": 742,
     *               "height": 355,
     *               "destination_url": "https://demoSDK.com"
     *             },
     *             "impression_tracking_url": "https://demo.io/events?client_id=123&event_name=funnel_impression&uclid=...",
     *             "uclid": "123",
     *             "crt": "osmos_sdk/image/v1/320x50"
     *           }
     *         ]
     *       }
     *     }
     * """.trimIndent()
     *
     * // Parse the response
     * val bannerData = adRenderer.parseDisplayAdResponse(
     *     bannerAdRaw,
     *     "xyz",
     *     DisplayAdFormat.Banner
     * )
     *
     * // Use the parsed data
     * if (bannerData.isNotEmpty()) {
     *     // Create BannerAdView
     *     val adView = sdk.bannerAdView().showAd(
     *         context = context,
     *         ad = bannerData,
     *         adViewSettings = BannerAdSettings(
     *             height = 300,
     *             width = 290
     *         )
     *     )
     *
     *     // Add to layout
     *     parentLayout.addView(adView)
     * }
     * ```
     */
    fun parseDisplayAdResponse(
        adData: String,
        cliUbid: String,
        adFormat: DisplayAdFormat
    ): Map<String, Any>
}