package com.ai.osmos.ads.renderer.loaders

import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.drawable.GradientDrawable
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.FrameLayout
import android.widget.ImageButton
import android.widget.ImageView
import androidx.lifecycle.LifecycleOwner
import androidx.media3.common.MediaItem
import androidx.media3.common.Player
import androidx.media3.exoplayer.ExoPlayer
import androidx.media3.ui.PlayerView
import com.ai.osmos.ads.renderer.VideoPlayerLifecycleObserver
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.models.events.VideoActionType
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.isVisibleInScreen
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.lang.ref.WeakReference
import java.util.Collections
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicBoolean

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: VideoLoader
 */
class VideoLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface
) {
    private var errorCallback: ErrorCallback? = null
    private lateinit var mButton: ImageButton
    private var isViewVisible = false
    private var wasPlaying = false

    // Thread-safe maps with synchronized access
    private val playerMap: MutableMap<PlayerView, ExoPlayer> = ConcurrentHashMap()
    private val muteStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val visibilityStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val wasPlayingStateMap: MutableMap<ExoPlayer, Boolean> = ConcurrentHashMap()
    private val progressJobMap: MutableMap<PlayerView, Job> = ConcurrentHashMap()
    private val lifecycleObserverMap: MutableMap<ExoPlayer, VideoPlayerLifecycleObserver> = ConcurrentHashMap()
    
    // Central ExoPlayer registry for guaranteed cleanup
    private val allExoPlayers: MutableSet<ExoPlayer> = Collections.newSetFromMap(ConcurrentHashMap<ExoPlayer, Boolean>())
    
    // Track ViewTreeObserver listeners for proper cleanup
    private val scrollListenerMap: MutableMap<PlayerView, ViewTreeObserver.OnScrollChangedListener> = ConcurrentHashMap()
    private val preDrawListenerMap: MutableMap<PlayerView, ViewTreeObserver.OnPreDrawListener> = ConcurrentHashMap()
    private val attachListenerMap: MutableMap<PlayerView, View.OnAttachStateChangeListener> = ConcurrentHashMap()
    
    // Track ExoPlayer.Player.Listener instances for proper cleanup
    private val playerListenerMap: MutableMap<ExoPlayer, Player.Listener> = ConcurrentHashMap()

    // Synchronization locks for atomic operations
    private val stateLock = Any()
    private val playerLock = Any()

    private var onViewLoad: ((adMetadata: Map<String, Any>, String) -> Unit)? = null
    private val isReleased = AtomicBoolean(false)

    /**
     * Sets a listener to be notified when the ad view is fully loaded.
     *
     * @param listener Callback receiving the loaded [ImageAd] and its tracking ID.
     */
    fun setViewLoadListener(listener: (adMetadata: Map<String, Any>, String) -> Unit) {
        this.onViewLoad = { adMetadata, cliUbid ->
            listener(adMetadata, cliUbid)
        }
    }

    /**
     * Set an error callback to handle errors that occur during video loading or playback.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
    }

    /**
     * Logs an error message using structured error handling.
     *
     * This function uses the ExceptionHandler pattern to log errors with proper error codes
     * and calls the error callback if one is set.
     *
     * @param message The error message to be logged, describing the issue encountered.
     * @param errorType The specific error type from the OsmosError enum.
     * @param throwable The original throwable that caused the error (optional).
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Tracks an ExoPlayer instance in the central registry for lifecycle management
     */
    private fun trackExoPlayerInstance(exoPlayer: ExoPlayer) {
        allExoPlayers.add(exoPlayer)
    }
    
    /**
     * Thread-safe cleanup of player state across all maps with guaranteed cleanup
     */
    private fun cleanupPlayerState(exoPlayer: ExoPlayer) {
        try {
            // Remove from all state maps
            muteStateMap.remove(exoPlayer)
            visibilityStateMap.remove(exoPlayer)
            wasPlayingStateMap.remove(exoPlayer)
            lifecycleObserverMap.remove(exoPlayer)?.clearErrorCallback()
            
            // Remove Player.Listener to prevent memory leaks
            removePlayerListener(exoPlayer)
            
            // Remove from playerMap and progressJobMap to prevent memory leaks
            val playerViewToRemove = playerMap.entries.find { it.value == exoPlayer }?.key
            playerViewToRemove?.let { playerView ->
                playerMap.remove(playerView)
                progressJobMap.remove(playerView)?.cancel()
                // Remove all listeners to prevent memory leaks
                removeScrollListener(playerView)
                removePreDrawListener(playerView)
                removeAttachListener(playerView)
            }
            
            // Remove from central registry
            allExoPlayers.remove(exoPlayer)
            
        } catch (e: Exception) {
            errorLog("Error during player state cleanup: ${e.message}", OsmosError.UNKNOWN, e)
            // Force removal from registry even if cleanup fails
            allExoPlayers.remove(exoPlayer)
        }
    }

    /**
     * Safely releases an ExoPlayer instance and removes it from all tracking maps
     * This method ensures ExoPlayer is properly released and removed from all tracking
     */
    private fun releaseAndUntrackExoPlayer(exoPlayer: ExoPlayer) {
        try {
            // Stop and release the player
            exoPlayer.apply {
                playWhenReady = false
                stop()
                clearVideoSurface()
                release()
            }
        } catch (e: Exception) {
            errorLog("Error releasing ExoPlayer: ${e.message}", OsmosError.UNKNOWN, e)
        } finally {
            // Always clean up tracking state, even if release fails
            cleanupPlayerState(exoPlayer)
        }
    }

    /**
     * Adds automatic cleanup listener to PlayerView to prevent memory leaks during view recycling
     * Smart fallback cleanup that only triggers when manual cleanup has failed
     */
    private fun addAutoCleanupListener(playerView: PlayerView, enableAutoCleanup: Boolean = true) {
        if (enableAutoCleanup) {
            val attachListener = object : View.OnAttachStateChangeListener {
                override fun onViewAttachedToWindow(v: View) = Unit
                override fun onViewDetachedFromWindow(v: View) {
                    // ATOMIC FALLBACK: Only cleanup if manual cleanup failed
                    // Use synchronized block to prevent race condition with manual cleanup
                    synchronized(playerLock) {
                        val exoPlayer = playerMap[playerView]
                        if (exoPlayer != null && allExoPlayers.contains(exoPlayer)) {
                            // Double-check inside synchronized block to prevent race condition
                            if (playerMap.containsKey(playerView)) {
                                // Manual cleanup failed - execute auto cleanup as fallback
                                safeStopAndReleasePlayer(playerView)
                                errorLog("Auto-cleanup fallback executed - manual cleanup failed for PlayerView", OsmosError.UNKNOWN)
                            }
                        }
                    }
                }
            }
            playerView.addOnAttachStateChangeListener(attachListener)
            attachListenerMap[playerView] = attachListener
        }
    }

    /**
     * Safely adds and tracks OnScrollChangedListener to prevent memory leaks
     */
    private fun addTrackedScrollListener(playerView: PlayerView, exoPlayer: ExoPlayer) {
        // Remove any existing listener first
        removeScrollListener(playerView)
        
        val scrollListener = ViewTreeObserver.OnScrollChangedListener {
            checkVisibilityAndHandlePlayback(playerView, exoPlayer)
        }
        
        // Add listener and track it
        playerView.viewTreeObserver.addOnScrollChangedListener(scrollListener)
        scrollListenerMap[playerView] = scrollListener
    }

    /**
     * Removes OnScrollChangedListener from PlayerView and tracking
     */
    private fun removeScrollListener(playerView: PlayerView) {
        scrollListenerMap.remove(playerView)?.let { listener ->
            try {
                playerView.viewTreeObserver.removeOnScrollChangedListener(listener)
            } catch (e: Exception) {
                errorLog("Error removing scroll listener: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }
    }

    /**
     * Safely adds and tracks Player.Listener to ExoPlayer to prevent memory leaks
     */
    private fun addTrackedPlayerListener(
        exoPlayer: ExoPlayer,
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        muteButton: ImageButton?
    ) {
        // Remove any existing listener first
        removePlayerListener(exoPlayer)
        
        val playerListener = createPlayerListener(videoAd, cliUbid, playerView, exoPlayer, muteButton)
        
        // Add listener and track it
        exoPlayer.addListener(playerListener)
        playerListenerMap[exoPlayer] = playerListener
    }

    /**
     * Removes Player.Listener from ExoPlayer and tracking
     */
    private fun removePlayerListener(exoPlayer: ExoPlayer) {
        playerListenerMap.remove(exoPlayer)?.let { listener ->
            try {
                exoPlayer.removeListener(listener)
            } catch (e: Exception) {
                errorLog("Error removing player listener: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }
    }

    /**
     * Removes OnPreDrawListener from PlayerView and tracking
     */
    private fun removePreDrawListener(playerView: PlayerView) {
        preDrawListenerMap.remove(playerView)?.let { listener ->
            try {
                playerView.viewTreeObserver.removeOnPreDrawListener(listener)
            } catch (e: Exception) {
                errorLog("Error removing predraw listener: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }
    }

    /**
     * Removes OnAttachStateChangeListener from PlayerView and tracking
     */
    private fun removeAttachListener(playerView: PlayerView) {
        attachListenerMap.remove(playerView)?.let { listener ->
            try {
                playerView.removeOnAttachStateChangeListener(listener)
            } catch (e: Exception) {
                errorLog("Error removing attach listener: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }
    }

    internal fun createVideoView(
        context: Context,
        videoAd: VideoAd,
        isClamped: Boolean,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null

    ): FrameLayout {
        val widthPx = if (isClamped) widthDp else ViewUtils.dpToPx(context, widthDp)
        val heightPx = if (isClamped) heightDp else ViewUtils.dpToPx(context, heightDp)
        val exoPlayer: ExoPlayer = ExoPlayer.Builder(context).build()
        val playerView = buildPlayerView(context, widthPx, heightPx)
        
        // Track ExoPlayer instance for lifecycle management
        trackExoPlayerInstance(exoPlayer)

        // Initialize states for this specific player
        muteStateMap[exoPlayer] = true  // default: muted
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing
        
        // Add automatic cleanup on view detachment to prevent memory leaks
        // Disabled for regular video views since they have manual cleanup in onDetachedFromWindow
        addAutoCleanupListener(playerView, enableAutoCleanup = false)

        val muteButton = ImageButton(context).apply {
            val currentMuteState = muteStateMap[exoPlayer] ?: false
            setImageResource(
                getDrawableResource(
                    context,
                    if (currentMuteState) "mute" else "unmute"
                )
            )
            setColorFilter(Color.BLACK, PorterDuff.Mode.SRC_IN)
            scaleType = ImageView.ScaleType.FIT_CENTER
            adjustViewBounds = true
            background = GradientDrawable().apply {
                shape = GradientDrawable.OVAL
                setStroke(3, Color.BLACK)
                setColor(Color.WHITE)
            }

            layoutParams = FrameLayout.LayoutParams(
                ViewUtils.dpToPx(context, 24),
                ViewUtils.dpToPx(context, 24),
                Gravity.END or Gravity.BOTTOM
            ).apply {
                bottomMargin = ViewUtils.dpToPx(context, 50)

                marginEnd = ViewUtils.dpToPx(context, 16)
                marginStart = ViewUtils.dpToPx(context, 16)
            }
            visibility = View.GONE
            setOnClickListener { toggleMute(this, videoAd, cliUbid, exoPlayer) }
        }

        // Load video asynchronously
        loadVideoUsingUrl(context, videoAd, cliUbid, playerView, exoPlayer, muteButton)

        //Add Listeners
        addListeners(videoAd, cliUbid, playerView, adClickListener, exoPlayer)

        return buildContainer(context, widthPx, heightPx, playerView, muteButton)
    }

    internal fun createPIPVideoView(
        context: Context,
        videoAd: VideoAd,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null,
        exoPlayer: ExoPlayer
    ): PlayerView {

        val playerView = buildPlayerView(context, widthDp, heightDp)
        
        // Track ExoPlayer instance for lifecycle management (external ExoPlayer)
        trackExoPlayerInstance(exoPlayer)

        // Initialize states for this specific PIP player
        muteStateMap[exoPlayer] = true  // default: muted for PIP
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing
        
        // Add automatic cleanup on view detachment to prevent memory leaks  
        // Disabled for PIP since it manages its own ExoPlayer lifecycle
        addAutoCleanupListener(playerView, enableAutoCleanup = false)

        // Load and play video
        loadVideoUsingUrl(context, videoAd, cliUbid, playerView, exoPlayer)

        addListeners(videoAd, cliUbid, playerView, adClickListener, exoPlayer)

        return playerView
    }

    internal fun createCarouselVideoView(
        context: Context,
        videoAd: VideoAd,
        widthDp: Int,
        heightDp: Int,
        cliUbid: String,
    ): FrameLayout {

        val widthPx = ViewUtils.dpToPx(context, widthDp)
        val heightPx = ViewUtils.dpToPx(context, heightDp)
        val exoPlayer: ExoPlayer = ExoPlayer.Builder(context).build()
        val playerView = buildPlayerView(context, widthPx, heightPx)
        
        // Track ExoPlayer instance for lifecycle management
        trackExoPlayerInstance(exoPlayer)

        // Initialize states for this specific carousel player
        muteStateMap[exoPlayer] = true  // default: muted
        visibilityStateMap[exoPlayer] = false  // default: not visible
        wasPlayingStateMap[exoPlayer] = false  // default: not playing

        // Add automatic cleanup on view detachment to prevent memory leaks
        // Enabled as fallback in case RecyclerView onViewRecycled cleanup fails
        addAutoCleanupListener(playerView, enableAutoCleanup = true)

        val buttonSize = ViewUtils.dpToPx(context, 36)
        val currentMuteState = muteStateMap[exoPlayer] ?: false
        mButton = ViewUtils.createStyledButton(
            context = context,
            iconRes = getDrawableResource(context, if (currentMuteState) "mute" else "unmute"),
            size = buttonSize,
            gravity = Gravity.BOTTOM or Gravity.RIGHT,
            marginBottom = ViewUtils.dpToPx(context, 40),
            marginEnd = ViewUtils.dpToPx(context, 10),
            onClick = {
                toggleMute(mButton, videoAd, cliUbid, exoPlayer)
            })
        mButton.visibility = View.GONE

        // Load video asynchronously
        loadVideoUsingUrl(context, videoAd, cliUbid, playerView, exoPlayer, mButton)

        // Use tracked scroll listener to prevent memory leaks
        addTrackedScrollListener(playerView, exoPlayer)

        val preDrawListener = object : ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                if (playerView.isVisibleInScreen()) {
                    onViewLoad?.invoke(videoAd.toMap(), cliUbid)
                    trackImpression(videoAd.uclid, cliUbid)
                    playerView.viewTreeObserver.removeOnPreDrawListener(this)
                    preDrawListenerMap.remove(playerView)
                }
                return true
            }
        }
        playerView.viewTreeObserver.addOnPreDrawListener(preDrawListener)
        preDrawListenerMap[playerView] = preDrawListener
        return buildContainer(context, widthPx, heightPx, playerView, mButton)
    }

    private fun buildPlayerView(context: Context, widthPx: Int, heightPx: Int): PlayerView {
        return PlayerView(context).apply {
            layoutParams = FrameLayout.LayoutParams(widthPx, heightPx, Gravity.CENTER)
            useController = false
        }
    }

    private fun buildContainer(
        context: Context,
        widthPx: Int,
        heightPx: Int,
        playerView: PlayerView,
        muteButton: View? = null
    ): FrameLayout {
        return FrameLayout(context).apply {
            layoutParams = ViewGroup.LayoutParams(widthPx, heightPx)
            addView(playerView, FrameLayout.LayoutParams(widthPx, heightPx).apply {
                gravity = Gravity.CENTER
            })
            muteButton?.let {
                addView(it, FrameLayout.LayoutParams(90, 90).apply {
                    gravity = Gravity.BOTTOM or Gravity.END
                    bottomMargin = 26
                    rightMargin = 10
                })
            }
        }
    }

    private fun loadVideoUsingUrl(
        context: Context,
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer,
        muteButton: ImageButton? = null
    ) {
        val loadingJob = coroutineScope.launch(Dispatchers.IO) {  // Fetch video URL in background
            try {
                val videoUrl = videoAd.elements.value
                withContext(Dispatchers.Main) { // Switch to Main thread for ExoPlayer

                    exoPlayer.apply {
                        // Safest approach: Let ExoPlayer handle all format detection automatically
                        val mediaItem = MediaItem.Builder()
                            .setUri(videoUrl)
                            .build()
                        setMediaItem(mediaItem)
                        prepare()

                        val playerMuteState = muteStateMap[exoPlayer] ?: false
                        try {
                            volume = if (playerMuteState) 0f else 1f
                        } catch (e: IllegalStateException) {
                            errorLog(
                                "Cannot set initial volume on released player: ${e.message}",
                                OsmosError.UNKNOWN,
                                e
                            )
                        }
                    }
                    
                    // Use tracked player listener to prevent memory leaks
                    addTrackedPlayerListener(exoPlayer, videoAd, cliUbid, playerView, muteButton)
                    playerMap[playerView] = exoPlayer
                    playerView.player = exoPlayer
                    exoPlayer.playWhenReady = true

                    // Attach lifecycle observer after player is added to map with WeakReference
                    if (context is LifecycleOwner) {
                        val contextRef = WeakReference(context)
                        val observer = VideoPlayerLifecycleObserver(exoPlayer, errorCallback)
                        contextRef.get()?.lifecycle?.addObserver(observer)
                        if (contextRef.get() != null) {
                            lifecycleObserverMap[exoPlayer] = observer
                        }
                    }
                }

            } catch (e: Exception) {
                errorLog("Error loading video: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }

        // Store loading job for cleanup - using a temporary key until progress tracking starts
        progressJobMap[playerView] = loadingJob

    }

    private fun addListeners(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)?,
        exoPlayer: ExoPlayer
    ) {
        playerView.setOnClickListener {
            trackAdClick(videoAd.uclid, cliUbid)
            adClickListener?.invoke(videoAd.toMap())
        }

        // Use tracked scroll listener to prevent memory leaks
        addTrackedScrollListener(playerView, exoPlayer)

        val preDrawListener = object : ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                if (playerView.isVisibleInScreen()) {
                    onViewLoad?.invoke(videoAd.toMap(), cliUbid)
                    trackImpression(videoAd.uclid, cliUbid)
                    playerView.viewTreeObserver.removeOnPreDrawListener(this)
                    preDrawListenerMap.remove(playerView)
                }
                return true
            }
        }
        playerView.viewTreeObserver.addOnPreDrawListener(preDrawListener)
        preDrawListenerMap[playerView] = preDrawListener
    }

    private fun createPlayerListener(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer,
        muteButton: ImageButton? = null,
    ) = object : Player.Listener {

        override fun onPlaybackStateChanged(state: Int) {

            when (state) {
                Player.STATE_IDLE -> {}
                Player.STATE_BUFFERING -> {}
                Player.STATE_READY -> {
                    Log.d(
                        "VideoProgress",
                        "Progress: Start"
                    )
                    if (muteButton != null)
                        muteButton.visibility = View.VISIBLE

                    trackVideoProgressWithEvent(videoAd, cliUbid, playerView, exoPlayer)
                }

                Player.STATE_ENDED -> {
                    if (playerView.isVisibleInScreen()) {
                        val currentPosition = exoPlayer.currentPosition ?: 0L  // Current time (ms)
                        val totalDuration = exoPlayer.duration ?: 0L  // Total duration (ms)

                        Log.d(
                            "VideoProgress",
                            "Progress: End "
                        )

                        fireVideoProgressEvent(
                            videoAd.uclid,
                            cliUbid,
                            currentPosition,
                            totalDuration
                        )
                    }
                    exoPlayer?.seekTo(0)
                    exoPlayer?.playWhenReady = true
                }
            }
        }

        override fun onPlayerError(error: androidx.media3.common.PlaybackException) {
            val errorMessage = when (error.errorCode) {
                androidx.media3.common.PlaybackException.ERROR_CODE_IO_NETWORK_CONNECTION_FAILED ->
                    "Network connection failed while loading video"

                androidx.media3.common.PlaybackException.ERROR_CODE_IO_NETWORK_CONNECTION_TIMEOUT ->
                    "Network connection timeout while loading video"

                androidx.media3.common.PlaybackException.ERROR_CODE_PARSING_CONTAINER_MALFORMED ->
                    "Video file is corrupted or malformed"

                androidx.media3.common.PlaybackException.ERROR_CODE_PARSING_MANIFEST_MALFORMED ->
                    "Video manifest is corrupted or malformed"

                androidx.media3.common.PlaybackException.ERROR_CODE_DECODER_INIT_FAILED ->
                    "Video decoder initialization failed"

                androidx.media3.common.PlaybackException.ERROR_CODE_AUDIO_TRACK_INIT_FAILED ->
                    "Audio track initialization failed"

                androidx.media3.common.PlaybackException.ERROR_CODE_DRM_SYSTEM_ERROR ->
                    "DRM system error"

                else -> "Video playback error: ${error.message}"
            }

            val osmosError = when (error.errorCode) {
                androidx.media3.common.PlaybackException.ERROR_CODE_IO_NETWORK_CONNECTION_FAILED,
                androidx.media3.common.PlaybackException.ERROR_CODE_IO_NETWORK_CONNECTION_TIMEOUT ->
                    OsmosError.NETWORK_FAILURE

                androidx.media3.common.PlaybackException.ERROR_CODE_PARSING_CONTAINER_MALFORMED,
                androidx.media3.common.PlaybackException.ERROR_CODE_PARSING_MANIFEST_MALFORMED ->
                    OsmosError.UNSUPPORTED_VIDEO_FORMAT

                else -> OsmosError.UNKNOWN
            }

            errorLog(errorMessage, osmosError, error)
        }
    }


    private fun trackVideoProgressWithEvent(
        videoAd: VideoAd,
        cliUbid: String,
        playerView: PlayerView,
        exoPlayer: ExoPlayer
    ) {
        val job = coroutineScope.launch(Dispatchers.Main) {
            val eventProgressTime =
                videoAd.elements.videoProgressSec ?: Constants.VIDEO_PROGRESS_SEC_DEFAULT

            var lastFiredSec = -eventProgressTime

            try {
                while (coroutineContext.isActive &&  allExoPlayers.contains(exoPlayer)) {
                    if (exoPlayer.isPlaying && playerView.isVisibleInScreen()) {
                        val currentPositionMs = exoPlayer.currentPosition ?: 0L
                        val currentSec = (currentPositionMs / 1000).toInt()

                        if (currentSec >= lastFiredSec + eventProgressTime) {
                            fireVideoProgressEvent(
                                videoAd.uclid,
                                cliUbid,
                                currentPositionMs,
                                exoPlayer.duration ?: 0L
                            )
                            lastFiredSec = currentSec
                        }
                    }
                    delay(1000) // Check every second
                }
            } catch (e: Exception) {
                // Job was cancelled or ExoPlayer is released - exit gracefully
                errorLog("Progress tracking stopped: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }

        // Cancel and replace any existing job (like loadingJob) for this playerView
        progressJobMap.remove(playerView)?.cancel()
        progressJobMap[playerView] = job
    }

    private fun fireVideoProgressEvent(
        uclid: String,
        cliUbid: String,
        currentPosition: Long,
        totalDuration: Long
    ) {
        trackVideoProgress(
            uclid,
            cliUbid,
            (currentPosition / 1000).toFloat(),
            (totalDuration / 1000).toFloat()
        )
    }

    private fun toggleMute(
        button: ImageButton,
        videoAd: VideoAd,
        cliUbid: String,
        exoPlayer: ExoPlayer
    ) {
        // Get current mute state for this specific player, default to false (unmuted)
        val currentMuteState = muteStateMap[exoPlayer] ?: false
        val newMuteState = !currentMuteState

        // Update mute state for this specific player
        muteStateMap[exoPlayer] = newMuteState

        val actionType = if (newMuteState) VideoActionType.MUTE else VideoActionType.UNMUTE
        exoPlayer?.volume = if (newMuteState) 0f else 1f
        button.setImageResource(
            getDrawableResource(
                button.context,
                if (newMuteState) "mute" else "unmute"
            )
        )

        val currentPosition = exoPlayer?.currentPosition ?: 0L

        //Call video action event for mute and unmute
        trackVideoAction(
            videoAd.uclid,
            cliUbid,
            actionType,
            (currentPosition / 1000).toFloat()
        )
    }

    private fun checkVisibilityAndHandlePlayback(playerView: PlayerView, exoPlayer: ExoPlayer) {
        val screenLocation = IntArray(2)
        playerView.getLocationOnScreen(screenLocation)

        val screenHeight = Resources.getSystem().displayMetrics.heightPixels
        val viewTop = screenLocation[1]
        val viewBottom = viewTop + playerView.height

        val isVisibleNow = viewTop < screenHeight && viewBottom > 0

        // Thread-safe state access and modification
        synchronized(stateLock) {
            // Get per-player visibility state
            val wasVisible = visibilityStateMap[exoPlayer] ?: false
            val wasPlayerPlaying = wasPlayingStateMap[exoPlayer] ?: false

            if (isVisibleNow && !wasVisible) {
                // Video became visible
                visibilityStateMap[exoPlayer] = true
                if (wasPlayerPlaying) {
                    val playerMuteState = muteStateMap[exoPlayer] ?: false
                    try {
                        exoPlayer.volume = if (playerMuteState) 0f else 1f
                        exoPlayer.play()
                    } catch (e: IllegalStateException) {
                        errorLog(
                            "Cannot control released player on visible: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                        // Clean up state for released player
                        cleanupPlayerState(exoPlayer)
                    }
                } else if (exoPlayer.playWhenReady && !exoPlayer.isPlaying) {
                    val playerMuteState = muteStateMap[exoPlayer] ?: false
                    try {
                        exoPlayer.volume = if (playerMuteState) 0f else 1f
                        wasPlayingStateMap[exoPlayer] = true
                    } catch (e: IllegalStateException) {
                        errorLog(
                            "Cannot set volume on released player: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                        // Clean up state for released player
                        cleanupPlayerState(exoPlayer)
                    }
                }
            } else if (!isVisibleNow && wasVisible) {
                // Video went out of view
                visibilityStateMap[exoPlayer] = false
                if (exoPlayer.isPlaying == true) {
                    try {
                        exoPlayer.pause()
                        // Don't change volume here - preserve individual mute state
                        // The volume should remain as per the player's mute state
                        wasPlayingStateMap[exoPlayer] = true
                    } catch (e: IllegalStateException) {
                        errorLog(
                            "Cannot pause released player: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                        // Clean up state for released player
                        cleanupPlayerState(exoPlayer)
                    }
                }
            }
        }
    }

    fun safeStopAndReleasePlayer(playerView: PlayerView?) {
        if (isReleased.get()) {
            // Bulk release already happened - don't proceed
            return
        }

        playerView?.let { pv ->
            try {
                pv.player?.apply {
                    playWhenReady = false
                    stop()
                    clearVideoSurface()
                }
            } catch (e: IllegalStateException) {
                // Player already released - continue with cleanup
            }
            releasePlayer(pv)
        }
    }

    fun releasePlayer(playerView: PlayerView?) {
        playerView?.let {
            try {
                progressJobMap.remove(it)?.cancel()
                // Remove all listeners to prevent memory leaks
                removeScrollListener(it)
                removePreDrawListener(it)
                removeAttachListener(it)

                // Thread-safe player release
                synchronized(playerLock) {
                    val player = playerMap.remove(it)

                    it.player = null
                    player?.apply {
                        // Thread-safe state cleanup
                        synchronized(stateLock) {
                            releaseAndUntrackExoPlayer(this)
                        }
                    }
                }

            } catch (e: Exception) {
                errorLog("releasePlayer failed: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }
    }


    fun pause(playerView: PlayerView) {
        val player = playerView.player
        player?.pause()
    }

    fun playExclusive(playerView: PlayerView) {
        // Thread-safe exclusive playback
        synchronized(playerLock) {
            // Pause all others
            for ((view, player) in playerMap) {
                if (view != playerView) {
                    try {
                        player.pause()
                    } catch (e: IllegalStateException) {
                        errorLog(
                            "Cannot pause released player during exclusive play: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                    }
                }
            }
            // Play the selected one
            try {
                playerMap[playerView]?.play()
            } catch (e: IllegalStateException) {
                errorLog(
                    "Cannot play released player during exclusive play: ${e.message}",
                    OsmosError.UNKNOWN,
                    e
                )
            }
        }
    }

    /**
     * Clear the view load listener to prevent memory leaks
     */
    private fun clearViewLoadListener() {
        onViewLoad = null
    }

    /**
     * Clear all callbacks to prevent memory leaks
     */
    private fun clearAllCallbacks() {
        onViewLoad = null
        errorCallback = null
    }


    /**
     * Safe version of releaseAllPlayers that handles dead thread scenarios
     */
    fun safeReleaseAllPlayers() {
        if (isReleased.compareAndSet(false, true)) {
            try {
                releaseAllPlayers()
            } catch (e: IllegalStateException) {
                errorLog(
                    "Handler thread is dead, performing minimal cleanup: ${e.message}",
                    OsmosError.UNKNOWN,
                    e
                )
                // Release all tracked ExoPlayer instances as fallback
                releaseAllTrackedPlayers()
            } catch (e: Exception) {
                errorLog("Error during safe release: ${e.message}", OsmosError.UNKNOWN, e)
                // Fallback to releasing all tracked players
                releaseAllTrackedPlayers()
            }
        }
    }
    
    /**
     * Releases all tracked ExoPlayer instances when normal cleanup fails
     * This method ensures no ExoPlayer instances remain in memory during fallback scenarios
     */
    private fun releaseAllTrackedPlayers() {
        try {
            // Create a copy of all tracked players to avoid concurrent modification
            val playersToCleanup = allExoPlayers.toList()
            
            // Clean up each player individually
            playersToCleanup.forEach { player ->
                releaseAndUntrackExoPlayer(player)
            }
            
            // Clear all maps as a final safety measure
            synchronized(playerLock) {
                progressJobMap.clear()
                playerMap.clear()
                // Clear all listeners
                scrollListenerMap.keys.forEach { playerView ->
                    removeScrollListener(playerView)
                }
                scrollListenerMap.clear()
                preDrawListenerMap.keys.forEach { playerView ->
                    removePreDrawListener(playerView)
                }
                preDrawListenerMap.clear()
                attachListenerMap.keys.forEach { playerView ->
                    removeAttachListener(playerView)
                }
                attachListenerMap.clear()
                playerListenerMap.keys.forEach { exoPlayer ->
                    removePlayerListener(exoPlayer)
                }
                playerListenerMap.clear()
            }
            synchronized(stateLock) {
                muteStateMap.clear()
                visibilityStateMap.clear()
                wasPlayingStateMap.clear()
                lifecycleObserverMap.values.forEach { it.clearErrorCallback() }
                lifecycleObserverMap.clear()
            }
            
            // Clear the central registry
            allExoPlayers.clear()
            clearAllCallbacks()
            
        } catch (e: Exception) {
            errorLog("Error during guaranteed cleanup: ${e.message}", OsmosError.UNKNOWN, e)
            // Force clear all collections as last resort
            try {
                allExoPlayers.clear()
                playerMap.clear()
                progressJobMap.clear()
                scrollListenerMap.clear()
                preDrawListenerMap.clear()
                attachListenerMap.clear()
                playerListenerMap.clear()
                muteStateMap.clear()
                visibilityStateMap.clear()
                wasPlayingStateMap.clear()
                lifecycleObserverMap.clear()
                clearAllCallbacks()
            } catch (e2: Exception) {
                errorLog("Error during force cleanup: ${e2.message}", OsmosError.UNKNOWN, e2)
            }
        }
    }

    private fun releaseAllPlayers() {
        try {
            // Cancel progress jobs with dead thread protection
            try {
                progressJobMap.values.forEach { it.cancel() }
                progressJobMap.clear()
            } catch (e: IllegalStateException) {
                progressJobMap.clear()
            } catch (e: Exception) {
                progressJobMap.clear()
            }

            clearViewLoadListener()

            // Clear player references with dead thread protection
            try {
                playerMap.keys.forEach { playerView ->
                    try {
                        playerView.player = null
                    } catch (e: IllegalStateException) {
                        // Handler thread might be dead, skip UI operation
                        errorLog(
                            "Handler thread is dead, skipping playerView.player = null: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                    } catch (e: Exception) {
                        errorLog(
                            "Error setting playerView.player = null: ${e.message}",
                            OsmosError.UNKNOWN,
                            e
                        )
                    }
                }
            } catch (e: IllegalStateException) {
                errorLog(
                    "Handler thread is dead, skipping all playerView operations: ${e.message}",
                    OsmosError.UNKNOWN,
                    e
                )
            } catch (e: Exception) {
                errorLog("Error iterating playerMap: ${e.message}", OsmosError.UNKNOWN, e)
            }

            // Use safe release for all tracked players
            val playersToCleanup = allExoPlayers.toList()
            playersToCleanup.forEach { player ->
                releaseAndUntrackExoPlayer(player)
            }
            
            // Set all PlayerView.player references to null
            playerMap.keys.forEach { playerView ->
                try {
                    playerView.player = null
                } catch (e: Exception) {
                    errorLog("Error setting playerView.player = null: ${e.message}", OsmosError.UNKNOWN, e)
                }
            }
            // Clear all collections (should already be cleared by guaranteedExoPlayerCleanup)
            playerMap.clear()
            scrollListenerMap.clear()
            preDrawListenerMap.clear()
            attachListenerMap.clear()
            playerListenerMap.clear()
            muteStateMap.clear()
            visibilityStateMap.clear()
            wasPlayingStateMap.clear()
            lifecycleObserverMap.values.forEach { it.clearErrorCallback() }
            lifecycleObserverMap.clear()
            allExoPlayers.clear()
        } catch (e: Exception) {
            errorLog("Error in releaseAllPlayers: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    /**
     * Helper method to track impressions using the common interface.
     */
    private fun trackImpression(uclid: String, cliUbid: String) {
        adTracker.trackImpression(uclid, cliUbid)
    }

    /**
     * Helper method to track ad clicks using the common interface.
     */
    private fun trackAdClick(uclid: String, cliUbid: String) {
        adTracker.trackAdClick(uclid, cliUbid)
    }

    /**
     * Helper method to track video progress events using the common interface.
     */
    private fun trackVideoProgress(
        uclid: String,
        cliUbid: String,
        videoViewSec: Float,
        videoDurationSec: Float
    ) {
        adTracker.videoProgressEvent(uclid, cliUbid, videoViewSec, videoDurationSec)
    }

    /**
     * Helper method to track video action events using the common interface.
     */
    private fun trackVideoAction(
        uclid: String,
        cliUbid: String,
        actionType: VideoActionType,
        videoViewSec: Float
    ) {
        adTracker.videoActionClick(uclid, cliUbid, actionType, videoViewSec)
    }

    /**
     * Helper function to get drawable resource ID by name for library projects
     */
    private fun getDrawableResource(context: Context, resourceName: String): Int {
        return context.resources.getIdentifier(resourceName, "drawable", context.packageName)
    }
}
