package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.renderer.loaders.InterstitialLoader
import com.ai.osmos.ads.views.interfaces.InterstitialAdViewInterface
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.AdType
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.error.ErrorCallback
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: InterstitialAdView
 */


/**
 * InterstitialAdView.
 *
 * @param context Application context.
 * @param adData Popup ad Data (defined data structure of ad)
 * @param width(optional): Width of ad view
 * @param height(optional): Height of ad view
 * @param alignment(optional): alignment of ad view ("CENTER" or "BOTTOM")
 * @param coordinates(optional): object containing the X and Y coordinates for absolute positioning of the popup.
 * @param attrs(optional): Attrs for ad view
 * @param errorCallback(optional): Callback for handling errors
 * @return InterstitialAdView or EmptyView
 * Example:
 * ```
 * val adView = InterstitialAdView(this@MainActivity,
 *                 adData,
 *                 width = 300,                                     //optional
 *                 height = 200,                                    //optional
 *                 coordinates = Coordinates(x = 10, y = 10),       //optional
 *                 alignment = Gravity.CENTER,                      //optional
 *                 errorCallback = object : ErrorCallback {
 *                     override fun onError(errorCode: String, errorMessage: String, throwable: Throwable?) {
 *                         Log.e("InterstitialAd", "Error: $errorCode - $errorMessage")
 *                     }
 *                 })
 * ```
 */

@SuppressLint("ViewConstructor")
internal class InterstitialAdView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private val width: Int? = null,
    private val height: Int? = null,
    private val alignment: Int? = Gravity.CENTER, // "CENTER", "BOTTOM"
    private var coordinates: Coordinates? = Coordinates(0, 0),
    attrs: AttributeSet? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config
) : BaseAdView(context, attrs, config), InterstitialAdViewInterface {

    private val persistentAdTracker by lazy {
        if (context.applicationContext != null) {
            PersistentAdTracker.create(context.applicationContext, coroutineScope, config)
        } else {
            PersistentAdTracker.create(context, coroutineScope, config)
        }
    }
    private val interstitialLoader = InterstitialLoader(coroutineScope, persistentAdTracker)

    init {
        errorCallback?.let {
            setErrorCallback(it)
            persistentAdTracker.setErrorCallback(it)
            // Pass error callback to InterstitialLoader's internal loaders
            interstitialLoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, context) else showEmptyView(context)
    }

    override fun onDetachedFromWindow() {
        interstitialLoader.cleanup()
        clearAllErrorCallbacks()
        super.onDetachedFromWindow()
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     */
    private fun clearAllErrorCallbacks() {
        interstitialLoader.clearErrorCallback()
        persistentAdTracker.setErrorCallback(null)
        clearErrorCallback()
    }

    /**
     * Loads and displays an ad as a popup based on the given ad data.
     *
     * @param adData A map containing the ad object and its associated metadata.
     *               Expected to include a key `"ad"` with value of type `BaseAd`.
     */
    override fun loadAd(adData: Map<String, Any>, context: Context) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? BaseAd

                if (ad != null) {
                    val cliUbid = getCliUbid(ad)
                    if (isSupportedCrtImage(ad.crt) || isSupportedCrtVideo(ad.crt)) {
                        withContext(Dispatchers.Main) {
                            when (ad.elementsType) {

                                AdType.IMAGE -> {
                                    val imageAd = ad as? ImageAd

                                    if (!imageAd?.elements?.value.isNullOrEmpty()) {
                                        if (isSupportedImageFormat(ad)) {
                                            // Since we've already checked imageAd is not null, we can safely use it
                                            imageAd?.let { nonNullImageAd ->
                                                val (screenWidth, screenHeight) = getScreenHeightWidth()

                                                val h = height ?: (nonNullImageAd.elements.height
                                                    ?: screenHeight)
                                                val w = width ?: (nonNullImageAd.elements.width
                                                    ?: screenWidth)

                                                val result = setMaxHeightWidth(h, w, coordinates)

                                                coordinates = result.coordinates

                                                val (width, height) = getAdDimensions(
                                                    result.finalWidth,
                                                    result.finalHeight,
                                                    nonNullImageAd.elements.width ?: screenWidth,
                                                    nonNullImageAd.elements.height ?: screenHeight
                                                )

                                                interstitialLoader.showPopup(
                                                    context = context,
                                                    ad = nonNullImageAd,
                                                    cliUbid = cliUbid,
                                                    width = width,
                                                    height = height,
                                                    alignment = alignment,
                                                    coordinates = coordinates,
                                                    isClamped = result.isClamped,
                                                    adClickListener = adClickedListener,
                                                    closeButtonSec = imageAd.elements.closeButtonSec
                                                        ?: Constants.CLOSE_BUTTON_SEC_DEFAULT ?: 5
                                                )
                                            }
                                        } else showEmptyView(context)
                                    } else {
                                        showEmptyView(context)
                                    }
                                }

                                AdType.VIDEO -> {
                                    val videoAd = ad as? VideoAd
                                    if (!videoAd?.elements?.value.isNullOrEmpty()) {
                                        if (isSupportedVideoFormat(ad)) {
                                            // Since we've already checked videoAd is not null, we can safely use it
                                            videoAd?.let { nonNullVideoAd ->
                                                val (screenWidth, screenHeight) = getScreenHeightWidth()

                                                val h = height ?: (nonNullVideoAd.elements.height
                                                    ?: screenHeight)
                                                val w = width ?: (nonNullVideoAd.elements.width
                                                    ?: screenWidth)

                                                val result = setMaxHeightWidth(h, w, coordinates)

                                                coordinates = result.coordinates
                                                val (width, height) = getAdDimensions(
                                                    result.finalWidth,
                                                    result.finalHeight,
                                                    nonNullVideoAd.elements.width ?: screenWidth,
                                                    nonNullVideoAd.elements.height ?: screenHeight
                                                )
                                                interstitialLoader.showPopup(
                                                    context = context,
                                                    ad = nonNullVideoAd,
                                                    cliUbid = cliUbid,
                                                    width = width,
                                                    height = height,
                                                    alignment = alignment,
                                                    coordinates = coordinates,
                                                    isClamped = result.isClamped,
                                                    adClickListener = adClickedListener,
                                                    closeButtonSec = videoAd.elements.closeButtonSec
                                                        ?: Constants.CLOSE_BUTTON_SEC_DEFAULT ?: 5
                                                )
                                            }
                                        } else showEmptyView(context)
                                    } else showEmptyView(context)
                                }

                                else -> {
                                    showEmptyView(context)
                                }
                            }
                        }
                    } else {
                        withContext(Dispatchers.Main) {
                            showEmptyView(context)
                        }
                    }
                } else {
                    withContext(Dispatchers.Main) {
                        showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    showEmptyView(context)
                }
            }
        }
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {
        // Empty implementation for activity context
    }

    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}