package com.ai.osmos.models.events

/**
 * Immutable tracking event parameters class following proper builder pattern
 */
class TrackingParams private constructor(
    private val parameters: Map<String, Any>
) {

    /**
     * Convert to dictionary for internal use
     */
    fun toMap(): Map<String, Any> {
        return parameters
    }

    /**
     * Merge with another TrackingParams object - returns new immutable instance
     */
    fun merge(other: Map<String, Any>): TrackingParams {
        val mergedParams = parameters.toMutableMap()
        other.forEach { (key, value) ->
            mergedParams[key] = value
        }
        return TrackingParams(mergedParams)
    }

    // MARK: - Static Factory Methods

    companion object {
        /**
         * Creates a new builder instance - PROPER BUILDER PATTERN
         */
        @JvmStatic
        fun builder(): Builder {
            return Builder()
        }

        /**
         * Creates an empty TrackingParams instance
         */
        @JvmStatic
        fun empty(): TrackingParams {
            return TrackingParams(emptyMap())
        }
    }

    /**
     * Builder class for tracking event parameters with fluent interface
     */
    class Builder {
        private val parameters: MutableMap<String, Any> = mutableMapOf()

        // MARK: - Product Parameters

        /**
         * Set SKU identifier for the product
         */
        fun skuId(value: String): Builder {
            parameters["sku_id"] = value
            return this
        }

        /**
         * Set product prices (can be single price or pipe-separated prices for multiple products)
         */
        fun productPrices(value: String?): Builder {
            value?.let { parameters["product_prices"] = it }
            return this
        }

        /**
         * Set product quantities (can be single quantity or pipe-separated quantities for multiple products)
         */
        fun productQuantities(value: String): Builder {
            parameters["product_quantities"] = value
            return this
        }

        // MARK: - Seller/Store Parameters

        /**
         * Set seller/merchant identifier
         */
        fun sellerId(value: String?): Builder {
            value?.let { parameters["seller_id"] = it }
            return this
        }

        /**
         * Set store identifier (used in grocery vertical)
         */
        fun storeId(value: String): Builder {
            parameters["store_id"] = value
            return this
        }

        // MARK: - Ad Parameters

        /**
         * Set unique click identifier from ad server
         */
        fun uclid(value: String): Builder {
            parameters["uclid"] = value
            return this
        }

        /**
         * Set ad position rank
         */
        fun position(value: Int?): Builder {
            value?.let { parameters["position"] = it }
            return this
        }

        // MARK: - Video Parameters

        /**
         * Set video view duration in seconds
         */
        fun videoViewSec(value: Float): Builder {
            parameters["video_view_sec"] = value
            return this
        }

        /**
         * Set total video duration in seconds
         */
        fun videoDurationSec(value: Float): Builder {
            parameters["video_duration_sec"] = value
            return this
        }

        /**
         * Set video action type (mute, unmute, pause, play, etc.)
         */
        fun actionType(value: VideoActionType): Builder {
            parameters["action_type"] = value.actionType
            return this
        }

        // MARK: - Purchase Parameters

        /**
         * Set source for offline purchase sync
         */
        fun source(value: String): Builder {
            parameters["source"] = value
            return this
        }


        // MARK: - Extensibility Methods

        /**
         * Add a custom key-value pair
         */
        fun add(key: String, value: Any): Builder {
            parameters[key] = value
            return this
        }

        /**
         * Merge a dictionary of parameters
         */
        fun merge(dictionary: Map<String, Any>): Builder {
            dictionary.forEach { (key, value) ->
                parameters[key] = value
            }
            return this
        }

        // MARK: - Build Method

        /**
         * Convert builder state to map (useful for checking current state)
         */
        fun toMap(): Map<String, Any> {
            return parameters.toMap()
        }

        /**
         * Build the final immutable TrackingParams object
         */
        fun build(): TrackingParams {
            return TrackingParams(parameters.toMap())
        }
    }
}