package com.ai.osmos.network

import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.common.HttpMethod
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.logging.DebugLogger
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody
import org.json.JSONException
import org.json.JSONObject
import java.io.IOException
import java.util.concurrent.TimeUnit

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: WebService
 */


/**
 * Handles API requests for fetching ads from a remote server using OkHttp.
 */
internal class WebService {

    companion object {
        private val JSON_MEDIA_TYPE = "application/json; charset=utf-8".toMediaType()
    }

    private val okHttpClient: OkHttpClient by lazy {
        val builder = OkHttpClient.Builder()
            .connectTimeout(Constants.CONNECT_TIMEOUT.toLong(), TimeUnit.MILLISECONDS)
            .readTimeout(Constants.READ_TIMEOUT.toLong(), TimeUnit.MILLISECONDS)
            .writeTimeout(Constants.READ_TIMEOUT.toLong(), TimeUnit.MILLISECONDS)
            .retryOnConnectionFailure(true)

        // Note: Manual logging is done in callApi method instead of using interceptor
        builder.build()
    }

    /**
     * Makes an API request using OkHttp and throws WebServiceException on error.
     *
     * @param apiUrl The URL for the request.
     * @param methodType The HTTP request method from HttpMethod enum.
     * @param requestBody The request body as a Map (converted to JSON).
     * @param requestHeaders The request headers as a Map.
     * @return The successful response as a Map<String,Any>.
     * @throws ExceptionHandler if the request fails.
     */
    internal suspend fun callApiWithException(
        apiUrl: String,
        methodType: HttpMethod,
        requestHeaders: Map<String, Any>? = null,
        requestBody: Map<String, Any?>? = null
    ): Map<String, Any> {
        return withContext(Dispatchers.IO) {
            try {
                DebugLogger.log("🌐 HTTP ${methodType.method}: $apiUrl")
                if (requestBody != null) DebugLogger.log("📦 BODY: ${JSONObject(requestBody)}")

                val request = buildRequest(apiUrl, methodType.method, requestHeaders, requestBody)
                val response = okHttpClient.newCall(request).execute()

                response.use { res ->
                    val responseCode = res.code
                    val responseBody = res.body?.string() ?: ""
                    val noContentStatusCodes = setOf(204, 205, 304)

                    DebugLogger.log(
                        "📡 RESPONSE ${responseCode}: ${
                            if (responseBody.length > 200) responseBody.take(
                                200
                            ) + "..." else responseBody
                        }"
                    )

                    if (res.isSuccessful) {
                        if (responseCode in noContentStatusCodes || responseBody.isEmpty()) {
                            createResponse(true, responseCode, "", null)
                        } else {
                            createResponse(true, responseCode, responseBody, null)
                        }
                    } else {
                        val errorMessage = if (responseBody.isNotEmpty()) {
                            responseBody
                        } else {
                            "HTTP $responseCode: ${res.message}"
                        }
                        throw ExceptionHandler(
                            OsmosError.BAD_STATUS,
                            errorMessage
                        )
                    }
                }
            } catch (e: java.net.MalformedURLException) {
                throw ExceptionHandler(
                    OsmosError.BAD_URL,
                    e.message ?: OsmosError.BAD_URL.message,
                    e
                )
            } catch (e: java.net.SocketTimeoutException) {
                throw ExceptionHandler(
                    OsmosError.TIMEOUT,
                    e.message ?: OsmosError.TIMEOUT.message,
                    e
                )
            } catch (e: IOException) {
                throw ExceptionHandler(
                    OsmosError.NETWORK_FAILURE,
                    e.message ?: OsmosError.NETWORK_FAILURE.message,
                    e
                )
            } catch (e: IllegalArgumentException) {
                throw ExceptionHandler(
                    OsmosError.BAD_URL,
                    e.message ?: OsmosError.BAD_URL.message,
                    e
                )
            } catch (e: JSONException) {
                throw ExceptionHandler(
                    OsmosError.INVALID_REQUEST,
                    e.message ?: OsmosError.INVALID_REQUEST.message,
                    e
                )
            } catch (e: ExceptionHandler) {
                // Re-throw ExceptionHandler as-is
                throw e
            } catch (e: Exception) {
                throw ExceptionHandler(
                    OsmosError.UNKNOWN,
                    e.message ?: OsmosError.UNKNOWN.message,
                    e
                )
            }
        }
    }


    /**
     * Builds an OkHttp Request object based on the provided parameters.
     */
    private fun buildRequest(
        apiUrl: String,
        methodType: String,
        requestHeaders: Map<String, Any>?,
        requestBody: Map<String, Any?>?
    ): Request {
        val requestBuilder = Request.Builder()
            .url(apiUrl)
            .addHeader("Accept", "application/json")

        // Add custom headers if provided
        requestHeaders?.forEach { (key, value) ->
            requestBuilder.addHeader(key, value.toString())
        }

        // Set request method and body
        when (methodType.uppercase()) {
            HttpMethod.GET.method -> requestBuilder.get()
            HttpMethod.POST.method -> {
                val body = if (!requestBody.isNullOrEmpty()) {
                    val jsonString = JSONObject(requestBody).toString()
                    RequestBody.create(JSON_MEDIA_TYPE, jsonString)
                } else {
                    RequestBody.create(JSON_MEDIA_TYPE, "{}")
                }
                requestBuilder.post(body)
            }

            else -> throw IllegalArgumentException("Unsupported HTTP method: $methodType")
        }

        return requestBuilder.build()
    }

    /**
     * Creates a standardized response JSON.
     */
    private fun createResponse(
        status: Boolean,
        code: Int,
        data: String?,
        error: String?
    ): Map<String, Any> {
        return mutableMapOf<String, Any>().apply {
            put(Constants.JSON_KEY_STATUS, status)

            put(
                Constants.JSON_KEY_RESPONSE, mapOf(
                    Constants.JSON_KEY_CODE to code,
                    Constants.JSON_KEY_DATA to (data ?: "")
                )
            )

            put(Constants.JSON_KEY_ERROR, error ?: JSONObject.NULL)
        }
    }
}