package com.ai.osmos.tracking.data

import androidx.work.Data

/**
 * Data class for storing ad click tracking information that can be
 * manually serialized for WorkManager without external dependencies.
 *
 * This class handles the conversion between Kotlin data types and WorkManager's
 * Data format, ensuring reliable persistence of ad click tracking data.
 */
data class AdClickData(
    val uclid: String,
    val cliUbid: String,
    val position: Int = 1,
    val timestamp: Long = System.currentTimeMillis(),
    val attemptCount: Int = 0,
    val maxRetries: Int = 3
) {

    companion object {
        // Data keys for WorkManager serialization
        private const val KEY_UCLID = "uclid"
        private const val KEY_CLI_UBID = "cli_ubid"
        private const val KEY_POSITION = "position"
        private const val KEY_TIMESTAMP = "timestamp"
        private const val KEY_ATTEMPT_COUNT = "attempt_count"
        private const val KEY_MAX_RETRIES = "max_retries"

        const val WORK_TAG_AD_CLICK = "ad_click_tracking"

        /**
         * Creates AdClickData from WorkManager Data input.
         * Returns null if required fields are missing or data is malformed.
         */
        fun fromWorkData(data: Data): AdClickData? {
            return try {
                val uclid = data.getString(KEY_UCLID) ?: return null
                val cliUbid = data.getString(KEY_CLI_UBID) ?: return null


                AdClickData(
                    uclid = uclid,
                    cliUbid = cliUbid,
                    position = data.getInt(KEY_POSITION, 1),
                    timestamp = data.getLong(KEY_TIMESTAMP, System.currentTimeMillis()),
                    attemptCount = data.getInt(KEY_ATTEMPT_COUNT, 0),
                    maxRetries = data.getInt(KEY_MAX_RETRIES, 3)
                )
            } catch (e: Exception) {
                null
            }
        }

        /**
         * Deserializes extra parameters from string format.
         * Format: "key1=value1;key2=value2;key3=value3"
         */
    }

    /**
     * Converts this AdClickData to WorkManager Data format.
     */
    fun toWorkData(): Data {

        return Data.Builder()
            .putString(KEY_UCLID, uclid)
            .putString(KEY_CLI_UBID, cliUbid)
            .putInt(KEY_POSITION, position)
            .putLong(KEY_TIMESTAMP, timestamp)
            .putInt(KEY_ATTEMPT_COUNT, attemptCount)
            .putInt(KEY_MAX_RETRIES, maxRetries)
            .build()
    }


    /**
     * Creates a new AdClickData with incremented attempt count.
     */
    fun incrementAttempt(): AdClickData = copy(attemptCount = attemptCount + 1)

    /**
     * Checks if this ad click should be retried based on attempt count.
     */
    fun shouldRetry(): Boolean = attemptCount < maxRetries

}