package com.ai.osmos.tracking.data

import androidx.work.Data

/**
 * Data class for storing video progress tracking information that can be
 * manually serialized for WorkManager without external dependencies.
 *
 * This class handles the conversion between Kotlin data types and WorkManager's
 * Data format, ensuring reliable persistence of video progress tracking data.
 */
data class VideoProgressData(
    val uclid: String,
    val cliUbid: String,
    val videoViewSec: Float,
    val videoDurationSec: Float,
    val position: Int = 1,
    val timestamp: Long = System.currentTimeMillis(),
    val attemptCount: Int = 0,
    val maxRetries: Int = 3
) {

    companion object {
        // Data keys for WorkManager serialization
        private const val KEY_UCLID = "uclid"
        private const val KEY_CLI_UBID = "cli_ubid"
        private const val KEY_VIDEO_VIEW_SEC = "video_view_sec"
        private const val KEY_VIDEO_DURATION_SEC = "video_duration_sec"
        private const val KEY_POSITION = "position"
        private const val KEY_TIMESTAMP = "timestamp"
        private const val KEY_ATTEMPT_COUNT = "attempt_count"
        private const val KEY_MAX_RETRIES = "max_retries"

        const val WORK_TAG_VIDEO_PROGRESS = "video_progress_tracking"

        /**
         * Creates VideoProgressData from WorkManager Data input.
         * Returns null if required fields are missing or data is malformed.
         */
        fun fromWorkData(data: Data): VideoProgressData? {
            return try {
                val uclid = data.getString(KEY_UCLID) ?: return null
                val cliUbid = data.getString(KEY_CLI_UBID) ?: return null


                VideoProgressData(
                    uclid = uclid,
                    cliUbid = cliUbid,
                    videoViewSec = data.getFloat(KEY_VIDEO_VIEW_SEC, 0f),
                    videoDurationSec = data.getFloat(KEY_VIDEO_DURATION_SEC, 0f),
                    position = data.getInt(KEY_POSITION, 1),
                    timestamp = data.getLong(KEY_TIMESTAMP, System.currentTimeMillis()),
                    attemptCount = data.getInt(KEY_ATTEMPT_COUNT, 0),
                    maxRetries = data.getInt(KEY_MAX_RETRIES, 3)
                )
            } catch (e: Exception) {
                null
            }
        }

        /**
         * Deserializes extra parameters from string format.
         * Format: "key1=value1;key2=value2;key3=value3"
         */
    }

    /**
     * Converts this VideoProgressData to WorkManager Data format.
     */
    fun toWorkData(): Data {

        return Data.Builder()
            .putString(KEY_UCLID, uclid)
            .putString(KEY_CLI_UBID, cliUbid)
            .putFloat(KEY_VIDEO_VIEW_SEC, videoViewSec)
            .putFloat(KEY_VIDEO_DURATION_SEC, videoDurationSec)
            .putInt(KEY_POSITION, position)
            .putLong(KEY_TIMESTAMP, timestamp)
            .putInt(KEY_ATTEMPT_COUNT, attemptCount)
            .putInt(KEY_MAX_RETRIES, maxRetries)
            .build()
    }

    /**
     * Serializes extra parameters to string format.
     */

    /**
     * Creates a new VideoProgressData with incremented attempt count.
     */
    fun incrementAttempt(): VideoProgressData = copy(attemptCount = attemptCount + 1)

    /**
     * Checks if this video progress should be retried based on attempt count.
     */
    fun shouldRetry(): Boolean = attemptCount < maxRetries

    /**
     * Converts Map<String, Any> to Map<String, String> for serialization.
     * to our string-only format required for WorkManager serialization.
     */
}