package com.ai.osmos.tracking.tracker

import androidx.work.Data
import com.ai.osmos.core.Config
import com.ai.osmos.models.events.VideoActionType
import com.ai.osmos.tracking.events.RegisterEvent
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: BaseAdTracker
 *
 * Base implementation for all ad tracking functionality.
 * Eliminates code duplication between AdTracker and PersistentAdTracker.
 */
abstract class BaseAdTracker(
    private val config: Config
) : AdTrackerInterface {

    private val registerEvent = RegisterEvent(config)
    private var _errorCallback: ErrorCallback? = null

    override fun setErrorCallback(callback: ErrorCallback?) {
        this._errorCallback = callback
    }

    /**
     * Logs errors consistently across all tracker implementations.
     */
    protected fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        _errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Common implementation for immediate ad click tracking.
     * Used by both AdTracker and PersistentAdTracker.
     */
    protected suspend fun executeAdClick(uclid: String, cliUbid: String) {
        try {
            registerEvent.registerAdClickEvent(
                cliUbid = cliUbid,
                uclid = uclid,
            )
        } catch (e: Exception) {
            errorLog("Click tracking failed: ${e.message}", OsmosError.NETWORK_FAILURE, e)
        }
    }

    /**
     * Common implementation for video action tracking.
     */
    protected suspend fun executeVideoAction(
        uclid: String,
        cliUbid: String,
        actionType: VideoActionType,
        videoViewSec: Float
    ) {
        try {
            registerEvent.registerVideoActionEvent(
                cliUbid = cliUbid,
                uclid = uclid,
                videoViewSec = videoViewSec,
                actionType = actionType,
            )
        } catch (e: Exception) {
            errorLog("Video action tracking failed: ${e.message}", OsmosError.NETWORK_FAILURE, e)
        }
    }

    /**
     * Common implementation for video progress tracking.
     */
    protected suspend fun executeVideoProgress(
        uclid: String,
        cliUbid: String,
        videoViewSec: Float,
        videoDurationSec: Float
    ) {
        try {
            registerEvent.registerVideoProgressEvent(
                cliUbid = cliUbid,
                uclid = uclid,
                videoViewSec = videoViewSec,
                videoDurationSec = videoDurationSec,
            )
        } catch (e: Exception) {
            errorLog("Video progress tracking failed: ${e.message}", OsmosError.NETWORK_FAILURE, e)
        }
    }

    /**
     * Common implementation for immediate impression tracking.
     * Used as fallback in PersistentAdTracker.
     */
    protected suspend fun executeImpression(uclid: String, cliUbid: String, position: Int) {
        try {
            registerEvent.registerAdImpressionEvent(
                cliUbid = cliUbid,
                uclid = uclid,
                position = position,
            )
        } catch (e: Exception) {
            errorLog("Impression tracking failed: ${e.message}", OsmosError.NETWORK_FAILURE, e)
        }
    }

    companion object {
        /**
         * Centralized method to extract config from WorkData.
         * This eliminates code duplication across all WorkManager worker classes.
         */
        fun getConfigFromWorkData(workData: Data): Config {
            val clientId = workData.getString("clientId") ?: ""
            val debug = workData.getBoolean("debug", false)
            val domainUri = workData.getString("eventTrackingHost") ?: ""

            return Config(
                clientId = clientId,
                debug = debug,
                eventTrackingHost = domainUri,
                displayAdsHost = "",
                productAdsHost = ""
            )
        }
    }
}