package com.ai.osmos.tracking.tracker

import android.content.Context
import androidx.work.BackoffPolicy
import androidx.work.Constraints
import androidx.work.Data
import androidx.work.NetworkType
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkInfo
import androidx.work.WorkManager
import com.ai.osmos.core.Config
import com.ai.osmos.models.events.VideoActionType
import com.ai.osmos.tracking.data.AdClickData
import com.ai.osmos.tracking.data.ImpressionData
import com.ai.osmos.tracking.data.VideoActionData
import com.ai.osmos.tracking.data.VideoProgressData
import com.ai.osmos.tracking.tracker.workers.AdClickTrackingWorker
import com.ai.osmos.tracking.tracker.workers.ImpressionTrackingWorker
import com.ai.osmos.tracking.tracker.workers.VideoActionTrackingWorker
import com.ai.osmos.tracking.tracker.workers.VideoProgressTrackingWorker
import kotlinx.coroutines.CoroutineScope
import java.util.concurrent.TimeUnit

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: PersistentAdTracker
 */

/**
 * Enhanced Ad Tracker that provides persistent impression tracking using WorkManager.
 *
 * This class acts as a drop-in replacement for AdTracker, adding reliability features:
 * - Persistent tracking across app restarts
 * - Network-aware retry mechanisms
 * - Battery-optimized background execution
 * - Graceful fallback when WorkManager is unavailable
 *
 * Now extends BaseAdTracker to eliminate code duplication.
 */
class PersistentAdTracker(
    private val context: Context,
    private val fallbackTracker: AdTracker? = null,
    private val config: Config
) : BaseAdTracker(config) {

    private val isWorkManagerAvailable: Boolean by lazy {
        isWorkManagerClassAvailable()
    }


    /**
     * Tracks an ad impression with persistent retry capability.
     *
     * This method provides enhanced reliability compared to immediate tracking:
     * - Uses WorkManager for background persistence if available
     * - Falls back to immediate tracking if WorkManager is not available
     *
     * @param uclid Unique creative ID
     * @param cliUbid Client-side user identifier
     * @param position Ad position in the view (default: 1)
     */
    override fun trackImpression(
        uclid: String,
        cliUbid: String,
        position: Int,
    ) {
        if (isWorkManagerAvailable) {
            trackImpressionPersistent(uclid, cliUbid, position)
        } else {
            // Fallback to immediate tracking when WorkManager is not available
            fallbackTracker?.trackImpression(uclid, cliUbid, position)
        }
    }

    /**
     * Legacy method for backward compatibility with existing AdTracker interface.
     * Maintains the same signature as the original AdTracker.trackImpression method.
     */
    fun trackImpression(uclid: String, cliUbid: String) {
        trackImpression(uclid, cliUbid, 1)
    }

    /**
     * Internal method that handles WorkManager-based persistent tracking.
     *
     * Creates a WorkManager job with network constraints and exponential backoff.
     * The job will persist across app restarts and wait for network connectivity.
     */
    private fun trackImpressionPersistent(
        uclid: String,
        cliUbid: String,
        position: Int,
    ) {
        try {
            // Create impression data
            val impressionData = ImpressionData(
                uclid = uclid,
                cliUbid = cliUbid,
                position = position,
            )

            // Create WorkManager request with network constraints and config data
            val baseWorkData = impressionData.toWorkData()
            val workDataBuilder = Data.Builder()
            baseWorkData.keyValueMap.forEach { (key, value) ->
                when (value) {
                    is String -> workDataBuilder.putString(key, value)
                    is Boolean -> workDataBuilder.putBoolean(key, value)
                    is Int -> workDataBuilder.putInt(key, value)
                    is Long -> workDataBuilder.putLong(key, value)
                    is Float -> workDataBuilder.putFloat(key, value)
                    is Double -> workDataBuilder.putDouble(key, value)
                    else -> workDataBuilder.putString(key, value.toString())
                }
            }
            val workDataWithConfig = workDataBuilder
                .putString("clientId", config.clientId)
                .putBoolean("debug", config.debug ?: false)
                .putString("eventTrackingHost", config.eventTrackingHost)
                .build()

            val workRequest = OneTimeWorkRequestBuilder<ImpressionTrackingWorker>()
                .setInputData(workDataWithConfig)
                .setConstraints(
                    Constraints.Builder()
                        .setRequiredNetworkType(NetworkType.CONNECTED)
                        .build()
                )
                .setBackoffCriteria(
                    BackoffPolicy.EXPONENTIAL,
                    30, TimeUnit.SECONDS
                )
                .addTag(ImpressionData.WORK_TAG_IMPRESSION)
                .build()

            // Enqueue the work for background execution
            WorkManager.getInstance(context).enqueue(workRequest)
        } catch (e: Exception) {
            errorLog(
                "Failed to queue impression for WorkManager: ${e.message}",
                com.ai.osmos.utils.error.OsmosError.UNKNOWN,
                e
            )
            // Fallback to immediate tracking on error
            fallbackTracker?.trackImpression(uclid, cliUbid, position)
        }
    }

    /**
     * Tracks when the user clicks on an ad with persistent retry capability.
     *
     * This method provides enhanced reliability compared to immediate tracking:
     * - Uses WorkManager for background persistence if available
     * - Falls back to immediate tracking if WorkManager is not available
     * - Ensures ad clicks are tracked even during network failures
     */
    override fun trackAdClick(uclid: String, cliUbid: String) {
        if (isWorkManagerAvailable) {
            trackAdClickPersistent(uclid, cliUbid)
        } else {
            // Fallback to immediate tracking when WorkManager is not available
            fallbackTracker?.trackAdClick(uclid, cliUbid)
        }
    }

    /**
     * Tracks video-related actions with persistent retry capability.
     *
     * This method provides enhanced reliability for video action tracking:
     * - Uses WorkManager for background persistence if available
     * - Falls back to immediate tracking if WorkManager is not available
     * - Ensures video actions (mute/unmute, fullscreen, etc.) are tracked reliably
     */
    override fun videoActionClick(
        uclid: String,
        cliUbid: String,
        actionType: VideoActionType,
        videoViewSec: Float
    ) {
        if (isWorkManagerAvailable) {
            trackVideoActionPersistent(uclid, cliUbid, actionType, videoViewSec)
        } else {
            // Fallback to immediate tracking when WorkManager is not available
            fallbackTracker?.videoActionClick(uclid, cliUbid, actionType, videoViewSec)
        }
    }

    /**
     * Tracks video progress events with persistent retry capability.
     *
     * This method provides enhanced reliability for video progress tracking:
     * - Uses WorkManager for background persistence if available
     * - Falls back to immediate tracking if WorkManager is not available
     * - Ensures video progress events are tracked reliably for analytics
     */
    override fun videoProgressEvent(
        uclid: String,
        cliUbid: String,
        videoViewSec: Float,
        videoDurationSec: Float
    ) {
        if (isWorkManagerAvailable) {
            trackVideoProgressPersistent(uclid, cliUbid, videoViewSec, videoDurationSec)
        } else {
            // Fallback to immediate tracking when WorkManager is not available
            fallbackTracker?.videoProgressEvent(uclid, cliUbid, videoViewSec, videoDurationSec)
        }
    }

    /**
     * Sets error callback on both this tracker and the fallback tracker.
     */
    override fun setErrorCallback(callback: com.ai.osmos.utils.error.ErrorCallback?) {
        super.setErrorCallback(callback)
        fallbackTracker?.setErrorCallback(callback)
    }

    /**
     * Internal method that handles WorkManager-based persistent ad click tracking.
     *
     * Creates a WorkManager job with network constraints and exponential backoff.
     * The job will persist across app restarts and wait for network connectivity.
     */
    private fun trackAdClickPersistent(uclid: String, cliUbid: String) {
        try {
            val adClickData = AdClickData(
                uclid = uclid,
                cliUbid = cliUbid
            )

            val baseWorkData = adClickData.toWorkData()
            val workDataBuilder = Data.Builder()
            baseWorkData.keyValueMap.forEach { (key, value) ->
                when (value) {
                    is String -> workDataBuilder.putString(key, value)
                    is Boolean -> workDataBuilder.putBoolean(key, value)
                    is Int -> workDataBuilder.putInt(key, value)
                    is Long -> workDataBuilder.putLong(key, value)
                    is Float -> workDataBuilder.putFloat(key, value)
                    is Double -> workDataBuilder.putDouble(key, value)
                    else -> workDataBuilder.putString(key, value.toString())
                }
            }
            val workDataWithConfig = workDataBuilder
                .putString("clientId", config.clientId)
                .putBoolean("debug", config.debug ?: false)
                .putString("eventTrackingHost", config.eventTrackingHost)
                .build()

            val workRequest = OneTimeWorkRequestBuilder<AdClickTrackingWorker>()
                .setInputData(workDataWithConfig)
                .setConstraints(
                    Constraints.Builder()
                        .setRequiredNetworkType(NetworkType.CONNECTED)
                        .build()
                )
                .setBackoffCriteria(
                    BackoffPolicy.EXPONENTIAL,
                    30, TimeUnit.SECONDS
                )
                .addTag(AdClickData.WORK_TAG_AD_CLICK)
                .build()

            WorkManager.getInstance(context).enqueue(workRequest)
        } catch (e: Exception) {
            errorLog(
                "Failed to queue ad click for WorkManager: ${e.message}",
                com.ai.osmos.utils.error.OsmosError.UNKNOWN,
                e
            )
            fallbackTracker?.trackAdClick(uclid, cliUbid)
        }
    }

    /**
     * Internal method that handles WorkManager-based persistent video action tracking.
     *
     * Creates a WorkManager job with network constraints and exponential backoff.
     * The job will persist across app restarts and wait for network connectivity.
     */
    private fun trackVideoActionPersistent(
        uclid: String,
        cliUbid: String,
        actionType: VideoActionType,
        videoViewSec: Float
    ) {
        try {
            val videoActionData = VideoActionData(
                uclid = uclid,
                cliUbid = cliUbid,
                actionType = actionType,
                videoViewSec = videoViewSec
            )

            val baseWorkData = videoActionData.toWorkData()
            val workDataBuilder = Data.Builder()
            baseWorkData.keyValueMap.forEach { (key, value) ->
                when (value) {
                    is String -> workDataBuilder.putString(key, value)
                    is Boolean -> workDataBuilder.putBoolean(key, value)
                    is Int -> workDataBuilder.putInt(key, value)
                    is Long -> workDataBuilder.putLong(key, value)
                    is Float -> workDataBuilder.putFloat(key, value)
                    is Double -> workDataBuilder.putDouble(key, value)
                    else -> workDataBuilder.putString(key, value.toString())
                }
            }
            val workDataWithConfig = workDataBuilder
                .putString("clientId", config.clientId)
                .putBoolean("debug", config.debug ?: false)
                .putString("eventTrackingHost", config.eventTrackingHost)
                .build()

            val workRequest = OneTimeWorkRequestBuilder<VideoActionTrackingWorker>()
                .setInputData(workDataWithConfig)
                .setConstraints(
                    Constraints.Builder()
                        .setRequiredNetworkType(NetworkType.CONNECTED)
                        .build()
                )
                .setBackoffCriteria(
                    BackoffPolicy.EXPONENTIAL,
                    30, TimeUnit.SECONDS
                )
                .addTag(VideoActionData.WORK_TAG_VIDEO_ACTION)
                .build()

            WorkManager.getInstance(context).enqueue(workRequest)
        } catch (e: Exception) {
            errorLog(
                "Failed to queue video action for WorkManager: ${e.message}",
                com.ai.osmos.utils.error.OsmosError.UNKNOWN,
                e
            )
            fallbackTracker?.videoActionClick(uclid, cliUbid, actionType, videoViewSec)
        }
    }

    /**
     * Internal method that handles WorkManager-based persistent video progress tracking.
     *
     * Creates a WorkManager job with network constraints and exponential backoff.
     * The job will persist across app restarts and wait for network connectivity.
     */
    private fun trackVideoProgressPersistent(
        uclid: String,
        cliUbid: String,
        videoViewSec: Float,
        videoDurationSec: Float
    ) {
        try {
            val videoProgressData = VideoProgressData(
                uclid = uclid,
                cliUbid = cliUbid,
                videoViewSec = videoViewSec,
                videoDurationSec = videoDurationSec
            )

            val baseWorkData = videoProgressData.toWorkData()
            val workDataBuilder = Data.Builder()
            baseWorkData.keyValueMap.forEach { (key, value) ->
                when (value) {
                    is String -> workDataBuilder.putString(key, value)
                    is Boolean -> workDataBuilder.putBoolean(key, value)
                    is Int -> workDataBuilder.putInt(key, value)
                    is Long -> workDataBuilder.putLong(key, value)
                    is Float -> workDataBuilder.putFloat(key, value)
                    is Double -> workDataBuilder.putDouble(key, value)
                    else -> workDataBuilder.putString(key, value.toString())
                }
            }
            val workDataWithConfig = workDataBuilder
                .putString("clientId", config.clientId)
                .putBoolean("debug", config.debug ?: false)
                .putString("eventTrackingHost", config.eventTrackingHost)
                .build()

            val workRequest = OneTimeWorkRequestBuilder<VideoProgressTrackingWorker>()
                .setInputData(workDataWithConfig)
                .setConstraints(
                    Constraints.Builder()
                        .setRequiredNetworkType(NetworkType.CONNECTED)
                        .build()
                )
                .setBackoffCriteria(
                    BackoffPolicy.EXPONENTIAL,
                    30, TimeUnit.SECONDS
                )
                .addTag(VideoProgressData.WORK_TAG_VIDEO_PROGRESS)
                .build()

            WorkManager.getInstance(context).enqueue(workRequest)
        } catch (e: Exception) {
            errorLog(
                "Failed to queue video progress for WorkManager: ${e.message}",
                com.ai.osmos.utils.error.OsmosError.UNKNOWN,
                e
            )
            fallbackTracker?.videoProgressEvent(uclid, cliUbid, videoViewSec, videoDurationSec)
        }
    }

    /**
     * Cancels all pending tracking work.
     *
     * This method is useful for:
     * - App shutdown cleanup
     * - Configuration changes that require work cancellation
     * - Testing scenarios
     */
    fun cancelAllPendingWork() {
        if (isWorkManagerAvailable) {
            try {
                val workManager = WorkManager.getInstance(context)
                workManager.cancelAllWorkByTag(ImpressionData.WORK_TAG_IMPRESSION)
                workManager.cancelAllWorkByTag(AdClickData.WORK_TAG_AD_CLICK)
                workManager.cancelAllWorkByTag(VideoActionData.WORK_TAG_VIDEO_ACTION)
                workManager.cancelAllWorkByTag(VideoProgressData.WORK_TAG_VIDEO_PROGRESS)
            } catch (e: Exception) {
                errorLog(
                    "Failed to cancel pending work: ${e.message}",
                    com.ai.osmos.utils.error.OsmosError.UNKNOWN,
                    e
                )
            }
        }
    }

    /**
     * Cancels all pending impression tracking work.
     *
     * This method is useful for:
     * - App shutdown cleanup
     * - Configuration changes that require work cancellation
     * - Testing scenarios
     */
    fun cancelAllPendingImpressions() {
        if (isWorkManagerAvailable) {
            try {
                WorkManager.getInstance(context)
                    .cancelAllWorkByTag(ImpressionData.WORK_TAG_IMPRESSION)
            } catch (e: Exception) {
                errorLog(
                    "Failed to cancel pending work: ${e.message}",
                    com.ai.osmos.utils.error.OsmosError.UNKNOWN,
                    e
                )
            }
        }
    }

    /**
     * Gets the count of all pending tracking jobs.
     *
     * This method provides insight into the work queue status and can be used for:
     * - Monitoring and debugging
     * - Analytics and reporting
     * - Performance optimization
     *
     * @param callback Called with the count of pending jobs (impressions, clicks, video actions, video progress)
     */
    fun getPendingTrackingCount(callback: (Int) -> Unit) {
        if (isWorkManagerAvailable) {
            try {
                val workManager = WorkManager.getInstance(context)

                val impressionInfos =
                    workManager.getWorkInfosByTag(ImpressionData.WORK_TAG_IMPRESSION).get()
                val adClickInfos =
                    workManager.getWorkInfosByTag(AdClickData.WORK_TAG_AD_CLICK).get()
                val videoActionInfos =
                    workManager.getWorkInfosByTag(VideoActionData.WORK_TAG_VIDEO_ACTION).get()
                val videoProgressInfos =
                    workManager.getWorkInfosByTag(VideoProgressData.WORK_TAG_VIDEO_PROGRESS).get()

                val pendingCount =
                    (impressionInfos + adClickInfos + videoActionInfos + videoProgressInfos).count {
                        it.state == WorkInfo.State.ENQUEUED || it.state == WorkInfo.State.RUNNING
                    }
                callback(pendingCount)
            } catch (e: Exception) {
                errorLog(
                    "Failed to get pending work count: ${e.message}",
                    com.ai.osmos.utils.error.OsmosError.UNKNOWN,
                    e
                )
                callback(0)
            }
        } else {
            callback(0)
        }
    }

    /**
     * Gets the count of pending impression tracking jobs.
     *
     * This method provides insight into the work queue status and can be used for:
     * - Monitoring and debugging
     * - Analytics and reporting
     * - Performance optimization
     *
     * @param callback Called with the count of pending jobs
     */
    fun getPendingImpressionCount(callback: (Int) -> Unit) {
        if (isWorkManagerAvailable) {
            try {
                val workInfos = WorkManager.getInstance(context)
                    .getWorkInfosByTag(ImpressionData.WORK_TAG_IMPRESSION)
                    .get()

                val pendingCount = workInfos.count {
                    it.state == WorkInfo.State.ENQUEUED || it.state == WorkInfo.State.RUNNING
                }
                callback(pendingCount)
            } catch (e: Exception) {
                errorLog(
                    "Failed to get pending work count: ${e.message}",
                    com.ai.osmos.utils.error.OsmosError.UNKNOWN,
                    e
                )
                callback(0)
            }
        } else {
            callback(0)
        }
    }

    /**
     * Checks if WorkManager classes are available at runtime.
     *
     * This method uses reflection to detect WorkManager availability without
     * causing ClassNotFoundException if the dependency is not included.
     *
     * @return true if WorkManager is available, false otherwise
     */
    private fun isWorkManagerClassAvailable(): Boolean {
        return try {
            Class.forName("androidx.work.WorkManager")
            Class.forName("androidx.work.Data")
            Class.forName("androidx.work.CoroutineWorker")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    companion object {
        /**
         * Factory method to create PersistentAdTracker with fallback support.
         *
         * This is the recommended way to create a PersistentAdTracker instance
         * as it automatically sets up the fallback mechanism.
         *
         * @param context Application context for WorkManager
         * @param coroutineScope CoroutineScope for fallback tracker
         * @return PersistentAdTracker instance with fallback support
         */
        fun create(
            context: Context,
            coroutineScope: CoroutineScope,
            config: Config
        ): AdTrackerInterface {
            val fallbackTracker = AdTracker(coroutineScope, config)
            return PersistentAdTracker(context, fallbackTracker, config)
        }
    }
}