package com.ai.osmos.tracking.tracker.workers

import android.content.Context
import androidx.work.CoroutineWorker
import androidx.work.WorkerParameters
import androidx.work.workDataOf
import com.ai.osmos.models.events.VideoActionType
import com.ai.osmos.tracking.data.VideoActionData
import com.ai.osmos.tracking.events.RegisterEvent
import com.ai.osmos.tracking.tracker.BaseAdTracker
import com.ai.osmos.utils.logging.DebugLogger
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import kotlin.math.min
import kotlin.math.pow

/**
 * WorkManager worker for persistent video action tracking.
 *
 * This worker handles video action tracking requests (mute/unmute, fullscreen, etc.)
 * that need to survive app restarts and network failures. It implements exponential
 * backoff retry logic and ensures video actions are properly tracked for analytics.
 */
class VideoActionTrackingWorker(
    context: Context,
    workerParams: WorkerParameters
) : CoroutineWorker(context, workerParams) {

    override suspend fun doWork(): Result = withContext(Dispatchers.IO) {
        try {
            // Deserialize video action data from input
            val videoActionData = VideoActionData.fromWorkData(inputData)
                ?: return@withContext Result.failure(
                    workDataOf("error" to "Invalid video action data")
                )
            DebugLogger.log("Processing video action: ${videoActionData.actionType}, attempt: ${videoActionData.attemptCount}")

            // Create RegisterEvent instance for API calls with instance-specific config
            val config = BaseAdTracker.getConfigFromWorkData(inputData)
            val registerEvent = RegisterEvent(config)

            try {
                // Attempt to track the video action
                val response = registerEvent.registerVideoActionEvent(
                    cliUbid = videoActionData.cliUbid,
                    uclid = videoActionData.uclid,
                    videoViewSec = videoActionData.videoViewSec,
                    actionType = videoActionData.actionType,
                )

                if (response != null) {
                    DebugLogger.log("Video action tracked successfully: ${videoActionData.actionType}")
                    Result.success(
                        workDataOf(
                            "uclid" to videoActionData.uclid,
                            "action_type" to videoActionData.actionType,
                            "result" to "success",
                            "attempt_count" to videoActionData.attemptCount
                        )
                    )
                } else {
                    handleRetry(videoActionData)
                }

            } catch (e: Exception) {
                DebugLogger.log("Video action tracking failed: ${e.message}")

                // Handle retry with proper attempt count management
                handleRetry(videoActionData)
            }

        } catch (e: Exception) {
            Result.failure(
                workDataOf(
                    "error" to "Worker execution failed: ${e.message}"
                )
            )
        }
    }

    /**
     * Handles retry logic with exponential backoff.
     *
     * @param videoActionData The video action data to retry
     * @return Result indicating whether to retry or give up
     */
    private fun handleRetry(videoActionData: VideoActionData): Result {
        val updatedData = videoActionData.incrementAttempt()

        return if (updatedData.shouldRetry()) {

            DebugLogger.log("Scheduling retry ${updatedData.attemptCount}/${updatedData.maxRetries} for video action")

            // Schedule retry with exponential backoff
            val retryDelay = calculateBackoffDelay(updatedData.attemptCount)
            scheduleRetryWork(updatedData.toWorkData(), retryDelay)

            // Return failure to indicate this attempt failed but retry is scheduled
            Result.failure()
        } else {
            DebugLogger.log("Max retries exceeded for video action. Giving up.")

            Result.failure(
                workDataOf(
                    "error" to "Max retries exceeded for video action tracking",
                    "uclid" to videoActionData.uclid,
                    "action_type" to videoActionData.actionType,
                    "attempt_count" to videoActionData.attemptCount
                )
            )
        }
    }

    /**
     * Calculates exponential backoff delay for retries.
     *
     * Uses 2^attempt formula with a maximum cap to prevent excessive delays.
     * This helps distribute retry load and respects server capacity.
     *
     * @param attemptCount Current attempt number (1-based)
     * @return Delay in seconds before next retry
     */
    private fun calculateBackoffDelay(attemptCount: Int): Long {
        // Exponential backoff: 2^attempt seconds, capped at 5 minutes (300s)

        val delaySeconds = min(
            2.0.pow(attemptCount.toDouble()).toLong(),
            300L
        )
        DebugLogger.log("Calculated backoff delay: ${delaySeconds}s for attempt $attemptCount")

        return delaySeconds
    }

    /**
     * Schedules a retry work request with appropriate constraints and backoff.
     *
     * @param data Serialized video action data for the retry
     * @param delaySeconds Delay before executing the retry
     */
    private fun scheduleRetryWork(data: androidx.work.Data, delaySeconds: Long) {
        try {
            val retryRequest = androidx.work.OneTimeWorkRequestBuilder<VideoActionTrackingWorker>()
                .setInputData(data)
                .setInitialDelay(delaySeconds, java.util.concurrent.TimeUnit.SECONDS)
                .setConstraints(
                    androidx.work.Constraints.Builder()
                        .setRequiredNetworkType(androidx.work.NetworkType.CONNECTED)
                        .build()
                )
                .addTag(VideoActionData.WORK_TAG_VIDEO_ACTION)
                .build()

            androidx.work.WorkManager.getInstance(applicationContext).enqueue(retryRequest)
            DebugLogger.log("Retry work scheduled with ${delaySeconds}s delay")

        } catch (e: Exception) {
            DebugLogger.log("Failed to schedule retry work: ${e.message}")
        }
    }
}