package com.ai.osmos.utils.common

import com.ai.osmos.models.ads.AdElement
import com.ai.osmos.models.ads.AdType
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.CarouselAd
import com.ai.osmos.models.ads.CarouselAdElement
import com.ai.osmos.models.ads.CarouselAdElementsWrapper
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.LandingProductList
import com.ai.osmos.models.ads.MediaElement
import com.ai.osmos.models.ads.MultiAd
import com.ai.osmos.models.ads.VideoAd
import org.json.JSONObject


internal class JSONParsing {
    companion object {
        fun parseAdData(response: String, cliUbid: String): Map<String, List<BaseAd>> {
            val result = mutableMapOf<String, List<BaseAd>>()
            try {
                val dataObject = JSONObject(response)

                val adsObject = dataObject.optJSONObject("ads") ?: return result

                val keys = adsObject.keys()
                while (keys.hasNext()) {
                    val key = keys.next()
                    val adArray = adsObject.getJSONArray(key)
                    val adList = mutableListOf<BaseAd>()

                    for (i in 0 until adArray.length()) {
                        val adObj = adArray.getJSONObject(i)
                        val elements = adObj.optJSONObject("elements") ?: continue

                        var adType = AdType.fromString(elements.optString("type", "IMAGE"))

                        if (elements.has("carousel_cards")) {
                            adType = AdType.fromString("CAROUSEL")
                        }
                        if (adArray.length() > 1) {
                            adType = AdType.fromString("MULTI_AD")
                        }

                        val baseAd: BaseAd? = when (adType) {
                            AdType.IMAGE -> parseImageAd(adObj, cliUbid)
                            AdType.VIDEO -> parseVideoAd(adObj, cliUbid)
                            AdType.CAROUSEL -> parseCarouselAd(adObj, cliUbid)
                            AdType.MULTI_AD -> parseMultiAds(adObj, cliUbid)
                            AdType.UNKNOWN -> null
                        }
                        baseAd?.let { adList.add(it) }
                    }
                    result[key] = adList
                }
                return result
            } catch (e: Exception) {
                return result
            }
        }

        private fun JSONObject.safeString(key: String, default: String = ""): String =
            if (has(key)) optString(key, default) else default

        private fun JSONObject.safeInt(key: String, default: Int = 0): Int =
            if (has(key)) optInt(key, default) else default

        private fun parseImageAd(adObject: JSONObject, cliUbid: String): ImageAd? {
            val elementsObject = adObject.optJSONObject("elements") ?: return null

            val landingProducts = mutableListOf<LandingProductList>()
//            if (elementsObject.has("landing_product_list")) {
//                val landingArray = elementsObject.getJSONArray("landing_product_list")
//                for (j in 0 until landingArray.length()) {
//                    val productObj = landingArray.getJSONObject(j)
//                    landingProducts.add(
//                        LandingProductList(
//                            sku_id = productObj.getString("sku_id"),
//                            e_currency = productObj.getString("e_currency"),
//                            e_image_link = productObj.getString("e_image_link")
//                        )
//                    )
//                }
//            }

            val adElement = AdElement(
                value = elementsObject.safeString("value"),
                height = elementsObject.safeInt("height"),
                width = elementsObject.safeInt("width"),
                type = elementsObject.safeString("type"),
                destinationUrl = elementsObject.optString("destination_url", null),
                landingProductList = landingProducts,
                closeButtonSec = if (elementsObject.has("close_button_sec")) {
                    elementsObject.getInt("close_button_sec")
                } else null
            )

            return ImageAd(
                clientId = adObject.optLong("client_id"),
                au = adObject.optString("au"),
                rank = adObject.optInt("rank"),
                clickTrackingUrl = adObject.optString("click_tracking_url"),
                impressionTrackingUrl = adObject.optString("impression_tracking_url"),
                uclid = adObject.optString("uclid"),
                crt = adObject.optString("crt"),
                elements = adElement,
                cliUbid = cliUbid
            )
        }

        private fun parseVideoAd(adObject: JSONObject, cliUbid: String): VideoAd? {
            val elementsObject = adObject.optJSONObject("elements") ?: return null
            val adElement = AdElement(
                value = elementsObject.safeString("value"),
                height = elementsObject.safeInt("height"),
                width = elementsObject.safeInt("width"),
                type = elementsObject.safeString("type"),
                destinationUrl = elementsObject.optString("destination_url", null),
                videoProgressSec = if (elementsObject.has("video_progress_sec")) {
                    elementsObject.getInt("video_progress_sec")
                } else null,
                closeButtonSec = if (elementsObject.has("close_button_sec")) {
                    elementsObject.getInt("close_button_sec")
                } else null
            )

            return VideoAd(
                clientId = adObject.optLong("client_id"),
                au = adObject.optString("au"),
                rank = adObject.optInt("rank"),
                clickTrackingUrl = adObject.optString("click_tracking_url"),
                impressionTrackingUrl = adObject.optString("impression_tracking_url"),
                uclid = adObject.optString("uclid"),
                crt = adObject.optString("crt"),
                elements = adElement,
                cliUbid = cliUbid
            )
        }

        private fun parseCarouselAd(adObject: JSONObject, cliUbid: String): CarouselAd? {

            val elementsObject = adObject.optJSONObject("elements") ?: return null
            val carouselArray = elementsObject.optJSONArray("carousel_cards") ?: return null

            val carouselCards = mutableListOf<CarouselAdElement>()
            for (i in 0 until carouselArray.length()) {
                val cardObj = carouselArray.optJSONObject(i) ?: continue
                val mediaObj = cardObj.optJSONObject("media") ?: continue

                val media = MediaElement(
                    value = mediaObj.optString("value"),
                    height = mediaObj.optInt("height"),
                    width = mediaObj.optInt("width"),
                    type = mediaObj.optString("type")
                )

                val element = CarouselAdElement(
                    media = media,
                    destinationUrl = cardObj.optString("destination_url")
                )

                carouselCards.add(element)
            }

            return CarouselAd(
                clientId = adObject.optLong("client_id"),
                au = adObject.optString("au"),
                rank = adObject.optInt("rank"),
                clickTrackingUrl = adObject.optString("click_tracking_url"),
                impressionTrackingUrl = adObject.optString("impression_tracking_url"),
                uclid = adObject.optString("uclid"),
                crt = adObject.optString("crt"),
                elements = CarouselAdElementsWrapper(carouselCards = carouselCards), // wrapped
                cliUbid = cliUbid
            )
        }

        private fun parseMultiAds(adObject: JSONObject, cliUbid: String): MultiAd? {

            val elementsObject = adObject.optJSONObject("elements") ?: return null

            val adElement = AdElement(
                value = elementsObject.safeString("value"),
                height = elementsObject.safeInt("height"),
                width = elementsObject.safeInt("width"),
                type = elementsObject.safeString("type"),
                destinationUrl = elementsObject.optString("destination_url", null),
                videoProgressSec = if (elementsObject.has("video_progress_sec")) {
                    elementsObject.getInt("video_progress_sec")
                } else null,
                closeButtonSec = if (elementsObject.has("close_button_sec")) {
                    elementsObject.getInt("close_button_sec")
                } else null
            )

            val multiAd = MultiAd(
                clientId = adObject.optLong("client_id"),
                au = adObject.optString("au"),
                rank = adObject.optInt("rank"),
                clickTrackingUrl = adObject.optString("click_tracking_url"),
                impressionTrackingUrl = adObject.optString("impression_tracking_url"),
                uclid = adObject.optString("uclid"),
                crt = adObject.optString("crt"),
                elements = adElement,
                cliUbid = cliUbid
            )
            return multiAd
        }
    }
}