package com.ai.osmos.ads.fetcher

import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.ProductAdsFilter
import com.ai.osmos.models.ads.TargetingParams
import com.ai.osmos.models.ads.TargetingParamsBuilder
import com.ai.osmos.models.enums.PlaPageType
import com.ai.osmos.network.QueryParameterBuilder
import com.ai.osmos.network.WebConstants
import com.ai.osmos.network.WebService
import com.ai.osmos.utils.common.ApiEndpoint
import com.ai.osmos.utils.common.HttpMethod
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.StandardErrorHandler
import com.ai.osmos.utils.logging.DebugLogger

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: AdFetcherSDK
 *
 * This class is responsible for making API calls to fetch different types of ads using
 * defined function signature for display ads with au and display ads with pt.
 */
internal class AdFetcherSDK(private val config: Config) : AdFetcherSDKInterface {

    private val webService = WebService()

    /**
     * Fetches display ads using Ad Unit-based targeting with type-safe targeting parameters.
     *
     * @param cliUbid Unique client identifier
     * @param pageType Type of page requesting ads
     * @param productCount Number of products to display ads for
     * @param adUnits List of ad unit identifiers
     * @param targetingParams Optional: Array of targeting parameter objects (PlatformTargeting, MerchandiseTargeting, GeoTargeting, ContextTargeting, CustomLabelTargeting)
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any> on success, emptyMap() on error.
     *
     * Example:
     * ```
     * val contextTargeting = ContextTargeting().keyword("potato chips")
     * val geoTargeting = GeoTargeting()
     *     .storeIds(listOf("store_123", "store_456"))
     *     .networks(listOf("wifi", "5G"))
     * val merchandiseTargeting = MerchandiseTargeting()
     *     .brands(listOf("Apple", "Samsung"))
     *     .udps(listOf("premium", "featured"))
     * val response = adFetcherSDK.fetchDisplayAdsWithAu(
     *     cliUbid = "1234",
     *     pageType = "demo_page",
     *     productCount = 10,
     *     adUnits = listOf("image_ad_inv"),
     *     targetingParams = listOf(contextTargeting, geoTargeting, merchandiseTargeting)
     * )
     * ```
     */
    override suspend fun fetchDisplayAdsWithAu(
        cliUbid: String,
        pageType: String,
        productCount: Int,
        adUnits: List<String>,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {

        DebugLogger.log("cli_ubid = $cliUbid")
        DebugLogger.log("page_type = $pageType")
        DebugLogger.log("product_count = $productCount")
        DebugLogger.log("ad_units = $adUnits")
        DebugLogger.log("targeting_params = ${targetingParams?.toString()}")

        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            // Build targeting parameters
            val targetingDict = targetingParams?.let { params ->
                TargetingParamsBuilder.from(params).toMap()
            } ?: emptyMap()

            val queryParams = QueryParameterBuilder.buildDisplayAdParams(
                clientId = config.clientId,
                isDisplayAuAds = true,
                cliUbid = cliUbid,
                pageType = pageType,
                productCount = productCount,
                adUnits = adUnits,
                targetingParams = targetingDict
            )

            val baseUrl = config.displayAdsHost
            val apiUrl =
                WebConstants.buildUrl("$baseUrl${ApiEndpoint.DISPLAY_ADS.endpoint}", queryParams)

            webService.callApiWithException(
                apiUrl,
                HttpMethod.GET,
                requestHeaders = null,
                requestBody = null
            )
        }
    }

    /**
     * Fetches display ads using Page Type-based targeting with type-safe targeting parameters.
     *
     * @param cliUbid Unique client identifier
     * @param pageType Type of page requesting ads
     * @param productCount Number of products to display ads for
     * @param targetingParams Optional: Array of targeting parameter objects (PlatformTargeting, MerchandiseTargeting, GeoTargeting, ContextTargeting, CustomLabelTargeting)
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>
     *
     * Example:
     * ```
     * val contextTargeting = ContextTargeting().keyword("potato chips")
     * val geoTargeting = GeoTargeting()
     *     .storeIds(listOf("store_123", "store_456"))
     *     .city("New York")
     * val response = adFetcherSDK.fetchDisplayAdsWithPt(
     *     cliUbid = "1234",
     *     pageType = "demo_page",
     *     productCount = 10,
     *     targetingParams = listOf(contextTargeting, geoTargeting)
     * )
     * ```
     */
    override suspend fun fetchDisplayAdsWithPt(
        cliUbid: String,
        pageType: String,
        productCount: Int,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {

        DebugLogger.log("cli_ubid = $cliUbid")
        DebugLogger.log("page_type = $pageType")
        DebugLogger.log("product_count = $productCount")
        DebugLogger.log("targeting_params = ${targetingParams?.toString()}")

        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {

            // Build targeting parameters
            val targetingDict = targetingParams?.let { params ->
                TargetingParamsBuilder.from(params).toMap()
            } ?: emptyMap()

            val queryParams = QueryParameterBuilder.buildDisplayAdParams(
                clientId = config.clientId,
                isDisplayAuAds = false,
                cliUbid = cliUbid,
                pageType = pageType,
                productCount = productCount,
                targetingParams = targetingDict
            )

            val baseUrl = config.displayAdsHost
            val apiUrl =
                WebConstants.buildUrl("$baseUrl${ApiEndpoint.DISPLAY_ADS_PT.endpoint}", queryParams)

            webService.callApiWithException(
                apiUrl,
                HttpMethod.GET,
                requestHeaders = null,
                requestBody = null
            )
        }
    }

    /**
     * Fetches Product Listing Ads (PLA) with type-safe parameters.
     * Generic function for PLA Ads with enhanced type safety.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional name of the page requesting ads. Defaults to null.
     * @param pageType Type of the page using PlaPageType enum.
     * @param productCount Number of products to display ads for.
     * @param filters Optional: ProductAdsFilter object with filtering parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val filter = ProductAdsFilter()
     *     .skuIds(listOf("sku_1", "sku_2", "sku_3"))
     *     .brands(listOf("Apple", "Samsung"))
     * val response = adFetcherSDK.fetchPlaAds(
     *     cliUbid = "1234",
     *     pageType = PlaPageType.PRODUCT,
     *     productCount = 10,
     *     pageName = "PRODUCT_RECOMMENDATIONS",
     *     filters = filter
     * )
     * ```
     */
    override suspend fun fetchPlaAds(
        cliUbid: String,
        pageType: PlaPageType,
        productCount: Int,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {

        DebugLogger.log("cli_ubid = $cliUbid")
        DebugLogger.log("page_name = $pageName")
        DebugLogger.log("page_type = ${pageType.pageTypeName}")
        DebugLogger.log("product_count = $productCount")
        DebugLogger.log("filters = ${filters?.toString()}")

        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {

            // Build filter parameters
            val filterDict = filters?.toMap() ?: emptyMap()

            val queryParams = QueryParameterBuilder.buildPlaAdParams(
                clientId = config.clientId,
                cliUbid = cliUbid,
                pageType = pageType.pageTypeName,
                productCount = productCount,
                pageName = pageName,
                filters = filterDict
            )

            val baseUrl = config.productAdsHost
            val apiUrl =
                WebConstants.buildUrl("$baseUrl${ApiEndpoint.PLA_ADS.endpoint}", queryParams)

            webService.callApiWithException(
                apiUrl,
                HttpMethod.GET,
                requestHeaders = null,
                requestBody = null
            )
        }
    }

    /**
     * Fetches PLA ads specifically for search pages with type-safe parameters.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the search results page.
     * @param productCount Number of products to display ads for.
     * @param keywords List of search keywords.
     * @param filters Optional: ProductAdsFilter object with filtering parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val filter = ProductAdsFilter()
     *     .storeIds(listOf("store_1"))
     *     .brands(listOf("Apple"))
     * val response = adFetcherSDK.fetchPlaSearchPageAds(
     *     cliUbid = "1234",
     *     productCount = 10,
     *     keyword = "potato chips",
     *     pageName = "TOP_OF_SEARCH",
     *     filters = filter
     * )
     * ```
     */
    override suspend fun fetchPlaSearchPageAds(
        cliUbid: String,
        productCount: Int,
        keyword: String,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {

        // Create or use existing filter and add keyword to it
        val pFilter = ProductAdsFilter.keywords(listOf(keyword))
        val finalFilters = filters?.mergeFilters(pFilter) ?: pFilter

        return fetchPlaAds(
            cliUbid = cliUbid,
            pageName = pageName,
            pageType = PlaPageType.SEARCH,
            productCount = productCount,
            filters = finalFilters,
            errorCallback = errorCallback
        )
    }

    /**
     * Fetches PLA ads for home page with type-safe parameters.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the home page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param filters Optional: ProductAdsFilter object with filtering parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val filter = ProductAdsFilter()
     *     .storeIds(listOf("store_1"))
     *     .brands(listOf("Whitakers"))
     * val response = adFetcherSDK.fetchPlaHomePageAds(
     *     cliUbid = "1234",
     *     productCount = 10,
     *     pageName = "HOME_PAGE",
     *     filters = filter
     * )
     * ```
     */
    override suspend fun fetchPlaHomePageAds(
        cliUbid: String,
        productCount: Int,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return fetchPlaAds(
            cliUbid = cliUbid,
            pageName = pageName,
            pageType = PlaPageType.HOME,
            productCount = productCount,
            filters = filters,
            errorCallback = errorCallback
        )
    }

    /**
     * Fetches PLA ads for a category page with type-safe parameters.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the category page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param categories List of categories to filter ads.
     * @param filters Optional: ProductAdsFilter object with filtering parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val filter = ProductAdsFilter()
     *     .storeIds(listOf("store_1"))
     * val response = adFetcherSDK.fetchPlaCategoryPageAds(
     *     cliUbid = "1234",
     *     pageName = "category_listing",
     *     productCount = 3,
     *     categories = listOf("Kitchen & Pantry"),
     *     filters = filter
     * )
     * ```
     */
    override suspend fun fetchPlaCategoryPageAds(
        cliUbid: String,
        productCount: Int,
        categories: List<String>,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {

        // Create or use existing filter and add categories to it
        val pFilter = ProductAdsFilter.categories(categories)
        val finalFilters = filters?.mergeFilters(pFilter) ?: pFilter

        return fetchPlaAds(
            cliUbid = cliUbid,
            pageName = pageName,
            pageType = PlaPageType.CATEGORY,
            productCount = productCount,
            filters = finalFilters,
            errorCallback = errorCallback
        )
    }

    /**
     * Fetches PLA ads for a product detail page with type-safe parameters.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the product page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param skuIds List of SKU identifiers for the products.
     * @param filters Optional: ProductAdsFilter object with filtering parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val filter = ProductAdsFilter()
     *     .storeIds(listOf("store_1"))
     * val response = adFetcherSDK.fetchPlaProductPageAds(
     *     cliUbid = "1234",
     *     productCount = 20,
     *     skuIds = listOf("sku_123"),
     *     filters = filter
     * )
     * ```
     */
    override suspend fun fetchPlaProductPageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<String>,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {

        // Create or use existing filter and add skuIds to it
        val pFilter = ProductAdsFilter.skuIds(skuIds)
        val finalFilters = filters?.mergeFilters(pFilter) ?: pFilter

        return fetchPlaAds(
            cliUbid = cliUbid,
            pageName = pageName,
            pageType = PlaPageType.PRODUCT,
            productCount = productCount,
            filters = finalFilters,
            errorCallback = errorCallback
        )
    }


    /**
     * Fetches PLA ads for a purchase detail page with type-safe parameters.
     *
     * @param cliUbid Unique identifier for the client.
     * @param pageName Optional: Name of the product page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param skuIds List of SKU identifiers for the products.
     * @param filters Optional: ProductAdsFilter object with filtering parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val filter = ProductAdsFilter()
     *     .storeIds(listOf("store_1"))
     * val response = adFetcherSDK.fetchPlaPurchasePageAds(
     *     cliUbid = "1234",
     *     productCount = 20,
     *     skuIds = listOf("sku_123", "sku_456"),
     *     filters = filter
     * )
     * ```
     */
    override suspend fun fetchPlaPurchasePageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<String>,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {

        val pFilter = ProductAdsFilter.skuIds(skuIds)
        val finalFilters = filters?.mergeFilters(pFilter) ?: pFilter

        return fetchPlaAds(
            cliUbid = cliUbid,
            pageName = pageName,
            pageType = PlaPageType.PURCHASE,
            productCount = productCount,
            filters = finalFilters,
            errorCallback = errorCallback
        )
    }

    /**
     * POST API Call for Tagged Product Ads.
     *
     * This function constructs a JSON payload with the provided parameters and sends
     * a POST request to the configured Tagged product ads. The response is logged for
     * debugging purposes.
     *
     * @param cliUbid Unique identifier for the client session.
     * @param pageName Optional: Name of the page requesting ads.
     * @param productCount Number of products to display ads for.
     * @param skuIds List of SKU identifiers related to the products.
     * @param filters Optional: ProductAdsFilter object with filtering parameters
     * @param errorCallback Optional: Callback to handle error scenarios
     * @return The API response as a Map<String,Any>.
     *
     * Example:
     * ```
     * val filter = ProductAdsFilter().storeIds(listOf("Astore"))
     * val res  = adFetcherSDK.fetchTpaPageAds(
     *             cliUbid = "9bd111c7-6a57-4a90-8ad0-51a5eb98cfc1",
     *             pageName = "",
     *             productCount = 5,
     *             filters = filter,
     *             skuIds = arrayListOf("2777534___Astore___Anet", "2777515___Astore___Anet")
     *         )
     * ```
     */
    override suspend fun fetchTpaPageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<Any>,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {

        DebugLogger.log("cli_ubid = $cliUbid")
        DebugLogger.log("page_name = $pageName")
        DebugLogger.log("product_count = $productCount")
        DebugLogger.log("sku_ids = $skuIds")
        DebugLogger.log("filters = ${filters?.toString()}")

        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {

            val data = QueryParameterBuilder.buildTpaAdData(
                clientId = config.clientId,
                cliUbid = cliUbid,
                productCount = productCount,
                skuIds = skuIds,
                pageName = pageName,
                filters = filters?.toMap() ?: emptyMap()
            )
            val baseUrl = config.productAdsHost
            webService.callApiWithException(
                "$baseUrl${ApiEndpoint.TPA_ADS.endpoint}",
                HttpMethod.POST,
                requestHeaders = null,
                data
            )
        }
    }
}