package com.ai.osmos.ads.fetcher

import com.ai.osmos.models.ads.ProductAdsFilter
import com.ai.osmos.models.ads.TargetingParams
import com.ai.osmos.models.enums.PlaPageType
import com.ai.osmos.utils.error.ErrorCallback

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: AdFetcherSDKInterface
 *
 * Interface for ad fetching functionality in the OSMOS SDK.
 * Provides methods for fetching different types of ads.
 */
public interface AdFetcherSDKInterface {

    /**
     * Fetches display ads using Ad Unit-based targeting.
     */
    suspend fun fetchDisplayAdsWithAu(
        cliUbid: String,
        pageType: String,
        productCount: Int,
        adUnits: List<String>,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetches display ads using Page Type-based targeting.
     */
    suspend fun fetchDisplayAdsWithPt(
        cliUbid: String,
        pageType: String,
        productCount: Int,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetches Product Listing Ads (PLA) based on the provided parameters.
     */
    suspend fun fetchPlaAds(
        cliUbid: String,
        pageType: PlaPageType,
        productCount: Int,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetches PLA ads specifically for search pages.
     */
    suspend fun fetchPlaSearchPageAds(
        cliUbid: String,
        productCount: Int,
        keyword: String,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetches Product Listing Ads (PLA) for the home page.
     */
    suspend fun fetchPlaHomePageAds(
        cliUbid: String,
        productCount: Int,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetches PLA ads for a category page.
     */
    suspend fun fetchPlaCategoryPageAds(
        cliUbid: String,
        productCount: Int,
        categories: List<String>,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetches PLA ads for a product detail page.
     */
    suspend fun fetchPlaProductPageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<String>,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetches PLA ads for a purchase detail page.
     */
    suspend fun fetchPlaPurchasePageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<String>,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * POST API Call for Tagged Product Ads.
     */
    suspend fun fetchTpaPageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<Any>,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?
}
