package com.ai.osmos.ads.renderer

import com.ai.osmos.ads.fetcher.AdFetcherSDK
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.ProductAdsFilter
import com.ai.osmos.models.ads.TargetingParams
import com.ai.osmos.models.enums.DisplayAdFormat
import com.ai.osmos.models.enums.NativeAdFormat
import com.ai.osmos.models.enums.PlaPageType
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.common.JSONParsing
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.StandardErrorHandler

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: AdRenderer
 */

/**
 * The [AdRenderer] class is responsible for fetching and preparing ad data for rendering.
 * It acts as a bridge between the ad fetcher module and UI-level ad views.
 */
internal class AdRenderer(private val config: Config) : AdRendererInterface {
    private val adFetcherSDK = AdFetcherSDK(config)

    /**
     * Common method to fetch display ads with ad units.
     * Reduces code duplication across banner, carousel, and multi-ad carousel methods.
     */
    private suspend fun fetchDisplayAdsWithAuCommon(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        productCount: Int = 1,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null,
        adFormat: DisplayAdFormat
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adUnits = arrayListOf(adUnit)
            val adData = adFetcherSDK.fetchDisplayAdsWithAu(
                cliUbid = cliUbid,
                pageType = pageType,
                productCount = productCount,
                adUnits = adUnits,
                targetingParams = targetingParams,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            val responseMap = adData[Constants.JSON_KEY_RESPONSE] as? Map<*, *>
            val responseDataString = responseMap?.get(Constants.JSON_KEY_DATA) as? String

            if (responseDataString.isNullOrBlank()) return@executeWithErrorHandling null

            parseDisplayAdResponse(responseDataString.toString(), cliUbid, adFormat)
        }
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams(optional): Filter params that you want to apply
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     * Example:
     * ```
     * val adData = adRender.fetchBannerAdsWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnit = "banner_ads",
     * ```
     */
    override suspend fun fetchBannerAdsWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return fetchDisplayAdsWithAuCommon(
            cliUbid = cliUbid,
            pageType = pageType,
            adUnit = adUnit,
            productCount = 1,
            targetingParams = targetingParams,
            errorCallback = errorCallback,
            adFormat = DisplayAdFormat.Banner
        )
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams(optional): Filter params that you want to apply
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *Example:
     * ```
     * val adData = adRender.fetchCarouselAdWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnit = "carousel_ads",
     * ```
     */
    override suspend fun fetchCarouselAdWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return fetchDisplayAdsWithAuCommon(
            cliUbid = cliUbid,
            pageType = pageType,
            adUnit = adUnit,
            productCount = 1,
            targetingParams = targetingParams,
            errorCallback = errorCallback,
            adFormat = DisplayAdFormat.Carousel
        )
    }

    /**
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit:  Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param productCount: No of ads responded. MAX 10.
     * @param targetingParams(optional): Filter params that you want to apply
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     * Example:
     * ```
     * val adData = adRender.fetchMultiAdCarouselWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnit = "all_in_one",
     *                 productCount = 3,
     * ```
     */
    override suspend fun fetchMultiAdCarouselWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        productCount: Int,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return fetchDisplayAdsWithAuCommon(
            cliUbid = cliUbid,
            pageType = pageType,
            adUnit = adUnit,
            productCount = productCount,
            targetingParams = targetingParams,
            errorCallback = errorCallback,
            adFormat = DisplayAdFormat.MultiAdCarousel
        )
    }

    override fun parseDisplayAdResponse(
        adData: String,
        cliUbid: String,
        adFormat: DisplayAdFormat
    ): Map<String, Any> {

        if (adData.isBlank()) return emptyMap()

        return StandardErrorHandler.executeWithParsingErrorHandling {
            val adMap = JSONParsing.parseAdData(adData, cliUbid)
            when (adFormat) {
                DisplayAdFormat.Banner,
                DisplayAdFormat.Carousel,
                DisplayAdFormat.PDA -> {
                    val firstAd = adMap.values.firstOrNull()?.firstOrNull()
                    if (firstAd != null) {
                        mapOf("ad" to firstAd)
                    } else {
                        emptyMap()
                    }
                }

                DisplayAdFormat.MultiAdCarousel -> {
                    val allAds = adMap.values.flatten()
                    if (allAds.isNotEmpty()) {
                        mapOf("ad" to allAds)
                    } else {
                        emptyMap()
                    }
                }
            }
        }
    }

    /**
     * Fetch Product Listing Ads (PLA) for category pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param categories: List of category names to filter ads
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaCategoryPageAds(
     *     cliUbid = "123",
     *     productCount = 5,
     *     categories = listOf("Electronics", "Computers"),
     *     pageName = "category_listing",
     *     filters = ProductAdsFilter.create(brands = listOf("Apple")),
     *     errorCallback = null
     * )
     * ```
     */
    override suspend fun fetchNativePlaCategoryPageAds(
        cliUbid: String,
        productCount: Int,
        categories: List<String>,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adData = adFetcherSDK.fetchPlaCategoryPageAds(
                cliUbid = cliUbid,
                productCount = productCount,
                categories = categories,
                pageName = pageName,
                filters = filters,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            val responseMap = adData[Constants.JSON_KEY_RESPONSE] as? Map<*, *>
            val responseDataString = responseMap?.get(Constants.JSON_KEY_DATA) as? String

            if (responseDataString.isNullOrBlank()) return@executeWithErrorHandling emptyMap()

            if (productCount > 1) {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeCarouselAd)
            } else {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeAd)
            }
        }
    }

    /**
     * Fetch Tagged Product Ads (TPA) for specific products.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param skuIds: List of SKU identifiers for the products
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchTpaPageAds(
     *     cliUbid = "123",
     *     productCount = 5,
     *     skuIds = listOf("sku_1", "sku_2"),
     *     pageName = "product_listing",
     *     filters = ProductAdsFilter.create(storeIds = listOf("store_1")),
     *     errorCallback = null
     * )
     * ```
     */
    override suspend fun fetchNativePlaTpaPageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<Any>,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adData = adFetcherSDK.fetchTpaPageAds(
                cliUbid = cliUbid,
                productCount = productCount,
                skuIds = skuIds,
                pageName = pageName,
                filters = filters,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            val responseMap = adData[Constants.JSON_KEY_RESPONSE] as? Map<*, *>
            val responseDataString = responseMap?.get(Constants.JSON_KEY_DATA) as? String

            if (responseDataString.isNullOrBlank()) return@executeWithErrorHandling emptyMap()

            if (productCount > 1) {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeCarouselAd)
            } else {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeAd)
            }
        }
    }

    /**
     * Fetch Native Product Listing Ads (PLA) for home pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaHomePageAds(
     *     cliUbid = "123",
     *     productCount = 10,
     *     pageName = "home_page",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("store_1"),
     *         brands = listOf("Brand1")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    override suspend fun fetchNativePlaHomePageAds(
        cliUbid: String,
        productCount: Int,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adData = adFetcherSDK.fetchPlaHomePageAds(
                cliUbid = cliUbid,
                productCount = productCount,
                pageName = pageName,
                filters = filters,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            val responseMap = adData[Constants.JSON_KEY_RESPONSE] as? Map<*, *>
            val responseDataString = responseMap?.get(Constants.JSON_KEY_DATA) as? String

            if (responseDataString.isNullOrBlank()) return@executeWithErrorHandling emptyMap()

            if (productCount > 1) {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeCarouselAd)
            } else {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeAd)
            }
        }
    }

    /**
     * Fetch Native Product Listing Ads (PLA) for search pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param keyword: Search keyword to filter ads
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaSearchPageAds(
     *     cliUbid = "123",
     *     productCount = 8,
     *     keyword = "laptop",
     *     pageName = "search_results",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("store_1"),
     *         brands = listOf("Apple")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    override suspend fun fetchNativePlaSearchPageAds(
        cliUbid: String,
        productCount: Int,
        keyword: String,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adData = adFetcherSDK.fetchPlaSearchPageAds(
                cliUbid = cliUbid,
                productCount = productCount,
                keyword = keyword,
                pageName = pageName,
                filters = filters,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            val responseMap = adData[Constants.JSON_KEY_RESPONSE] as? Map<*, *>
            val responseDataString = responseMap?.get(Constants.JSON_KEY_DATA) as? String

            if (responseDataString.isNullOrBlank()) return@executeWithErrorHandling emptyMap()

            if (productCount > 1) {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeCarouselAd)
            } else {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeAd)
            }
        }
    }

    /**
     * Fetch Native Product Listing Ads (PLA) for product pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param skuIds: List of SKU identifiers for the products
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaProductPageAds(
     *     cliUbid = "123",
     *     productCount = 10,
     *     skuIds = listOf("sku_123", "sku_456"),
     *     pageName = "product_detail",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("store_1"),
     *         brands = listOf("Apple")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    override suspend fun fetchNativePlaProductPageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<String>,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adData = adFetcherSDK.fetchPlaProductPageAds(
                cliUbid = cliUbid,
                productCount = productCount,
                skuIds = skuIds,
                pageName = pageName,
                filters = filters,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            val responseMap = adData[Constants.JSON_KEY_RESPONSE] as? Map<*, *>
            val responseDataString = responseMap?.get(Constants.JSON_KEY_DATA) as? String

            if (responseDataString.isNullOrBlank()) return@executeWithErrorHandling emptyMap()

            if (productCount > 1) {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeCarouselAd)
            } else {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeAd)
            }
        }
    }

    /**
     * Fetch Native Product Listing Ads (PLA) based on page type.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: Type of page (HOME, SEARCH, CATEGORY, PRODUCT, PURCHASE, TPA)
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaAds(
     *     cliUbid = "123",
     *     pageType = PlaPageType.HOME,
     *     productCount = 5,
     *     pageName = "homepage",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("store_1"),
     *         brands = listOf("Apple")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    override suspend fun fetchNativePlaAds(
        cliUbid: String,
        pageType: PlaPageType,
        productCount: Int,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adData = adFetcherSDK.fetchPlaAds(
                cliUbid = cliUbid,
                pageType = pageType,
                productCount = productCount,
                pageName = pageName,
                filters = filters,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            val responseMap = adData[Constants.JSON_KEY_RESPONSE] as? Map<*, *>
            val responseDataString = responseMap?.get(Constants.JSON_KEY_DATA) as? String

            if (responseDataString.isNullOrBlank()) return@executeWithErrorHandling emptyMap()

            if (productCount > 1) {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeCarouselAd)
            } else {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeAd)
            }
        }
    }

    /**
     * Fetch Native Product Listing Ads (PLA) for purchase pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param skuIds: List of SKU identifiers for the purchased products
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaPurchasePageAds(
     *     cliUbid = "123",
     *     productCount = 10,
     *     skuIds = listOf("purchased_sku_1", "purchased_sku_2"),
     *     pageName = "purchase_confirmation",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("premium_store"),
     *         brands = listOf("Apple")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    override suspend fun fetchNativePlaPurchasePageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<String>,
        pageName: String?,
        filters: ProductAdsFilter?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return StandardErrorHandler.executeWithErrorHandling(errorCallback) {
            val adData = adFetcherSDK.fetchPlaPurchasePageAds(
                cliUbid = cliUbid,
                productCount = productCount,
                skuIds = skuIds,
                pageName = pageName,
                filters = filters,
                errorCallback = errorCallback
            ) ?: return@executeWithErrorHandling null

            val responseMap = adData[Constants.JSON_KEY_RESPONSE] as? Map<*, *>
            val responseDataString = responseMap?.get(Constants.JSON_KEY_DATA) as? String

            if (responseDataString.isNullOrBlank()) return@executeWithErrorHandling emptyMap()

            if (productCount > 1) {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeCarouselAd)
            } else {
                parseNativeAdResponse(responseDataString, cliUbid, NativeAdFormat.NativeAd)
            }
        }
    }

    /**
     * Fetch PDA (Product Display Ad) ads with ad units.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit: Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams(optional): Filter params that you want to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchPdaAdsWithAu(
     *                 cliUbid = "123",
     *                 pageType = "demo_page",
     *                 adUnit = "pda_ads",
     *                 targetingParams = listOf(TargetingParams("category", "electronics")),
     *                 errorCallback = null
     * )
     * ```
     */
    override suspend fun fetchPdaAdWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>?,
        errorCallback: ErrorCallback?
    ): Map<String, Any>? {
        return fetchDisplayAdsWithAuCommon(
            cliUbid = cliUbid,
            pageType = pageType,
            adUnit = adUnit,
            productCount = 1,
            targetingParams = targetingParams,
            errorCallback = errorCallback,
            adFormat = DisplayAdFormat.PDA
        )
    }


    override fun parseNativeAdResponse(
        adData: String,
        cliUbid: String,
        adFormat: NativeAdFormat
    ): Map<String, Any> {
        if (adData.isEmpty()) return emptyMap()

        return StandardErrorHandler.executeWithParsingErrorHandling {

            val adMap = JSONParsing.parseNativeAdData(adData.toString(), cliUbid)
            when (adFormat) {
                NativeAdFormat.NativeAd -> {
                    val firstAd = adMap.values.firstOrNull()?.firstOrNull()
                    if (firstAd != null) {
                        mapOf("ad" to firstAd)
                    } else {
                        emptyMap()
                    }
                }

                NativeAdFormat.NativeCarouselAd -> {
                    val allAds = adMap.values.flatten()
                    if (allAds.isNotEmpty()) {
                        mapOf("ad" to allAds)
                    } else {
                        emptyMap()
                    }
                }
            }
        }
    }
}