package com.ai.osmos.ads.renderer

import com.ai.osmos.models.ads.ProductAdsFilter
import com.ai.osmos.models.ads.TargetingParams
import com.ai.osmos.models.enums.DisplayAdFormat
import com.ai.osmos.models.enums.NativeAdFormat
import com.ai.osmos.utils.error.ErrorCallback

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: AdRendererInterface
 *
 * Interface for ad rendering functionality in the OSMOS SDK.
 * Provides methods for fetching and preparing different types of ads.
 */
public interface AdRendererInterface {

    /**
     * Fetch banner ads with ad units.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit: Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams: Optional filter params that you want to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchBannerAdsWithAu(
     *     cliUbid = "123",
     *     pageType = "demo_page",
     *     adUnit = "banner_ads",
     *     targetingParams = listOf(TargetingParams("category", "electronics")),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchBannerAdsWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch carousel ads with ad units.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit: Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams: Optional filter params that you want to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchCarouselAdWithAu(
     *     cliUbid = "123",
     *     pageType = "demo_page",
     *     adUnit = "carousel_ads",
     *     targetingParams = listOf(TargetingParams("category", "electronics")),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchCarouselAdWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch multiple ads in carousel format with ad units.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit: Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param productCount: Number of ads responded. Maximum 10.
     * @param targetingParams: Optional filter params that you want to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchMultiAdCarouselWithAu(
     *     cliUbid = "123",
     *     pageType = "demo_page",
     *     adUnit = "all_in_one",
     *     productCount = 3,
     *     targetingParams = listOf(TargetingParams("category", "electronics")),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchMultiAdCarouselWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        productCount: Int,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch Product Listing Ads (PLA) for category pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param categories: List of category names to filter ads
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaCategoryPageAds(
     *     cliUbid = "123",
     *     productCount = 5,
     *     categories = listOf("Electronics", "Computers"),
     *     pageName = "category_listing",
     *     filters = ProductAdsFilter.create(brands = listOf("Apple")),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchNativePlaCategoryPageAds(
        cliUbid: String,
        productCount: Int,
        categories: List<String>,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch Tagged Product Ads (TPA) for specific products.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param skuIds: List of SKU identifiers for the products
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchTpaPageAds(
     *     cliUbid = "123",
     *     productCount = 5,
     *     skuIds = listOf("sku_1", "sku_2"),
     *     pageName = "product_listing",
     *     filters = ProductAdsFilter.create(storeIds = listOf("store_1")),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchNativePlaTpaPageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<Any>,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch Native Product Listing Ads (PLA) for home pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaHomePageAds(
     *     cliUbid = "123",
     *     productCount = 10,
     *     pageName = "home_page",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("store_1"),
     *         brands = listOf("Brand1")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchNativePlaHomePageAds(
        cliUbid: String,
        productCount: Int,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch Native Product Listing Ads (PLA) for search pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param keyword: Search keyword to filter ads
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaSearchPageAds(
     *     cliUbid = "123",
     *     productCount = 8,
     *     keyword = "laptop",
     *     pageName = "search_results",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("store_1"),
     *         brands = listOf("Apple")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchNativePlaSearchPageAds(
        cliUbid: String,
        productCount: Int,
        keyword: String,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch Native Product Listing Ads (PLA) for product pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param skuIds: List of SKU identifiers for the products
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaProductPageAds(
     *     cliUbid = "123",
     *     productCount = 10,
     *     skuIds = listOf("sku_123", "sku_456"),
     *     pageName = "product_detail",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("store_1"),
     *         brands = listOf("Apple")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchNativePlaProductPageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<String>,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch Native Product Listing Ads (PLA) based on page type.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: Type of page (HOME, SEARCH, CATEGORY, PRODUCT, PURCHASE, TPA)
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaAds(
     *     cliUbid = "123",
     *     pageType = PlaPageType.HOME,
     *     productCount = 5,
     *     pageName = "homepage",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("store_1"),
     *         brands = listOf("Apple")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchNativePlaAds(
        cliUbid: String,
        pageType: com.ai.osmos.models.enums.PlaPageType,
        productCount: Int,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch Native Product Listing Ads (PLA) for purchase pages.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param productCount: Number of ads to fetch. Maximum 10.
     * @param skuIds: List of SKU identifiers for the purchased products
     * @param pageName: Optional page name identifier
     * @param filters: Optional additional filters to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed ad data, or an empty map if no ads were found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchNativePlaPurchasePageAds(
     *     cliUbid = "123",
     *     productCount = 10,
     *     skuIds = listOf("purchased_sku_1", "purchased_sku_2"),
     *     pageName = "purchase_confirmation",
     *     filters = ProductAdsFilter.create(
     *         storeIds = listOf("premium_store"),
     *         brands = listOf("Apple")
     *     ),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchNativePlaPurchasePageAds(
        cliUbid: String,
        productCount: Int,
        skuIds: List<String>,
        pageName: String? = null,
        filters: ProductAdsFilter? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Fetch PDA (Product Display Ad) ads with ad units.
     *
     * @param cliUbid: Retailers Generated Id to identify unique shopper
     * @param pageType: page type from which request made. It should be registered while configuring ad inventory
     * @param adUnit: Ad unit tags defined for the different slots on the page. au parameters are case sensitive. You can fetch multiple ad units using by specifying list of au. You can use maximum 5 au per ad request
     * @param targetingParams: Optional filter params that you want to apply
     * @param errorCallback: Optional error callback for handling errors
     * @return A map containing a single `"ad"` key with the parsed [BaseAd] object, or an empty map if no ad was found.
     *
     * Example:
     * ```
     * val adData = adRender.fetchPdaAdsWithAu(
     *     cliUbid = "123",
     *     pageType = "demo_page",
     *     adUnit = "pda_ads",
     *     targetingParams = listOf(TargetingParams("category", "electronics")),
     *     errorCallback = null
     * )
     * ```
     */
    suspend fun fetchPdaAdWithAu(
        cliUbid: String,
        pageType: String,
        adUnit: String,
        targetingParams: List<TargetingParams>? = null,
        errorCallback: ErrorCallback? = null
    ): Map<String, Any>?

    /**
     * Parse display ad response data into structured format
     *
     * @param adData Raw ad data from API response
     * @param cliUbid Client unique browser ID for tracking
     * @param adFormat Display ad format type (Banner, Carousel, MultiAdCarousel)
     * @return Parsed ad data map with "ad" key containing the BaseAd object
     *
     * @example
     * ```kotlin
     * // Example usage - similar to MainActivity showBannerAd function
     * val sdk = OsmosSDK.clientId("10088010").debug(true).build()
     * val adRenderer = sdk.adRenderer()
     *
     * // Sample banner ad JSON response from server
     * val bannerAdRaw = """
     *     {
     *       "ads": {
     *         "banner_ads": [
     *           {
     *             "client_id": 123,
     *             "au": "banner_ads",
     *             "rank": 1,
     *             "click_tracking_url": "https://demo.io/click?client_id=123&...",
     *             "elements": {
     *               "type": "IMAGE",
     *               "value": "https://osads.gumlet.io/image/upload/v1747206595/lacy7ljgoxevl9rgmeqo.jpg",
     *               "width": 742,
     *               "height": 355,
     *               "destination_url": "https://demoSDK.com"
     *             },
     *             "impression_tracking_url": "https://demo.io/events?client_id=123&event_name=funnel_impression&uclid=...",
     *             "uclid": "123",
     *             "crt": "osmos_sdk/image/v1/320x50"
     *           }
     *         ]
     *       }
     *     }
     * """.trimIndent()
     *
     * // Parse the response
     * val bannerData = adRenderer.parseDisplayAdResponse(
     *     bannerAdRaw,
     *     "xyz",
     *     DisplayAdFormat.Banner
     * )
     *
     * // Use the parsed data
     * if (bannerData.isNotEmpty()) {
     *     // Create BannerAdView
     *     val adView = sdk.bannerAdView().showAd(
     *         context = context,
     *         ad = bannerData,
     *         adViewSettings = BannerAdSettings(
     *             height = 300,
     *             width = 290
     *         )
     *     )
     *
     *     // Add to layout
     *     parentLayout.addView(adView)
     * }
     * ```
     */
    fun parseDisplayAdResponse(
        adData: String,
        cliUbid: String,
        adFormat: DisplayAdFormat
    ): Map<String, Any>

    /**
     * Parse native ad response data into structured format.
     *
     * @param adData Raw ad data from API response
     * @param cliUbid Client unique browser ID for tracking
     * @param adFormat Native ad format type (NativeAd, NativeCarouselAd)
     * @return Parsed ad data map with "ad" key containing the BaseAd object, or empty map if parsing fails
     *
     * Example:
     * ```
     * val rawAdData = """{"ads": {"native_ads": [{"client_id": 123, "au": "native_ads", ...}]}}"""
     * val parsedData = adRenderer.parseNativeAdResponse(
     *     adData = rawAdData,
     *     cliUbid = "xyz",
     *     adFormat = NativeAdFormat.NativeAd
     * )
     *
     * // Use the parsed data
     * if (parsedData.isNotEmpty()) {
     *     val nativeAdView = osmosSDK.nativeAdView().showAd(
     *         context = context,
     *         ad = parsedData,
     *         adViewSettings = NativeAdSettings()
     *     )
     * }
     * ```
     */
    fun parseNativeAdResponse(
        adData: String,
        cliUbid: String,
        adFormat: NativeAdFormat
    ): Map<String, Any>
}