package com.ai.osmos.ads.renderer

import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.media3.exoplayer.ExoPlayer
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: PIPPlayerLifecycleObserver
 */

class PIPPlayerLifecycleObserver(
    private val player: ExoPlayer,
    private var errorCallback: ErrorCallback? = null
) : DefaultLifecycleObserver {

    /**
     * Set an error callback to handle errors that occur during lifecycle events.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
    }

    /**
     * Logs an error message using structured error handling.
     *
     * @param message The error message to be logged.
     * @param errorType The specific error type from the OsmosError enum.
     * @param throwable The original throwable that caused the error (optional).
     */
    private fun errorLog(message: String, error: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(error, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    override fun onPause(owner: LifecycleOwner) {
        try {
            player.pause()
        } catch (e: IllegalStateException) {
            errorLog("${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    override fun onResume(owner: LifecycleOwner) {
        try {
            player.play()
        } catch (e: IllegalStateException) {
            errorLog("${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    override fun onDestroy(owner: LifecycleOwner) {
        try {
            player.release()
            owner.lifecycle.removeObserver(this)
        } catch (e: Exception) {
            errorLog("${e.message}", OsmosError.UNKNOWN, e)
        } finally {
            // Clear error callback to prevent memory leaks
            clearErrorCallback()
        }
    }
}
