package com.ai.osmos.ads.renderer.loaders

import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.graphics.Rect
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.FrameLayout
import android.widget.LinearLayout
import android.widget.TextView
import androidx.media3.ui.PlayerView
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.ai.osmos.models.ads.AdElement
import com.ai.osmos.models.ads.CarouselAd
import com.ai.osmos.models.ads.CarouselAdElement
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.MultiAd
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.isVisibleInScreen
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope
import kotlin.math.abs

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: CarouselLoader
 */
class CarouselLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface
) {
    private val imageLoader = ImageLoader(coroutineScope, adTracker)
    private val videoLoader = VideoLoader(coroutineScope, adTracker)
    private var currentIndex = -1
    private var dotsContainer: LinearLayout? = null
    private lateinit var recyclerView: RecyclerView

    private lateinit var multiRecyclerView: RecyclerView

    /**
     * Set an error callback to handle errors that occur during carousel ad loading or playback.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        imageLoader.setErrorCallback(callback)
        videoLoader.setErrorCallback(callback)
    }

    /**
     * Clear the error callback to prevent memory leaks.
     * Note: Only clears ImageLoader callback since VideoLoader callback
     * is already cleared by releaseAllPlayers() -> safeReleaseAllPlayers().
     */
    fun clearErrorCallback() {
        imageLoader.clearErrorCallback()
        // videoLoader.clearErrorCallback() - Not needed, cleared by releaseAllPlayers()
    }


    /**
     * Logs an error message using structured error handling.
     *
     * @param message The error message to be logged.
     * @param errorType The specific error type from the OsmosError enum.
     * @param throwable The original throwable that caused the error (optional).
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    private var isCarouselAd: Boolean = false
    private var errorCallback: ErrorCallback? = null
    fun renderCarouselView(
        context: Context,
        width: Int?,
        height: Int?,
        elementWidth: Int?,
        elementHeight: Int?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        cliUbid: String,
        adList: List<CarouselAdElement>,
        carouselAd: CarouselAd,
        adClickedListener: ((adData: Map<String, Any>, clickedItem: Map<String, Any>, destinationUrl: String) -> Unit)?,
        onViewLoadListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null
    ): View {

        val container = FrameLayout(context)
        val result = setMaxHeightWidth(context, width, height, true, carouselAd = carouselAd, null)

        recyclerView = RecyclerView(context).apply {
            layoutManager =
                LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false).apply {
                    layoutParams = ViewGroup.LayoutParams(
                        ViewGroup.LayoutParams.WRAP_CONTENT,
                        result.finalCarouselHeight
                    )
                }
            isCarouselAd = true
            adapter = CarouselAdapter(
                context,
                adList,
                carouselAd,
                cliUbid,
                elementWidth,
                elementHeight,
                result.carouselWidth,
                result.carouselHeight,
                result.firstElementWidth,
                result.firstElementHeight,
                result.isClamped,
                adLabelText,
                adLabelAlignment,
                adClickedListener,
                onViewLoadListener
            )
        }

        setItemDecoration(context, recyclerView)

        container.addView(recyclerView)

        addDotIndicator(context, container, null, adList)

        addScrollListener(isCarouselAd, recyclerView)

        // Add visibility tracking for carousel impression - only fires when 50%+ visible
        setupCarouselVisibilityTracking(container, carouselAd, cliUbid)

        return container
    }

    fun renderMultiAdCarouselView(
        context: Context,
        width: Int?,
        height: Int?,
        multiAd: List<MultiAd>,
        elementWidth: Int?,
        elementHeight: Int?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        adClickListener: ((Map<String, Any>?) -> Unit)?,
        onViewLoadListener: ((adMetadata: Map<String, Any>, String) -> Unit)? = null

    ): View {
        val container = FrameLayout(context)

        val result = setMaxHeightWidth(context, width, height, false, null, multiAd.firstOrNull())

        multiRecyclerView = RecyclerView(context).apply {
            layoutManager =
                LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false).apply {
                    layoutParams = ViewGroup.LayoutParams(
                        ViewGroup.LayoutParams.WRAP_CONTENT,
                        result.finalCarouselHeight
                    )
                }
            isCarouselAd = false

            adapter = MultiAdAdapter(
                context,
                multiAd,
                result.carouselWidth,
                result.carouselHeight,
                result.firstElementWidth,
                result.firstElementHeight,
                result.isClamped,
                elementWidth,
                elementHeight,
                adLabelText,
                adLabelAlignment,
                adClickListener,
                onViewLoadListener
            )
        }

        setItemDecoration(context, multiRecyclerView)

        container.addView(multiRecyclerView)

        addDotIndicator(context, container, multiAd, null)

        addScrollListener(isCarouselAd, multiRecyclerView)

        return container
    }

    private fun addScrollListener(isCarouselAd: Boolean, recyclerView: RecyclerView) {

        recyclerView.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrolled(rv: RecyclerView, dx: Int, dy: Int) {
                handleVisibleItem(isCarouselAd)
            }
        })

        recyclerView.post {
            handleVisibleItem(isCarouselAd)
        }
    }

    private fun addDotIndicator(
        context: Context,
        container: FrameLayout,
        multiAd: List<MultiAd>?,
        adList: List<CarouselAdElement>?
    ) {
        dotsContainer = LinearLayout(context).apply {
            orientation = LinearLayout.HORIZONTAL
            gravity = Gravity.CENTER
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT,
                FrameLayout.LayoutParams.WRAP_CONTENT
            ).apply {
                gravity = Gravity.BOTTOM or Gravity.CENTER_HORIZONTAL
                bottomMargin = 16 // Optional: add some bottom margin
            }
        }
        (multiAd ?: adList)?.forEach { _ ->
            dotsContainer?.addView(TextView(context).apply {
                text = "•"
                textSize = 20f
                setTextColor(Color.GRAY)
            })
        }
        container.addView(dotsContainer)
    }

    private fun setItemDecoration(context: Context, multiRecyclerView: RecyclerView) {
        val density = context.resources.displayMetrics.density

        val margin = (10 * density).toInt()
        multiRecyclerView.addItemDecoration(object : RecyclerView.ItemDecoration() {
            override fun getItemOffsets(
                outRect: Rect,
                view: View,
                parent: RecyclerView,
                state: RecyclerView.State
            ) {
                val position = parent.getChildAdapterPosition(view)
                outRect.left = if (position == 0) margin else margin / 2
                outRect.right = if (position == state.itemCount - 1) margin else margin / 2
            }
        })
    }

    private fun handleVisibleItem(isCarouselAd: Boolean) {
        val center = if (isCarouselAd) recyclerView.width / 2 else multiRecyclerView.width / 2
        var bestIndex = -1
        var minDistance = Int.MAX_VALUE

        // Find the closest item to the center
        if (isCarouselAd) {
            for (i in 0 until recyclerView.childCount) {
                if (recyclerView.childCount == 0) return
                val view = recyclerView.getChildAt(i)
                val distance = abs(view.left + view.width / 2 - center)
                if (distance < minDistance) {
                    bestIndex = recyclerView.getChildAdapterPosition(view)
                    minDistance = distance
                }
            }
        } else {
            for (i in 0 until multiRecyclerView.childCount) {
                if (multiRecyclerView.childCount == 0) return
                val view = multiRecyclerView.getChildAt(i)
                val distance = abs(view.left + view.width / 2 - center)
                if (distance < minDistance) {
                    bestIndex = multiRecyclerView.getChildAdapterPosition(view)
                    minDistance = distance
                }
            }
        }

        // If the bestIndex has changed, update the visible item
        if (bestIndex != currentIndex) {
            // Pause the currently playing video (if any)

            pauseCurrentPlayingVideo(
                if (isCarouselAd) recyclerView else multiRecyclerView,
                currentIndex,
                isCarouselAd
            )

            // Update the currentIndex
            currentIndex = bestIndex
            updateDots()


            // Play the new video
            playCurrentVideo(
                if (isCarouselAd) recyclerView else multiRecyclerView,
                currentIndex,
                isCarouselAd
            )
        }
    }

    private fun playCurrentVideo(
        recyclerView: RecyclerView,
        currentIndex: Int,
        isCarouselAd: Boolean
    ) {
        val holder = recyclerView.findViewHolderForAdapterPosition(currentIndex)
        when {
            isCarouselAd && holder is CarouselAdapter.VideoViewHolder -> {
                holder.frame.post {
                    videoLoader.playExclusive(holder.playerView)
                }
            }

            !isCarouselAd && holder is MultiAdAdapter.VideoViewHolder -> {
                holder.frame.post {
                    videoLoader.playExclusive(holder.playerView)
                }
            }
        }
    }

    private fun pauseCurrentPlayingVideo(
        recyclerView: RecyclerView,
        currentIndex: Int,
        isCarouselAd: Boolean
    ) {
        val holder = recyclerView.findViewHolderForAdapterPosition(currentIndex)
        when {
            isCarouselAd && holder is CarouselAdapter.VideoViewHolder -> {
                holder.playerView.player?.playWhenReady = false
            }

            !isCarouselAd && holder is MultiAdAdapter.VideoViewHolder -> {
                holder.playerView.player?.playWhenReady = false
            }
        }
    }

    private fun updateDots() {
        dotsContainer?.let {
            for (i in 0 until it.childCount) {
                (it.getChildAt(i) as? TextView)?.setTextColor(
                    if (i == currentIndex) Color.WHITE else Color.GRAY
                )
            }
        }
    }


    data class LayoutResult(
        val finalCarouselWidth: Int,
        val finalCarouselHeight: Int,
        val firstElementWidth: Int,
        val firstElementHeight: Int,
        val carouselWidth: Int,
        val carouselHeight: Int,
        val isClamped: Boolean
    )

    private fun setMaxHeightWidth(
        context: Context,
        width: Int?,
        height: Int?,
        isCarouselAd: Boolean,
        carouselAd: CarouselAd?,
        multiAd: MultiAd?
    ): LayoutResult {

        val screenWidth = Resources.getSystem().displayMetrics.widthPixels.toFloat()
        val screenHeight = Resources.getSystem().displayMetrics.heightPixels.toFloat()

        val maxAdWidth = (screenWidth * 0.8f).toInt()
        val maxAdHeight = (screenHeight * 0.8f).toInt()

        var firstElementWidth = maxAdWidth
        var firstElementHeight = maxAdHeight

        if (isCarouselAd) {
            val firstElement = carouselAd?.elements?.carouselCards?.firstOrNull()
            firstElementWidth = firstElement?.media?.width ?: maxAdWidth
            firstElementHeight = firstElement?.media?.height ?: maxAdHeight
        } else {
            firstElementWidth = multiAd?.elements?.width ?: maxAdWidth
            firstElementHeight = multiAd?.elements?.height ?: maxAdHeight
        }

        val carouselWidth = minOf(width ?: maxAdWidth, maxAdWidth)
        val carouselHeight = minOf(height ?: firstElementHeight, maxAdHeight)

        var isClamped = false
        if (width != null && height != null) {
            isClamped = (width > maxAdWidth || height > maxAdHeight)
        }

        val finalCarouselWidth =
            if (isClamped) carouselWidth else ViewUtils.dpToPx(context, carouselWidth)
        val finalCarouselHeight =
            if (isClamped) carouselHeight else ViewUtils.dpToPx(context, carouselHeight)

        return LayoutResult(
            finalCarouselWidth = finalCarouselWidth,
            finalCarouselHeight = finalCarouselHeight,
            firstElementWidth = firstElementWidth,
            firstElementHeight = firstElementHeight,
            carouselHeight = carouselHeight,
            carouselWidth = carouselWidth,
            isClamped = isClamped
        )
    }

    /**
     * Set up 50% visibility tracking for carousel impression - same pattern as ImageLoader
     * Only fires impression when carousel container is 50%+ visible in screen
     */
    private fun setupCarouselVisibilityTracking(
        container: FrameLayout,
        carouselAd: CarouselAd,
        cliUbid: String
    ) {
        var preDrawListener: ViewTreeObserver.OnPreDrawListener? = null
        var attachStateListener: View.OnAttachStateChangeListener? = null

        preDrawListener = object : ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                // Check if container is 50%+ visible, same as ImageLoader
                if (container.isVisibleInScreen()) {
                    // Fire carousel impression tracking
                    trackCarouselImpression(carouselAd.uclid, cliUbid)

                    // Remove listener once impression is fired - prevents duplicate firing
                    container.viewTreeObserver.removeOnPreDrawListener(this)
                    attachStateListener?.let {
                        container.removeOnAttachStateChangeListener(it)
                    }
                }
                return true
            }
        }

        attachStateListener = object : View.OnAttachStateChangeListener {
            override fun onViewDetachedFromWindow(v: View) {
                preDrawListener?.let {
                    container.viewTreeObserver.removeOnPreDrawListener(it)
                }
                container.removeOnAttachStateChangeListener(this)
            }

            override fun onViewAttachedToWindow(v: View) {
                // No-op
            }
        }

        container.addOnAttachStateChangeListener(attachStateListener)
        container.viewTreeObserver.addOnPreDrawListener(preDrawListener)
    }

    /**
     * Track carousel impression with ad tracker
     */
    private fun trackCarouselImpression(uclid: String, cliUbid: String) {
        // Track impression via ad tracker
        adTracker.trackImpression(uclid, cliUbid)
    }


    inner class CarouselAdapter(
        private val context: Context,
        private val items: List<CarouselAdElement>,
        private val carouselAd: CarouselAd,
        private val cliUbid: String,
        private val elementWidth: Int?,
        private val elementHeight: Int?,
        private val carouselWidth: Int,
        private val carouselHeight: Int,
        private val firstElementWidth: Int,
        private val firstElementHeight: Int,
        private val isClamped: Boolean,
        private val adLabelText: String? = null,
        private val adLabelAlignment: Int? = null,
        private val adClickedListener: ((adData: Map<String, Any>, clickedItem: Map<String, Any>, destinationUrl: String) -> Unit)?,
        private val onViewLoadListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null

    ) : RecyclerView.Adapter<RecyclerView.ViewHolder>() {

        private val TYPE_VIDEO = 1
        private val TYPE_IMAGE = 0
        private var isFirstVisible = true

        override fun getItemViewType(position: Int): Int {
            val type = items[position].media.type
            return if (type.equals("video", ignoreCase = true)) TYPE_VIDEO else TYPE_IMAGE
        }

        override fun getItemCount(): Int {
            return items.size
        }

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
            val frame = FrameLayout(context)
            return if (viewType == TYPE_VIDEO) VideoViewHolder(frame) else ImageViewHolder(frame)
        }

        override fun onViewRecycled(holder: RecyclerView.ViewHolder) {
            if (holder is VideoViewHolder) {
                videoLoader.safeStopAndReleasePlayer(holder.playerView)
            }
            super.onViewRecycled(holder)
        }

        override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
            val item = items[position]
            val url = item.media.value
            val cardWidth = minOf((elementWidth ?: firstElementWidth), carouselWidth)
            val cardHeight = minOf((elementHeight ?: firstElementHeight), carouselHeight)

            val density = context.resources.displayMetrics.density
            val pxW = (cardWidth * density).toInt()
            val pxH = (cardHeight * density).toInt()

            when (holder) {
                is VideoViewHolder -> {
                    holder.frame.removeAllViews()
                    // Generate a "per-element" video metadata object
                    val videoAd = getVideoAdObject(carouselAd, item, cardWidth, cardHeight)

                    val videoView = videoLoader.createCarouselVideoView(
                        context,
                        videoAd,
                        cardWidth,
                        cardHeight,
                        "$cliUbid-$position",
                        isCarousal = true
                    )
                    holder.playerView = videoView.getChildAt(0) as PlayerView
                    holder.frame.setOnClickListener {
                        adClickedListener?.invoke(
                            carouselAd.toMap() ?: emptyMap(),
                            createCarouselCardMap(item),
                            videoAd.elements.destinationUrl.toString()
                        )
                        videoAd.cliUbid?.let { cliUbid ->
                            adTracker.trackAdClick(videoAd.uclid, cliUbid)
                        }
                    }
                    val adTag = adLabelText
                    if (!adTag.isNullOrEmpty()) {
                        val container = ViewUtils.createAdIndicatorContainer(
                            context,
                            adTag,
                            adLabelAlignment,
                            videoView.layoutParams.width,
                            videoView.layoutParams.height,
                            null, videoView
                        )
                        holder.frame.addView(container, FrameLayout.LayoutParams(pxW, pxH))
                    } else holder.frame.addView(videoView, FrameLayout.LayoutParams(pxW, pxH))
                }

                is ImageViewHolder -> {
                    holder.frame.removeAllViews()
                    val imageAd = getImageAdObject(carouselAd, item, pxW, pxH)

                    val imageView = imageLoader.createImageView(
                        context,
                        cardWidth,
                        cardHeight,
                        isClamped = isClamped
                    )
                    imageLoader.loadImage(
                        imageAd,
                        imageView,
                        "$cliUbid-$position",
                        isCarouselAd = true
                    )
                    holder.frame.setOnClickListener {
                        adClickedListener?.invoke(
                            carouselAd.toMap() ?: emptyMap(),
                            createCarouselCardMap(item),
                            imageAd.elements.destinationUrl.toString()
                        )
                        imageAd.cliUbid?.let { cliUbid ->
                            adTracker.trackAdClick(imageAd.uclid, cliUbid)
                        }

                    }
                    val adTag = adLabelText
                    if (!adTag.isNullOrEmpty()) {
                        val container = ViewUtils.createAdIndicatorContainer(
                            context,
                            adTag,
                            adLabelAlignment,
                            imageView.layoutParams.width,
                            imageView.layoutParams.height,
                            imageView, null
                        )
                        holder.frame.addView(container, FrameLayout.LayoutParams(pxW, pxH))
                    } else holder.frame.addView(imageView, FrameLayout.LayoutParams(pxW, pxH))
                }
            }
            if (isFirstVisible && position == 0) {
                isFirstVisible = false
                onViewLoadListener?.invoke(carouselAd.toMap())
            }
        }

        inner class VideoViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame) {
            lateinit var playerView: PlayerView
        }

        inner class ImageViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame)

        private fun getVideoAdObject(
            carouselAd: CarouselAd,
            item: CarouselAdElement,
            width: Int,
            height: Int
        ): VideoAd {
            return VideoAd(
                clientId = carouselAd.clientId, // or extract if available
                au = carouselAd.au,        // optional ad unit or campaign string
                rank = carouselAd.rank,       // optional rank
                clickTrackingUrl = carouselAd.clickTrackingUrl,
                impressionTrackingUrl = carouselAd.impressionTrackingUrl, // add if available
                uclid = carouselAd.uclid,     // unique click ID if available
                crt = carouselAd.crt,       // creative type if relevant
                elements = AdElement(
                    type = item.media.type,
                    value = item.media.value,
                    height = height,
                    width = width,
                    destinationUrl = item.destinationUrl
                ),
                adMetadata = carouselAd.adMetadata,
                cliUbid = cliUbid
            )
        }

        private fun getImageAdObject(
            carouselAd: CarouselAd,
            item: CarouselAdElement,
            width: Int,
            height: Int
        ): ImageAd {
            return ImageAd(
                clientId = carouselAd.clientId, // or extract if available
                au = carouselAd.au,        // optional ad unit or campaign string
                rank = carouselAd.rank,       // optional rank
                clickTrackingUrl = carouselAd.clickTrackingUrl,
                impressionTrackingUrl = carouselAd.impressionTrackingUrl, // add if available
                uclid = carouselAd.uclid,     // unique click ID if available
                crt = carouselAd.crt,       // creative type if relevant
                elements = AdElement(
                    type = item.media.type,
                    value = item.media.value,
                    height = height,
                    width = width,
                    destinationUrl = item.destinationUrl
                ),
                adMetadata = carouselAd.adMetadata,
                cliUbid = cliUbid
            )
        }

        private fun createCarouselCardMap(carouselAdElement: CarouselAdElement): Map<String, Any> {
            return mapOf(
                "value" to carouselAdElement.media.value,
                "height" to carouselAdElement.media.height,
                "width" to carouselAdElement.media.width,
                "type" to carouselAdElement.media.type,
                "destination_url" to carouselAdElement.destinationUrl
            )
        }
    }

    fun releaseAllPlayers() {
        videoLoader.safeReleaseAllPlayers()
        // Note: ImageLoader doesn't have players to release, but error callback is cleared in clearErrorCallback()
    }

    /**
     * Comprehensive cleanup method that releases all resources and clears all callbacks.
     */
    fun cleanup() {
        videoLoader.safeReleaseAllPlayers() // This clears VideoLoader's error callback
        imageLoader.clearErrorCallback() // Clear ImageLoader's error callback separately
    }

    inner class MultiAdAdapter(
        private val context: Context,
        private val items: List<MultiAd>,
        private val carouselWidth: Int,
        private val carouselHeight: Int,
        private val firstElementWidth: Int,
        private val firstElementHeight: Int,
        private val isClamped: Boolean,
        private val elementWidth: Int?,
        private val elementHeight: Int?,
        private val adLabelText: String?,
        private val adLabelAlignment: Int?,
        private val adClickListener: ((Map<String, Any>?) -> Unit)?,
        private val onViewLoadListener: ((adMetadata: Map<String, Any>, String) -> Unit)? = null

    ) : RecyclerView.Adapter<RecyclerView.ViewHolder>() {


        private val TYPE_VIDEO = 1
        private val TYPE_IMAGE = 0

        override fun getItemViewType(position: Int): Int {
            val multiAd = items[position]
            val type = multiAd.elements.type
            return if (type.equals("video", ignoreCase = true)) TYPE_VIDEO else TYPE_IMAGE
        }

        override fun getItemCount(): Int {
            return items.size
        }

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
            val frame = FrameLayout(context)
            return if (viewType == TYPE_VIDEO) VideoViewHolder(frame) else ImageViewHolder(frame)
        }

        override fun onViewRecycled(holder: RecyclerView.ViewHolder) {
            if (holder is VideoViewHolder) {
                videoLoader.safeStopAndReleasePlayer(holder.playerView)
            }
            super.onViewRecycled(holder)
        }

        override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
            val multiAd = items[position]
            val cardWidth = minOf((multiAd.elements.width ?: firstElementWidth), carouselWidth)
            val cardHeight = minOf((multiAd.elements.height ?: firstElementHeight), carouselHeight)

            val density = context.resources.displayMetrics.density
            val pxW = (cardWidth * density).toInt()
            val pxH = (cardHeight * density).toInt()

            onViewLoadListener?.invoke(multiAd.toMap(), multiAd.cliUbid.toString())
            when (holder) {
                is VideoViewHolder -> {
                    holder.frame.removeAllViews()
                    // Generate a "per-element" video metadata object
                    val videoAd = getVideoAdObject(multiAd, cardHeight, cardWidth)
                    val videoView = videoLoader.createCarouselVideoView(
                        context,
                        videoAd,
                        cardWidth,
                        cardHeight,
                        "${videoAd.cliUbid}-$position"
                    )

                    val playerView = videoView.getChildAt(0) as PlayerView
                    holder.playerView = playerView
                    holder.frame.setOnClickListener {
                        adClickListener?.invoke(videoAd.toMap())
                        videoAd.cliUbid?.let { cliUbid ->
                            adTracker.trackAdClick(videoAd.uclid, cliUbid)
                        }
                    }

                    if (!adLabelText.isNullOrEmpty()) {
                        val container = ViewUtils.createAdIndicatorContainer(
                            context,
                            adLabelText,
                            adLabelAlignment,
                            videoView.layoutParams.width,
                            videoView.layoutParams.height,
                            null, videoView
                        )
                        holder.frame.addView(container, FrameLayout.LayoutParams(pxW, pxH))
                    } else holder.frame.addView(videoView, FrameLayout.LayoutParams(pxW, pxH))
                }

                is ImageViewHolder -> {
                    holder.frame.removeAllViews()
                    val imageAd = getImageAdObject(multiAd, cardHeight, cardWidth)

                    val imageView =
                        imageLoader.createImageView(context, cardWidth, cardHeight, isClamped)

                    imageLoader.loadImage(
                        imageAd,
                        imageView,
                        "${imageAd.cliUbid}-$position"
                    )
                    holder.frame.setOnClickListener {
                        adClickListener?.invoke(imageAd.toMap())
                        imageAd.cliUbid?.let { cliUbid ->
                            adTracker.trackAdClick(imageAd.uclid, cliUbid)
                        }
                    }

                    if (!adLabelText.isNullOrEmpty()) {
                        val container = ViewUtils.createAdIndicatorContainer(
                            context,
                            adLabelText, adLabelAlignment,
                            imageView.layoutParams.width,
                            imageView.layoutParams.height,
                            imageView, null
                        )
                        holder.frame.addView(container, FrameLayout.LayoutParams(pxW, pxH))
                    } else holder.frame.addView(imageView, FrameLayout.LayoutParams(pxW, pxH))
                }
            }

        }

        inner class VideoViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame) {
            lateinit var playerView: PlayerView
        }

        inner class ImageViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame) {

        }


        private fun getVideoAdObject(multiAd: MultiAd, cardHeight: Int, cardWidth: Int): VideoAd {
            return VideoAd(
                clientId = multiAd.clientId,
                au = multiAd.au,
                rank = multiAd.rank,
                clickTrackingUrl = multiAd.clickTrackingUrl,
                impressionTrackingUrl = multiAd.impressionTrackingUrl,
                uclid = multiAd.uclid,
                crt = multiAd.crt,
                elements = AdElement(
                    type = multiAd.elements.type,
                    value = multiAd.elements.value,
                    height = cardHeight,
                    width = cardWidth,
                    destinationUrl = multiAd.elements.destinationUrl
                ),
                adMetadata = multiAd.adMetadata,
                cliUbid = multiAd.cliUbid
            )
        }

        private fun getImageAdObject(multiAd: MultiAd, cardHeight: Int, cardWidth: Int): ImageAd {
            return ImageAd(
                clientId = multiAd.clientId,
                au = multiAd.au,
                rank = multiAd.rank,
                clickTrackingUrl = multiAd.clickTrackingUrl,
                impressionTrackingUrl = multiAd.impressionTrackingUrl,
                uclid = multiAd.uclid,
                crt = multiAd.crt,
                elements = AdElement(
                    type = multiAd.elements.type,
                    value = multiAd.elements.value,
                    height = cardHeight,
                    width = cardWidth,
                    destinationUrl = multiAd.elements.destinationUrl
                ),
                adMetadata = multiAd.adMetadata,
                cliUbid = multiAd.cliUbid
            )
        }
    }
}
