package com.ai.osmos.ads.renderer.loaders

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.ImageDecoder
import android.graphics.drawable.AnimatedImageDrawable
import android.os.Build
import android.view.Gravity
import android.view.View
import android.view.ViewTreeObserver
import android.widget.FrameLayout.LayoutParams
import android.widget.ImageView
import android.widget.LinearLayout
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.isVisibleInScreen
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.net.URL
import java.nio.ByteBuffer

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: ImageLoader
 */

/**
 * ImageLoader is responsible for asynchronously loading and displaying image ads,
 * including tracking impressions and handling click interactions.
 *
 * @param coroutineScope The CoroutineScope in which image loading and tracking operations are executed.
 * @param adTracker A utility class responsible for tracking ad impressions and clicks.
 */
class ImageLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface
) {

    private var viewLoadListener: ((ImageAd, String) -> Unit)? = null
    private var errorCallback: ErrorCallback? = null


    /**
     * Set an error callback to handle errors that occur during image loading.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
    }

    /**
     * Logs an error message using structured error handling.
     *
     * This function uses the ExceptionHandler pattern to log errors with proper error codes
     * and calls the error callback if one is set.
     *
     * @param message The error message to be logged, describing the issue encountered.
     * @param errorType The specific error type from the OsmosError enum.
     * @param throwable The original throwable that caused the error (optional).
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Creates and returns a configured [ImageView] with specified dimensions.
     *
     * The resulting ImageView:
     * - Uses FIT_CENTER scale type to maintain aspect ratio while fitting within bounds.
     * - Has layout parameters set using the given width and height, converted to pixels if needed.
     * - Is centered using [Gravity.CENTER].
     *
     * @param context The context used to create the ImageView.
     * @param widthDp The desired width of the ImageView in DP.
     * @param heightDp The desired height of the ImageView in DP.
     * @param isClamped Whether dimensions are already in pixels (true) or need DP to pixel conversion (false).
     * @return A new [ImageView] instance with the specified layout and scale settings.
     */
    internal fun createImageView(
        context: Context,
        widthDp: Int,
        heightDp: Int,
        isClamped: Boolean
    ): ImageView {
        val finalWidth = if (isClamped) widthDp else ViewUtils.dpToPx(context, widthDp)
        val finalHeight = if (isClamped) heightDp else ViewUtils.dpToPx(context, heightDp)

        return ImageView(context).apply {
            layoutParams = LayoutParams(
                finalWidth,
                finalHeight
            ).apply {
                gravity = Gravity.CENTER
            }
            scaleType = ImageView.ScaleType.FIT_CENTER
        }
    }


    /**
     * Loads an image ad (supports static images and GIFs), renders it into the given [ImageView],
     * and handles ad tracking and click interaction.
     *
     * This function supports:
     * - Static image formats (e.g., PNG, JPG)
     * - Animated GIFs via [AnimatedImageDrawable]
     *
     * SVG format is currently not supported.
     *
     * @param imageAd The [ImageAd] containing the image URL.
     * @param imageView The [ImageView] where the ad image will be displayed.
     * @param cliUbid The unique identifier for tracking ad impressions and clicks.
     * @param adClickListener Optional listener triggered when the ad is clicked.
     */
    internal fun loadImage(
        imageAd: ImageAd?,
        imageView: ImageView,
        cliUbid: String,
        isCarouselAd: Boolean? = false,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null
    ) {
        val url = imageAd?.elements?.value?.lowercase()
        coroutineScope.launch(Dispatchers.IO) {
            try {

                if (url != null) {
                    when {
                        url.endsWith(".svg") -> {
                            throw ExceptionHandler(OsmosError.UNSUPPORTED_IMAGE_FORMAT)
                        }

                        url.endsWith(".gif") -> {
                            URL(url).openStream().use { stream ->
                                val bytes = stream.readBytes()
                                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                                    val source = ImageDecoder.createSource(ByteBuffer.wrap(bytes))
                                    val drawable = ImageDecoder.decodeDrawable(source)
                                    if (drawable is AnimatedImageDrawable) {
                                        drawable.start()
                                    }

                                    withContext(Dispatchers.Main) {
                                        imageView.setImageDrawable(drawable)
                                        if (isCarouselAd == false) {
                                            handleAdLoaded(
                                                imageAd,
                                                imageView,
                                                cliUbid,
                                                adClickListener,
                                            )
                                        }
                                    }
                                }
                            }
                        }

                        else -> {
                            URL(url).openStream().use { stream ->
                                val bitmap: Bitmap = BitmapFactory.decodeStream(stream)

                                withContext(Dispatchers.Main) {
                                    imageView.setImageBitmap(bitmap)
                                    if (isCarouselAd == false)
                                        handleAdLoaded(
                                            imageAd,
                                            imageView,
                                            cliUbid,
                                            adClickListener
                                        )
                                }
                            }

                        }
                    }
                }
            } catch (e: Exception) {
                errorLog("Image load failed: ${e.message}", OsmosError.UNSUPPORTED_IMAGE_FORMAT, e)
            }
        }
    }

    internal fun loadNativeImage(
        imageUrl: String,
        imageView: ImageView,
    ) {
        val url = imageUrl.lowercase()
        coroutineScope.launch(Dispatchers.IO) {
            try {
                if (url.isNotEmpty()) {
                    when {
                        url.endsWith(".svg") -> {
                            throw ExceptionHandler(OsmosError.UNSUPPORTED_IMAGE_FORMAT)
                        }

                        url.endsWith(".gif") -> {
                            URL(url).openStream().use { stream ->
                                val bytes = stream.readBytes()
                                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                                    val source = ImageDecoder.createSource(ByteBuffer.wrap(bytes))
                                    val drawable = ImageDecoder.decodeDrawable(source)
                                    if (drawable is AnimatedImageDrawable) {
                                        drawable.start()
                                    }

                                    withContext(Dispatchers.Main) {
                                        imageView.setImageDrawable(drawable)
                                    }
                                }
                            }
                        }

                        else -> {
                            URL(url).openStream().use { stream ->
                                val bitmap: Bitmap = BitmapFactory.decodeStream(stream)

                                withContext(Dispatchers.Main) {
                                    imageView.setImageBitmap(bitmap)
                                }
                            }
                        }
                    }
                }
            } catch (e: Exception) {
                errorLog(
                    "Native image load failed: ${e.message}",
                    OsmosError.UNSUPPORTED_IMAGE_FORMAT,
                    e
                )
            }
        }
    }

    /**
     * Handles the post-load behavior of an image ad by:
     * - Notifying that the ad has loaded
     * - Tracking ad visibility for impression logging
     * - Setting up click interaction
     *
     * @param imageAd The ad data object.
     * @param imageView The view in which the ad is rendered.
     * @param cliUbid The unique identifier used for tracking this specific ad instance.
     * @param adClickListener Optional callback triggered when the ad is clicked.
     */
    private fun handleAdLoaded(
        imageAd: ImageAd,
        imageView: ImageView,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)?,
    ) {
        viewLoadListener?.invoke(imageAd, cliUbid) // Notify listener that the image is loaded
        trackAdVisibility(imageView, imageAd, cliUbid) // Track impression on visibility
        setupClickListener(imageView, imageAd, cliUbid, adClickListener) // Handle clicks
    }

    /**
     * Sets up a click listener on the ad's ImageView.
     * When clicked, it:
     * - Triggers the adClickListener callback with ad metadata.
     * - Tracks the ad click via the adTracker.
     *
     * @param imageView The ad's visual component.
     * @param imageAd The ad data.
     * @param cliUbid Unique identifier for tracking.
     * @param adClickListener Optional listener invoked on click, providing metadata.
     */
    private fun setupClickListener(
        imageView: ImageView,
        imageAd: ImageAd,
        cliUbid: String,
        adClickListener: ((adMetadata: Map<String, Any>?) -> Unit)?
    ) {
        imageView.setOnClickListener {
            adClickListener?.invoke(imageAd.toMap())
            adTracker.trackAdClick(imageAd.uclid, cliUbid)
        }
    }

    /**
     * Sets a listener to be notified when the ad view is fully loaded.
     *
     * @param listener Callback receiving the loaded [ImageAd] and its tracking ID.
     */
    fun setViewLoadListener(listener: (ImageAd, String) -> Unit) {
        this.viewLoadListener = { imageAd, cliUbid ->
            listener(imageAd, cliUbid)
        }
    }

    /**
     * Tracks ad visibility and triggers impression tracking when the view becomes visible.
     * Uses [ViewTreeObserver.OnPreDrawListener] to detect visibility.
     *
     * @param imageView The view representing the ad.
     * @param imageAd The ad being tracked.
     * @param cliUbid The unique tracking ID for this ad.
     */
    private fun trackAdVisibility(
        imageView: ImageView,
        imageAd: ImageAd,
        cliUbid: String
    ) {
        var preDrawListener: ViewTreeObserver.OnPreDrawListener? = null
        var attachStateListener: View.OnAttachStateChangeListener? = null

        preDrawListener = object : ViewTreeObserver.OnPreDrawListener {
            override fun onPreDraw(): Boolean {
                if (imageView.isVisibleInScreen()) {
                    trackImpression(imageAd.uclid, cliUbid)
                    imageView.viewTreeObserver.removeOnPreDrawListener(this)
                    attachStateListener?.let {
                        imageView.removeOnAttachStateChangeListener(it)
                    }
                }
                return true
            }
        }

        attachStateListener = object : View.OnAttachStateChangeListener {
            override fun onViewDetachedFromWindow(v: View) {
                preDrawListener?.let {
                    imageView.viewTreeObserver.removeOnPreDrawListener(it)
                }
                imageView.removeOnAttachStateChangeListener(this)
            }

            override fun onViewAttachedToWindow(v: View) {
                // No-op
            }
        }

        imageView.addOnAttachStateChangeListener(attachStateListener)
        imageView.viewTreeObserver.addOnPreDrawListener(preDrawListener)
    }

    /**
     * Helper method to track impressions using the common interface.
     */
    private fun trackImpression(uclid: String, cliUbid: String) {
        adTracker.trackImpression(uclid, cliUbid)
    }

    /**
     * Helper method to track ad clicks using the common interface.
     */
    private fun trackAdClick(uclid: String, cliUbid: String) {
        adTracker.trackAdClick(uclid, cliUbid)
    }

    /**
     * Clear the view load listener to prevent memory leaks
     */
    fun clearViewLoadListener() {
        viewLoadListener = null


    }
}

