package com.ai.osmos.ads.renderer.loaders

import android.content.Context
import android.graphics.Color
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.FrameLayout
import com.ai.osmos.ads.renderer.BaseNativeAdView
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.isVisibleInScreen
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: NativeAdLoader
 *
 * NativeAdLoader handles the rendering of native advertisements with proper error handling and cleanup.
 * Follows the same architecture pattern as PIPLoader and SliderLoader.
 */
internal class NativeAdLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface,
    private val config: Config
) {
    private val baseNativeAdView = BaseNativeAdView(coroutineScope, adTracker, config)
    private var errorCallback: ErrorCallback? = null
    private var viewLoadListener: ((adData: Map<String, Any>, String) -> Unit)? = null
    private var containers: MutableList<FrameLayout> = mutableListOf()
    private val preDrawListeners =
        mutableMapOf<FrameLayout, ViewTreeObserver.OnPreDrawListener>()

    /**
     * Set an error callback to handle errors that occur during native ad loading or rendering.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
        baseNativeAdView.setErrorCallback(callback)
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
        baseNativeAdView.clearErrorCallback()
    }

    /**
     * Set listener for view load events
     */
    fun setViewLoadListener(listener: (adData: Map<String, Any>, String) -> Unit) {
        this.viewLoadListener = listener
    }

    /**
     * Clear view load listener to prevent memory leaks.
     */
    fun clearViewLoadListener() {
        this.viewLoadListener = null
    }

    /**
     * Safe version that ensures cleanup never fails.
     */
    fun safeClearViewLoadListener() {
        this.viewLoadListener = null
    }

    /**
     * Logs an error message using structured error handling.
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Renders a native ad in vertical layout.
     */
    fun renderNativeAdVerticalView(
        context: Context,
        cliUbid: String,
        nativeAd: NativeAd,
        width: Int,
        height: Int?,
        customStyle: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        adLabelText: String?,
        adLabelAlignment: Int? = null,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)?,
        onViewLoadListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null
    ): View {
        return try {
            val container = FrameLayout(context)
            containers.add(container)

            val nativeView = baseNativeAdView.createVerticalAdView(
                context,
                nativeAd,
                width,
                height,
                adLabelText,
                adLabelAlignment,
                customStyle,
                customCtaView,
                customBadgeView,
                isCarouselView = false
            )

            container.addView(nativeView)

            setAdClickedListener(container, nativeAd, cliUbid, adClickedListener)
            setViewLoadListener(container, nativeAd, cliUbid, onViewLoadListener)

            container
        } catch (e: Exception) {
            errorLog(
                "Failed to render native ad vertical view: ${e.message}",
                OsmosError.UNKNOWN,
                e
            )
            FrameLayout(context)
        }
    }

    /**
     * Renders a native ad in horizontal layout.
     */
    fun renderNativeAdHorizontalView(
        context: Context,
        cliUbid: String,
        nativeAd: NativeAd,
        width: Int,
        height: Int?,
        customStyle: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)?,
        onViewLoadListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null
    ): View {
        return try {
            val container = FrameLayout(context)
            containers.add(container)

            val nativeView = baseNativeAdView.createHorizontalAdView(
                context,
                nativeAd,
                width,
                height,
                adLabelText,
                adLabelAlignment,
                customStyle,
                customCtaView,
                customBadgeView,
                isCarouselView = false
            )

            container.addView(nativeView)

            setAdClickedListener(container, nativeAd, cliUbid, adClickedListener)
            setViewLoadListener(container, nativeAd, cliUbid, onViewLoadListener)

            container
        } catch (e: Exception) {
            errorLog(
                "Failed to render native ad horizontal view: ${e.message}",
                OsmosError.UNKNOWN,
                e
            )
            FrameLayout(context)
        }
    }

    /**
     * Renders a native ad with custom content view.
     */
    fun renderNativeAdView(
        context: Context,
        cliUbid: String,
        nativeAd: NativeAd,
        contentView: View,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)?,
        onViewLoadListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null
    ): View {
        return try {
            val container = FrameLayout(context)
            containers.add(container)

            // Use createNativeAdView to properly wrap the contentView
            val nativeView = baseNativeAdView.createNativeAdView(context, contentView)

            container.addView(nativeView)

            setAdClickedListener(container, nativeAd, cliUbid, adClickedListener)
            setViewLoadListener(container, nativeAd, cliUbid, onViewLoadListener)

            container
        } catch (e: Exception) {
            errorLog("Failed to render native ad view: ${e.message}", OsmosError.UNKNOWN, e)
            FrameLayout(context)
        }
    }

    /**
     * Renders a native ad in carousel layout.
     */
    fun renderNativeAdCarouselView(
        context: Context,
        nativeAd: List<NativeAd>,
        adType: NativeAdStyle,
        width: Int,
        height: Int,
        cardElementWidth: Int? = null,
        customStyle: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)?,
        onViewLoadListener: ((adData: Map<String, Any>, cliUbid: String) -> Unit)? = null
    ): View {
        return try {
            val container = FrameLayout(context)
                .apply {
                    layoutParams = ViewGroup.LayoutParams(ViewGroup.LayoutParams.WRAP_CONTENT,
                        ViewGroup.LayoutParams.WRAP_CONTENT)
                }

            containers.add(container)

            val carouselView = baseNativeAdView.createCarouselAdView(
                context,
                nativeAd,
                adType,
                width,
                height,
                cardElementWidth,
                adLabelText,
                adLabelAlignment,
                customStyle,
                customCtaView,
                customBadgeView,
                adClickedListener,
                onViewLoadListener,
                baseNativeAdView = baseNativeAdView
            )

            container.addView(carouselView)
            container
        } catch (e: Exception) {
            errorLog(
                "Failed to render native ad carousel view: ${e.message}",
                OsmosError.UNKNOWN,
                e
            )
            FrameLayout(context)
        }
    }

    /**
     * Sets up view load listener with 50% viewport impression tracking.
     */
    private fun setViewLoadListener(
        container: FrameLayout,
        nativeAd: NativeAd,
        cliUbid: String,
        onViewLoadListener: ((adMetadata: Map<String, Any>?) -> Unit)?
    ) {
        try {
            lateinit var preDrawListener: ViewTreeObserver.OnPreDrawListener
            preDrawListener = ViewTreeObserver.OnPreDrawListener {
                try {
                    if (container.isVisibleInScreen()) {
                        onViewLoadListener?.invoke(nativeAd.toMap())
                        viewLoadListener?.invoke(nativeAd.toMap(), cliUbid)

                        if (nativeAd.uclid != null) {
                            adTracker.trackImpression(
                                uclid = nativeAd.uclid,
                                cliUbid = cliUbid
                            )
                        }

                        // Remove listener after first impression to prevent repeated calls
                        container.viewTreeObserver.removeOnPreDrawListener(preDrawListener)
                        preDrawListeners.remove(container)
                    }
                } catch (e: Exception) {
                    errorLog("Error in 50% viewport impression tracking: ${e.message}", OsmosError.UNKNOWN, e)
                    container.viewTreeObserver.removeOnPreDrawListener(preDrawListener)
                    preDrawListeners.remove(container)
                }
                true
            }

            preDrawListeners[container] = preDrawListener
            container.viewTreeObserver.addOnPreDrawListener(preDrawListener)
        } catch (e: Exception) {
            errorLog("Failed to set view load listener: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    /**
     * Sets up ad click listener with click tracking.
     */
    private fun setAdClickedListener(
        container: FrameLayout,
        nativeAd: NativeAd,
        cliUbid: String,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)?
    ) {
        try {
            container.setOnClickListener {
                adClickedListener?.invoke(nativeAd.toMap())
                if (nativeAd.uclid != null) {
                    adTracker.trackAdClick(nativeAd.uclid, cliUbid)
                }
            }
        } catch (e: Exception) {
            errorLog("Failed to set ad clicked listener: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    /**
     * Cleans up all resources to prevent memory leaks.
     * Made safe by using null checks and safe operations.
     */
    fun cleanUp() {
        // Remove ViewTreeObserver listeners to prevent memory leaks
        preDrawListeners.entries.removeAll { entry ->
            val container = entry.key
            val listener = entry.value
            container?.viewTreeObserver?.removeOnPreDrawListener(listener)
            true
        }

        // Clear all containers safely
        containers.removeAll { container ->
            container?.removeAllViews()
            container?.setOnClickListener(null)
            true
        }

        // Clean up BaseNativeAdView resources (critical for video player cleanup)
        baseNativeAdView.cleanUp()

        // Clear listeners
        this.viewLoadListener = null
        this.errorCallback = null
        baseNativeAdView.clearErrorCallback()
    }
}
