package com.ai.osmos.ads.renderer.loaders

import android.content.Context
import android.view.Gravity
import android.view.View
import android.view.ViewTreeObserver
import android.widget.FrameLayout
import com.ai.osmos.ads.views.BasePLAAdView
import com.ai.osmos.ads.views.style.PDAAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.PDAAd
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.isVisibleInScreen
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: NativePDAAdLoader
 *
 * NativePDAAdLoader handles the rendering of PDA (Product Display Ad) advertisements with proper error handling and cleanup.
 * Follows the same architecture pattern as NativeAdLoader.
 */
internal class NativePDAAdLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface,
    private val config: Config
) {
    private val basePLAAdView = BasePLAAdView(coroutineScope, adTracker, config)
    private var errorCallback: ErrorCallback? = null
    private var viewLoadListener: ((adData: Map<String, Any>, String) -> Unit)? = null
    private var containers: MutableList<FrameLayout> = mutableListOf()
    private val preDrawListeners =
        mutableMapOf<FrameLayout, ViewTreeObserver.OnPreDrawListener>()

    /**
     * Set an error callback to handle errors that occur during PDA ad loading or rendering.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
        basePLAAdView.setErrorCallback(callback)
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
        basePLAAdView.clearErrorCallback()
    }

    /**
     * Set a view load listener to be called when the ad view is loaded.
     *
     * @param listener Callback that receives ad data and cli_ubid when view is loaded.
     */
    fun setViewLoadListener(listener: ((adData: Map<String, Any>, String) -> Unit)?) {
        this.viewLoadListener = listener
    }

    /**
     * Clear the view load listener to prevent memory leaks.
     */
    fun clearViewLoadListener() {
        this.viewLoadListener = null
    }

    /**
     * Render a PDA ad view with proper error handling and tracking.
     *
     * @param context Application context
     * @param cliUbid Client unique browser ID for tracking
     * @param pdaAd PDA ad data object
     * @param adType Type of PDA ad (VERTICAL or HORIZONTAL)
     * @param width Width of the ad view
     * @param height Height of the ad view
     * @param pdaAdCustomStyle Custom styling configuration
     * @param adLabelText Optional ad label text
     * @param adLabelAlignment Optional ad label alignment
     * @param customBadgeView Optional custom badge view
     * @param adClickedListener Callback for ad click events
     * @param onViewLoadListener Callback for view load events
     * @return Rendered ad view
     */
    fun renderNativePDAAdView(
        context: Context,
        cliUbid: String,
        pdaAd: PDAAd,
        adType: NativeAdStyle,
        width: Int,
        height: Int?,
        pdaAdCustomStyle: PDAAdCustomStyle,
        adLabelText: String?,
        adLabelAlignment: Int?,
        customBadgeView: View?,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)?,
        onViewLoadListener: ((adData: Map<String, Any>, String) -> Unit)?
    ): View {
        return try {
            val container = FrameLayout(context)
            containers.add(container)

            when (adType) {
                NativeAdStyle.Vertical -> {
                    val bannerAdView = basePLAAdView.createVerticalAdView(
                        context,
                        pdaAd,
                        width,
                        height,
                        pdaAdCustomStyle,
                        adLabelText,
                        adLabelAlignment,
                        customBadgeView,
                        onViewLoadListener,
                        adClickedListener
                    )
                    container.addView(bannerAdView)
                }

                NativeAdStyle.Horizontal -> {
                    val bannerAdView = basePLAAdView.createHorizontalAdView(
                        context,
                        pdaAd,
                        width,
                        height,
                        pdaAdCustomStyle,
                        adLabelText,
                        adLabelAlignment,
                        customBadgeView,
                        onViewLoadListener,
                        adClickedListener
                    )
                    container.addView(bannerAdView)
                }
            }

            setAdClickedListener(container, pdaAd, cliUbid, adClickedListener)
            // Note: Container-level impression tracking removed - ImageLoader handles main banner impressions
            // Product-level impression tracking handled by BasePLAAdView's ProductListAdapter
            onViewLoadListener?.invoke(pdaAd.toMap(), cliUbid)

            container
        } catch (e: Exception) {
            handleError("Failed to render PDA ad view", e)
            FrameLayout(context) // Return empty container on error
        }
    }

    /**
     * Set up view load listener with 50% viewport impression tracking.
     */
    private fun setViewLoadListener(
        container: FrameLayout,
        pdaAd: PDAAd,
        cliUbid: String,
        onViewLoadListener: ((adData: Map<String, Any>, String) -> Unit)?
    ) {
        try {
            lateinit var preDrawListener: ViewTreeObserver.OnPreDrawListener
            preDrawListener = ViewTreeObserver.OnPreDrawListener {
                try {
                    if (container.isVisibleInScreen()) {
                        // Trigger external view load listener
                        onViewLoadListener?.invoke(pdaAd.toMap(), cliUbid)

                        // Trigger internal view load listener
                        viewLoadListener?.invoke(pdaAd.toMap(), cliUbid)

                        // Track impression when 50% visible
                        if (pdaAd.uclid.isNotEmpty()) {
                            adTracker.trackImpression(
                                uclid = pdaAd.uclid,
                                cliUbid = cliUbid
                            )
                        }

                        // Remove listener after first impression to prevent repeated calls
                        container.viewTreeObserver.removeOnPreDrawListener(preDrawListener)
                        preDrawListeners.remove(container)
                    }
                } catch (e: Exception) {
                    handleError("Error in 50% viewport impression tracking", e)
                    container.viewTreeObserver.removeOnPreDrawListener(preDrawListener)
                    preDrawListeners.remove(container)
                }
                true
            }

            preDrawListeners[container] = preDrawListener
            container.viewTreeObserver.addOnPreDrawListener(preDrawListener)
        } catch (e: Exception) {
            handleError("Failed to set view load listener", e)
        }
    }

    /**
     * Set up ad click listener with proper click tracking.
     */
    private fun setAdClickedListener(
        container: FrameLayout,
        pdaAd: PDAAd,
        cliUbid: String,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)?
    ) {
        try {
            container.setOnClickListener { _ ->
                try {
                    adClickedListener?.invoke(pdaAd.toMap())

                    if (pdaAd.uclid.isNotEmpty()) {
                        adTracker.trackAdClick(pdaAd.uclid, cliUbid)
                    }
                } catch (e: Exception) {
                    handleError("Failed to handle ad click event", e)
                }
            }
        } catch (e: Exception) {
            handleError("Failed to set ad click listener", e)
        }
    }

    /**
     * Handle errors with proper error callback invocation.
     */
    private fun handleError(message: String, throwable: Throwable?) {
        val error = OsmosError.PARSING_ERROR
        val exception =
            ExceptionHandler(error, "$message: ${throwable?.message ?: "Unknown error"}", throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Clean up resources to prevent memory leaks.
     */
    fun cleanUp() {
        try {
            // Remove all PreDraw listeners
            preDrawListeners.forEach { (container, listener) ->
                try {
                    container.viewTreeObserver.removeOnPreDrawListener(listener)
                } catch (e: Exception) {
                    // Ignore cleanup errors
                }
            }
            preDrawListeners.clear()

            // Clear containers
            containers.forEach { container ->
                try {
                    container.removeAllViews()
                } catch (e: Exception) {
                    // Ignore cleanup errors
                }
            }
            containers.clear()

            // Clear callbacks
            viewLoadListener = null
            errorCallback = null

            // Clean up base PLA ad view
            basePLAAdView.cleanUp()
        } catch (e: Exception) {
            // Ignore cleanup errors to prevent crashes during cleanup
        }
    }
}