package com.ai.osmos.ads.renderer.loaders

import android.app.Activity
import android.content.Context
import android.graphics.Color
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.lifecycle.LifecycleOwner
import androidx.media3.exoplayer.ExoPlayer
import com.ai.osmos.ads.renderer.PIPPlayerLifecycleObserver
import com.ai.osmos.ads.views.Coordinates
import com.ai.osmos.ads.views.InAppPIPView
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope
import java.lang.ref.WeakReference

internal class SliderLoader(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface
) {
    private var sliderContainer: InAppPIPView? = null
    private var currentExoPlayer: ExoPlayer? = null
    private var videoView: View? = null
    private var isMuted: Boolean = true
    private var errorCallback: ErrorCallback? = null
    private var currentActivity: Activity? = null
    private var currentLifecycleObserver: PIPPlayerLifecycleObserver? = null

    private val videoLoader = VideoLoader(coroutineScope, adTracker)
    private val imageLoader = ImageLoader(coroutineScope, adTracker)

    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
        videoLoader.setErrorCallback(callback)
        imageLoader.setErrorCallback(callback)
    }

    fun clearErrorCallback() {
        this.errorCallback = null
        videoLoader.clearErrorCallback()
        imageLoader.clearErrorCallback()
    }

    fun showVideoSlider(
        activity: Activity,
        videoAd: VideoAd,
        cliUbid: String,
        width: Int,
        height: Int,
        coordinates: Coordinates? = Coordinates(0, 0),
        adLabelText: String?,
        adLabelAlignment: Int?,
        isClamped: Boolean,
        adClickListener: ((Map<String, Any>?) -> Unit)?,
        onViewLoadListener: ((Map<String, Any>, String) -> Unit)?
    ) {
        try {
            // Clean up existing resources
            cleanUp()

            // Create individual ExoPlayer for this slider ad
            val exoPlayer = createPlayer(activity)
            currentExoPlayer = exoPlayer

            val finalWidth = if (isClamped) width else ViewUtils.dpToPx(activity, width)
            val finalHeight = if (isClamped) height else ViewUtils.dpToPx(activity, height)

            videoView = videoLoader.createPIPVideoView(
                context = activity,
                videoAd = videoAd,
                widthDp = finalWidth,
                heightDp = finalHeight,
                cliUbid = cliUbid,
                adClickListener = { adData ->
                    adClickListener?.invoke(adData)
                },
                exoPlayer = exoPlayer,
            )

            val sliderView = InAppPIPView(
                context = activity,
                cliUbid = cliUbid,
                adTracker = adTracker,
                height = finalHeight,
                width = finalWidth,
                isSliderAd = true,
                imageAd = null,
                videoAd = videoAd,
                exoPlayer = exoPlayer,
                coordinates = coordinates,
                adLabelText = adLabelText,
                adLabelAlignment = adLabelAlignment,
                errorCallback = errorCallback
            ).apply {
                layoutParams = FrameLayout.LayoutParams(finalWidth, finalHeight).apply {
                    if (coordinates != null && (coordinates.x != 0 || coordinates.y != 0)) {
                        gravity = Gravity.TOP or Gravity.START
                        x = coordinates.x.toFloat()
                        y = coordinates.y.toFloat()
                    } else {
                        gravity = Gravity.BOTTOM or Gravity.END
                    }
                    setMargins(50, 50, 50, 50)
                }

                setVideoView(videoView!!)
                setOnCloseClick {
                    cleanUp()
                }
                setOnMuteToggleClick {
                    isMuted = !isMuted
                    updateMuteState()
                }
                setOnFullscreenClick {
                    // You can implement a showFullscreen() if needed (like in PIPLoader)
                }
            }.apply {
                setBackgroundColor(Color.BLACK)
            }

            sliderContainer = sliderView

            val decor = activity.window.decorView as ViewGroup
            decor.addView(sliderView)
            onViewLoadListener?.invoke(videoAd.toMap(), videoAd.cliUbid.toString())

            // Add lifecycle observer with proper cleanup tracking
            if (activity is LifecycleOwner) {
                // Remove existing observer if any
                currentLifecycleObserver?.let { observer ->
                    currentActivity?.let { oldActivity ->
                        if (oldActivity is LifecycleOwner) {
                            oldActivity.lifecycle.removeObserver(observer)
                        }
                    }
                }

                // Create and add new observer with WeakReference
                val activityRef = WeakReference(activity)
                val lifecycleObserver = PIPPlayerLifecycleObserver(exoPlayer, errorCallback)
                activityRef.get()?.lifecycle?.addObserver(lifecycleObserver)
                if (activityRef.get() != null) {
                    currentLifecycleObserver = lifecycleObserver
                    currentActivity = activity
                }
            }

        } catch (e: Exception) {
            errorLog("Failed to show video slider: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    fun showImageSlider(
        activity: Activity,
        imageAd: ImageAd,
        cliUbid: String,
        width: Int,
        height: Int,
        coordinates: Coordinates? = Coordinates(0, 0),
        adLabelText: String?,
        adLabelAlignment: Int?,
        isClamped: Boolean,
        adClickListener: ((Map<String, Any>?) -> Unit)?
    ) {
        try {
            cleanUp()
            val finalWidth = if (isClamped) width else ViewUtils.dpToPx(activity, width)
            val finalHeight = if (isClamped) height else ViewUtils.dpToPx(activity, height)

            val imageView =
                imageLoader.createImageView(activity, finalWidth, finalHeight, isClamped)

            imageLoader.loadImage(imageAd, imageView, cliUbid, true) {
                adClickListener?.invoke(imageAd.toMap())
            }

            val sliderView = InAppPIPView(
                context = activity,
                cliUbid = cliUbid,
                adTracker = adTracker,
                height = finalHeight,
                width = finalWidth,
                isSliderAd = true,
                imageAd = imageAd,
                videoAd = null,
                exoPlayer = null,
                coordinates = coordinates,
                adLabelText = adLabelText,
                adLabelAlignment = adLabelAlignment,
                errorCallback = errorCallback
            ).apply {
                layoutParams = FrameLayout.LayoutParams(finalWidth, finalHeight).apply {
                    if (coordinates != null && (coordinates.x != 0 || coordinates.y != 0)) {
                        gravity = Gravity.TOP or Gravity.START
                        x = coordinates.x.toFloat()
                        y = coordinates.y.toFloat()
                    } else {
                        gravity = Gravity.BOTTOM or Gravity.END
                    }
                    setMargins(50, 50, 50, 50)
                }

                setImageView(imageView)
                setOnCloseClick {
                    cleanUp()
                }
            }.apply {
                setBackgroundColor(Color.BLACK)
            }

            sliderContainer = sliderView

            val decor = activity.window.decorView as ViewGroup
            decor.addView(sliderView)

        } catch (e: Exception) {
            errorLog("Failed to show image slider: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    private fun updateMuteState() {
        currentExoPlayer?.let { player ->
            try {
                player.volume = if (isMuted) 0f else 1f
            } catch (e: Exception) {
                errorLog("Failed to set mute state: ${e.message}", OsmosError.UNKNOWN, e)
            }
        }
    }

    private fun createPlayer(context: Context): ExoPlayer {
        return ExoPlayer.Builder(context.applicationContext).build()
    }

    fun cleanUp() {
        try {

            // Remove lifecycle observer
            currentLifecycleObserver?.let { observer ->
                observer.clearErrorCallback()
                currentActivity?.let { activity ->
                    if (activity is LifecycleOwner) {
                        try {
                            activity.lifecycle.removeObserver(observer)
                        } catch (e: Exception) {
                            errorLog(
                                "Error removing lifecycle observer: ${e.message}",
                                OsmosError.UNKNOWN,
                                e
                            )
                        }
                    }
                }
            }
            currentLifecycleObserver = null
            currentActivity = null

            // Release ExoPlayer
            currentExoPlayer?.let { player ->
                try {
                    player.stop()
                    player.clearVideoSurface()
                    player.setVideoSurfaceView(null)
                    player.setVideoTextureView(null)
                    player.release()
                } catch (e: Exception) {
                    errorLog("Error releasing ExoPlayer: ${e.message}", OsmosError.UNKNOWN, e)
                }
            }
            currentExoPlayer = null

            // Clean up container
            sliderContainer?.let { container ->
                (container.parent as? ViewGroup)?.removeView(container)
                container.release()
            }
            sliderContainer = null

            // Clear video view reference
            videoView = null

        } catch (e: Exception) {
            errorLog("Error during cleanup: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }

    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }
}
