package com.ai.osmos.ads.views

import android.app.Activity
import android.content.Context
import android.content.ContextWrapper
import android.content.res.Resources
import android.util.AttributeSet
import android.widget.FrameLayout
import androidx.lifecycle.LifecycleOwner
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.logging.DebugLogger
import com.ai.osmos.utils.ui.ViewUtils
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancelChildren

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: BaseAdView
 */

/**
 * Abstract base class for rendering ads inside a custom view.
 * Provides shared functionality for all ad types (Banner, Popup, etc.)
 *
 * @param context The context of the view, usually the Activity.
 * @param attrs Optional XML attributes if the view is used in layout XML.
 */
abstract class BaseAdView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    private val config: Config
) : FrameLayout(context, attrs) {

    private enum class MediaType {
        IMAGE, VIDEO
    }

    private val supervisorJob = SupervisorJob()
    protected val coroutineScope = CoroutineScope(
        Dispatchers.Main + supervisorJob + CoroutineExceptionHandler() { _, throwable ->
            errorLog("Coroutine exception: ${throwable.message}", OsmosError.UNKNOWN)
        }
    )

    // Enhanced ad tracker with persistent impression tracking
    protected val adTracker: AdTrackerInterface by lazy {
        if (context.applicationContext != null) {
            PersistentAdTracker.create(context.applicationContext, coroutineScope, config)
        } else {
            // Fallback for edge cases where applicationContext is not available
            PersistentAdTracker.create(context, coroutineScope, config)
        }
    }

    @Volatile
    protected var adClickedListener: ((adMetadata: Map<String, Any>?) -> Unit)? = null

    // Callback for when the ad view has been successfully loaded (specific to ImageAd)
    protected var onViewLoadListener: ((adData: Map<String, Any>, cliUbid: String) -> Unit)? = null

    @Volatile
    protected var itemClickedListener: ((adData: Map<String, Any>, clickedItem: Map<String, Any>, destinationUrl: String) -> Unit)? =
        null

    @Volatile
    private var errorCallback: ErrorCallback? = null

    /**
     * Set a click listener for the ad.
     *
     * @param listener Callback that provides ad metadata when the ad is clicked.
     */
    fun setAdClickListener(listener: (adMetadata: Map<String, Any>?) -> Unit) {
        this.adClickedListener = listener
    }

    fun setAdItemClickListener(listener: (adData: Map<String, Any>, clickedItem: Map<String, Any>, destinationUrl: String) -> Unit) {
        this.itemClickedListener = listener
    }

    /**
     * Set a view load listener that gets triggered when an ad is fully loaded.
     *
     * @param listener Callback that returns the loaded ImageAd and tracking ID (cli_ubid).
     */
    fun setViewLoadListener(listener: (adData: Map<String, Any>, cliUbid: String) -> Unit) {
        this.onViewLoadListener = listener
    }

    /**
     * Set an error callback to handle errors that occur during ad loading or processing.
     *
     * @param callback Callback that handles error scenarios with structured error information.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
    }

    /**
     * Clear the error callback to prevent memory leaks when view is detached.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
    }

    /**
     * Abstract method to be implemented by subclasses for loading ads from given data.
     *
     * @param adData A map containing the ad object and related metadata.
     */
    abstract fun loadAd(adData: Map<String, Any>, context: Context)

    abstract fun loadAd(adData: Map<String, Any>, activity: Activity?)

    internal fun extractActivity(context: Context): Activity? {
        var mContext = context
        while (mContext is ContextWrapper) {
            if (mContext is Activity) return mContext
            mContext = mContext.baseContext
        }
        return null
    }

    /**
     * Resolves the final width and height to use for the ad view.
     * Priority: requested size → fallback from ad → hardcoded fallback.
     *
     * @param requestedWidth Optional width provided by caller (in dp).
     * @param requestedHeight Optional height provided by caller (in dp).
     * @param fallbackWidth Width from ad metadata (if present).
     * @param fallbackHeight Height from ad metadata (if present).
     *
     * @return A pair of width and height in dp.
     */
    protected fun getAdDimensions(
        requestedWidth: Int?,
        requestedHeight: Int?,
        fallbackWidth: Int,
        fallbackHeight: Int
    ): Pair<Int, Int> {
        val finalWidth = requestedWidth ?: fallbackWidth
        val finalHeight = requestedHeight ?: fallbackHeight
        return Pair(finalWidth, finalHeight)
    }

    protected fun getScreenHeightWidth(): Pair<Int, Int> {

        val screenWidth = Resources.getSystem().displayMetrics.widthPixels
        val screenHeight = Resources.getSystem().displayMetrics.heightPixels

        val maxAdWidth = (screenWidth * 0.8f).toInt()
        val maxAdHeight = (screenHeight * 0.8f).toInt()

        return Pair(maxAdWidth, maxAdHeight)
    }

    data class LayoutResult(
        val finalWidth: Int,
        val finalHeight: Int,
        val coordinates: Coordinates?,
        val isClamped: Boolean
    )

    protected fun setMaxHeightWidth(
        height: Int,
        width: Int,
        inputCoordinates: Coordinates?,
    ): LayoutResult {

        val screenWidth = Resources.getSystem().displayMetrics.widthPixels
        val screenHeight = Resources.getSystem().displayMetrics.heightPixels

        val (maxAdWidth, maxAdHeight) = ViewUtils.getScreenHeightWidth(0.8f, 0.8f)

        val finalWidth = minOf(width, maxAdWidth)
        val finalHeight = minOf(height, maxAdHeight)

        val maxX = screenWidth - finalWidth - 16
        val maxY = screenHeight - finalHeight - 34

        val finalX = inputCoordinates?.x?.coerceAtMost(maxX)
        val finalY = inputCoordinates?.y?.coerceAtMost(maxY)

        val clamped = (width >= screenWidth || height >= screenHeight)

        val coordinates = if (inputCoordinates != null && finalX != null && finalY != null) {
            Coordinates(finalX, finalY)
        } else null
        return LayoutResult(
            finalWidth = finalWidth,
            finalHeight = finalHeight,
            coordinates = coordinates,
            isClamped = clamped
        )
    }

    /**
     * Checks if the image format of the given ad is supported.
     *
     * This function verifies whether the ad is an instance of [ImageAd], and if so,
     * it checks the file extension of the URL to ensure it's a supported image format.
     * If the ad is not an [ImageAd], it returns `true` by default, assuming the format is supported.
     *
     * @param ad The ad to check. It must be of type [ImageAd] for this function to verify the image format.
     * @return `true` if the image format is supported, `false` if not, or `true` for non-ImageAds.
     */
    internal fun isSupportedImageFormat(ad: BaseAd?): Boolean {
        if (ad is ImageAd) {
            val url = ad.elements.value.trim().lowercase()
            val extension = url.substringAfterLast('.', "").substringBefore('?').lowercase()

            return isValidMediaUrl(url, extension, MediaType.IMAGE)
        }
        return true
    }

    /**
     * Checks if the video format of the given ad is supported.
     *
     * This function validates if the ad is a `VideoAd`, and if so, it performs checks to verify
     * whether the necessary dependencies for ExoPlayer and the required video formats (e.g., MP4, HLS, DASH) are available.
     *
     * @param ad The ad to check. It must be of type [VideoAd] for this function to verify the video format.
     * @return `true` if the video format is supported, `false` otherwise.
     */
    internal fun isSupportedVideoFormat(ad: BaseAd?): Boolean {
        if (ad is VideoAd) {
            // Check if core ExoPlayer + UI are present
            if (!isExoPlayerAvailable()) {
                errorLog(
                    OsmosError.EXOPLAYER_CORE_MISSING.message,
                    OsmosError.EXOPLAYER_CORE_MISSING
                )
                return false
            }

            if (!isExoPlayerUiAvailable()) {
                errorLog(OsmosError.EXOPLAYER_UI_MISSING.message, OsmosError.EXOPLAYER_UI_MISSING)
                return false
            }

            //Check for unsupported video format
            val url = ad.elements.value.trim().lowercase()
            val extension = url.substringAfterLast('.', "").substringBefore('?')
            val isValid = isValidMediaUrl(url, extension, MediaType.VIDEO)

            if (isValid) {
                return when {
                    extension in listOf("mp4", "webm", "mov", "m4v", "mkv", "wmv") -> true
                    extension == "m3u8" -> {
                        if (isHlsAvailable()) {
                            true
                        } else {
                            errorLog(
                                OsmosError.HLS_SUPPORT_MISSING.message,
                                OsmosError.HLS_SUPPORT_MISSING
                            )
                            false
                        }
                    }

                    extension == "mpd" -> {
                        if (isDashAvailable()) {
                            true
                        } else {
                            errorLog(
                                OsmosError.DASH_SUPPORT_MISSING.message,
                                OsmosError.DASH_SUPPORT_MISSING
                            )
                            false
                        }
                    }

                    else -> {
                        errorLog(
                            OsmosError.UNSUPPORTED_VIDEO_FORMAT.message,
                            OsmosError.UNSUPPORTED_VIDEO_FORMAT
                        )
                        false
                    }
                }
            } else {
                errorLog(
                    OsmosError.UNSUPPORTED_VIDEO_FORMAT.message,
                    OsmosError.UNSUPPORTED_VIDEO_FORMAT
                )
                return false
            }
        }
        return true
    }

    /**
     * Validates the given media URL asynchronously.
     *
     * This function uses a suspending coroutine to wrap the callback-based `isValidUrl` function
     * so that it can be used within Kotlin coroutines.
     *
     * It suspends the execution of the calling coroutine until the result of the URL validation is available,
     * and then resumes with a `true` or `false` value indicating whether the media URL is valid.
     *
     * @param url The URL to validate.
     * @param extension The file extension of the URL (e.g., ".mp4", ".jpg").
     * @param mediaType The type of media (IMAGE or VIDEO).
     * @return A `Boolean` indicating if the URL is valid or not. This is returned as a suspending function.
     */
    private fun isValidMediaUrl(
        url: String,
        extension: String,
        mediaType: MediaType
    ): Boolean {
        val supportedImageExtensions = listOf("png", "jpg", "jpeg", "gif")
        val supportedVideoExtensions =
            listOf("mp4", "webm", "mov", "m4v", "mkv", "wmv", "mpd", "m3u8")

        val supportedExtensions = when (mediaType) {
            MediaType.IMAGE -> supportedImageExtensions
            MediaType.VIDEO -> supportedVideoExtensions
        }
        if (extension !in supportedExtensions) {
            val errorType = when (mediaType) {
                MediaType.IMAGE -> OsmosError.UNSUPPORTED_IMAGE_FORMAT
                MediaType.VIDEO -> OsmosError.UNSUPPORTED_VIDEO_FORMAT
            }
            errorLog(errorType.message, errorType)
            return false
        }
        return true
    }

    /**
     * Logs an error message using structured error handling.
     *
     * This function uses the ExceptionHandler pattern to log errors with proper error codes
     * and calls the error callback if one is set.
     *
     * @param message The error message to be logged, describing the issue encountered.
     * @param errorType The specific error type from the Error enum (defaults to CONFIGURATION).
     */
    private fun errorLog(message: String, errorType: OsmosError = OsmosError.CONFIGURATION_ERROR) {
        val exception = ExceptionHandler(errorType, message)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    /**
     * Checks if the core ExoPlayer library is available in the classpath.
     *
     * @return true if 'androidx.media3.exoplayer.ExoPlayer' class is found; false otherwise.
     */
    private fun isExoPlayerAvailable(): Boolean {
        return try {
            Class.forName("androidx.media3.exoplayer.ExoPlayer")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Checks if the ExoPlayer UI module is available in the classpath.
     *
     * @return true if 'androidx.media3.ui.PlayerView' class is found; false otherwise.
     */
    private fun isExoPlayerUiAvailable(): Boolean {
        return try {
            Class.forName("androidx.media3.ui.PlayerView")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Checks if the HLS (HTTP Live Streaming) support is available.
     *
     * @return true if 'androidx.media3.exoplayer.hls.HlsMediaSource' is found; false otherwise.
     */
    private fun isHlsAvailable(): Boolean {
        return try {
            Class.forName("androidx.media3.exoplayer.hls.HlsMediaSource")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Checks if the DASH (Dynamic Adaptive Streaming over HTTP) support is available.
     *
     * @return true if 'androidx.media3.exoplayer.dash.DashMediaSource' is found; false otherwise.
     */
    private fun isDashAvailable(): Boolean {
        return try {
            Class.forName("androidx.media3.exoplayer.dash.DashMediaSource")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Checks if the RecyclerView library is available in the classpath.
     *
     * @return true if 'androidx.recyclerview.widget.RecyclerView' class is found; false otherwise.
     */
    internal fun isRecyclerViewAvailable(): Boolean {
        return try {
            Class.forName("androidx.recyclerview.widget.RecyclerView")
            true
        } catch (e: ClassNotFoundException) {
            false
        }
    }

    /**
     * Retrieves the client-side tracking ID from the ad.
     *
     * @param ad The ad from which to extract the tracking ID.
     * @return The cli_ubid string, or an empty string if null.
     */
    internal fun getCliUbid(ad: BaseAd?): String = ad?.cliUbid ?: ""

    /**
     * Displays a fallback empty view if no ad is available or loading fails.
     */
    internal fun showEmptyView(context: Context) {
        removeAllViews()
        addView(ViewUtils.getEmptyView(context))
    }

    /**
     * Checks if the creative type (crt) is supported by the SDK.
     *
     * @param crt Creative type string from the ad metadata.
     * @return True if supported (i.e., starts with "osmos_sdk/image/v1"), false otherwise.
     */
    protected fun isSupportedCrtImage(crt: String): Boolean {
        return crt.startsWith(Constants.CRT_BANNER_IMAGE)
    }

    protected fun isSupportedCrtVideo(crt: String): Boolean {
        return crt.startsWith(Constants.CRT_BANNER_VIDEO)
    }

    protected fun isSupportedCrtCarousel(crt: String): Boolean {
        return crt.startsWith(Constants.CRT_CAROUSEL)
    }

    /**
     * Override onDetachedFromWindow to properly clean up resources and prevent memory leaks
     */
    override fun onDetachedFromWindow() {
        try {
            if (context !is LifecycleOwner) {
                // Synchronized job cancellation to prevent race conditions
                synchronized(supervisorJob) {
                    if (supervisorJob.isActive) {
                        try {
                            supervisorJob.cancelChildren()
                            supervisorJob.cancel()
                        } catch (e: IllegalStateException) {
                            // Job already cancelled - this is fine
                            DebugLogger.log("BaseAdView Job already cancelled during cleanup")
                        }
                    }
                }
            }
            clearAllListeners()
        } catch (e: Exception) {
            DebugLogger.log("BaseAdView Error during cleanup")
            // Don't swallow exceptions - log them
        } finally {
            super.onDetachedFromWindow()
        }
    }


    /**
     * Clear all callback listeners to prevent memory leaks
     */
    private fun clearAllListeners() {
        synchronized(this) {
            adClickedListener = null
            onViewLoadListener = null
            itemClickedListener = null
            errorCallback = null
        }
    }

    protected fun isSupportedFormatPDA(crt: String): Boolean {
        return crt.startsWith("osmos_pda_banner/") ||
                crt.startsWith("osmos_pda_video/") ||
                crt.startsWith("osmos_pda_carousel/")
    }
}
