package com.ai.osmos.ads.renderer

import android.content.Context
import android.graphics.Color
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.LinearLayout
import android.widget.TextView
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.ai.osmos.ads.views.adapters.NativeAdCarouselAdapter
import com.ai.osmos.ads.views.helpers.NativeAdViewFactory
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.enums.NativeAdLayoutType
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import kotlinx.coroutines.CoroutineScope

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: BaseNativeAdView
 */

/**
 * Refactored BaseNativeAdView that follows Single Responsibility Principle.
 * This class acts as a coordinator that delegates rendering to specialized renderers
 * through the Strategy pattern via NativeAdViewFactory.
 *
 * All public method signatures remain exactly the same as before refactoring.
 */
internal class BaseNativeAdView(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface,
    private val config: Config
) {
    private val rendererFactory = NativeAdViewFactory(coroutineScope, adTracker, config)
    private var errorCallback: ErrorCallback? = null

    /**
     * Set an error callback to handle errors that occur during ad view creation.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
        rendererFactory.setErrorCallback(callback)
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
        rendererFactory.clearErrorCallbacks()
    }

    /**
     * Logs an error message using structured error handling.
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    fun createVerticalAdView(
        context: Context,
        adData: NativeAd,
        width: Int,
        height: Int?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        isCarouselView: Boolean
    ): View {
        val renderer = rendererFactory.getVerticalRenderer()
        return renderer.renderVerticalView(
            context = context,
            adData = adData,
            width = width,
            height = height,
            adLabelText = adLabelText,
            adLabelAlignment = adLabelAlignment,
            style = style,
            customCtaView = customCtaView,
            customBadgeView = customBadgeView,
            isCarouselView = isCarouselView
        )
    }

    fun createHorizontalAdView(
        context: Context,
        adData: NativeAd,
        width: Int,
        height: Int?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        isCarouselView: Boolean
    ): View {
        val renderer = rendererFactory.getHorizontalRenderer()
        return renderer.renderHorizontalView(
            context = context,
            adData = adData,
            width = width,
            height = height,
            adLabelText = adLabelText,
            adLabelAlignment = adLabelAlignment,
            style = style,
            customCtaView = customCtaView,
            customBadgeView = customBadgeView,
            isCarouselView = isCarouselView
        )
    }

    fun createNativeAdView(
        context: Context,
        contentView: View,
    ): View {

        val container = FrameLayout(context).apply {
            layoutParams = ViewGroup.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
        }
        container.addView(contentView)
        return container
    }

    fun createProductVerticalAdView(
        context: Context,
        adData: NativeAd,
        adType: NativeAdLayoutType,
        width: Int,
        height: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?
    ): View {
        val renderer = rendererFactory.getVerticalRenderer()
        return renderer.renderProductVerticalView(
            context = context,
            adData = adData,
            adType = adType,
            width = width,
            height = height,
            style = style,
            customCtaView = customCtaView
        )
    }

    fun createProductHorizontalAdView(
        context: Context,
        adData: NativeAd,
        adType: NativeAdLayoutType,
        width: Int,
        height: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?
    ): View {
        val renderer = rendererFactory.getHorizontalRenderer()
        return renderer.renderProductHorizontalView(
            context = context,
            adData = adData,
            adType = adType,
            width = width,
            height = height,
            style = style,
            customCtaView = customCtaView
        )
    }

    fun createCarouselAdView(
        context: Context,
        nativeAdList: List<NativeAd>,
        adType: NativeAdStyle,
        width: Int,
        height: Int,
        cardElementWidth: Int? = null,
        adLabelText: String?,
        adLabelAlignment: Int?,
        customStyle: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)?,
        onViewLoadListener: ((adData: Map<String, Any>, cliUbid: String) -> Unit)? = null,
        baseNativeAdView: BaseNativeAdView
    ): View {
        return try {
            val (screenWidth, screenHeight) = ViewUtils.getScreenHeightWidth(0.8f, 0.8f)

            val finalWidth = minOf(ViewUtils.dpToPx(context, width), screenWidth)
            val finalHeight = minOf(ViewUtils.dpToPx(context, height), screenHeight)

            // Use LinearLayout as main container to stack carousel and dots vertically
            val mainContainer = LinearLayout(context).apply {
                orientation = LinearLayout.VERTICAL
                layoutParams = ViewGroup.LayoutParams(
                    finalWidth,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                )
            }

            // Create dot indicator container
            val dotIndicatorContainer = LinearLayout(context).apply {
                orientation = LinearLayout.HORIZONTAL
                gravity = Gravity.CENTER
                layoutParams = LinearLayout.LayoutParams(
                    LinearLayout.LayoutParams.MATCH_PARENT,
                    LinearLayout.LayoutParams.WRAP_CONTENT
                ).apply {
                    topMargin = ViewUtils.dpToPx(context, 6)
                    bottomMargin = ViewUtils.dpToPx(context, 2)
                }
            }

            val carouselRecyclerView = RecyclerView(context).apply {
                layoutManager = LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false)
                layoutParams = LinearLayout.LayoutParams(
                    LinearLayout.LayoutParams.MATCH_PARENT,
                    LinearLayout.LayoutParams.WRAP_CONTENT
                )

                // Create adapter for native ads carousel
                val carouselAdapter = NativeAdCarouselAdapter(
                    context = context,
                    adType = adType,
                    width = finalWidth,
                    height = height,
                    cardElementWidth = cardElementWidth,
                    nativeAdList = nativeAdList,
                    adLabelText = adLabelText,
                    adLabelAlignment = adLabelAlignment,
                    customStyle = customStyle,
                    customCtaView = customCtaView,
                    customBadgeView = customBadgeView,
                    adClickedListener = adClickedListener,
                    onViewLoadListener = onViewLoadListener,
                    baseNativeAdView = baseNativeAdView,
                    dotIndicatorContainer = dotIndicatorContainer,
                    trackImpression = { uclid, cliUbid -> adTracker.trackImpression(uclid, cliUbid) }
                )
                adapter = carouselAdapter

                // Setup scroll listener for dot indicator updates
                carouselAdapter.setupScrollListener(this)
            }

            mainContainer.addView(carouselRecyclerView)
            mainContainer.addView(dotIndicatorContainer)

            mainContainer
        } catch (e: Exception) {
            errorLog("Failed to create carousel ad view: ${e.message}", OsmosError.UNKNOWN, e)
            FrameLayout(context)
        }
    }

    private fun ViewGroup.addViewsInOrder(vararg views: View?) {
        views.forEach { view ->
            view?.let { this.addView(it) }
        }
    }

    /**
     * Cleans up all resources to prevent memory leaks.
     * Delegates cleanup to the renderer factory which handles all renderers.
     */
    fun cleanUp() {
        try {
            // Clear error callbacks to prevent memory leaks
            rendererFactory.clearErrorCallbacks()
            clearErrorCallback()

        } catch (e: Exception) {
            errorLog("Error during cleanup: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }
}
