package com.ai.osmos.ads.views

import android.content.Context
import android.graphics.Color
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.media3.ui.PlayerView
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.ai.osmos.ads.renderer.BaseNativeAdView
import com.ai.osmos.ads.renderer.loaders.ImageLoader
import com.ai.osmos.ads.renderer.loaders.VideoLoader
import com.ai.osmos.ads.views.PDAProductStyle.Custom
import com.ai.osmos.ads.views.PDAProductStyle.Horizontal
import com.ai.osmos.ads.views.PDAProductStyle.Vertical
import com.ai.osmos.ads.views.style.PDAAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.AdElement
import com.ai.osmos.models.ads.CarouselAdElement
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.ads.PDAAd
import com.ai.osmos.models.ads.ProductItem
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.isVisibleInScreen
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.CoroutineScope

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: BasePLAAdView
 *
 * BasePLAAdView handles the creation and rendering of PDA (Product Display Ad) ad views.
 * This class provides methods for creating vertical and horizontal PDA ad layouts
 * with proper error handling and memory management.
 * Follows the same architecture pattern as BaseNativeAdView.
 */
internal class BasePLAAdView(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface,
    private val config: Config
) {
    private val baseNativeAdView = BaseNativeAdView(coroutineScope, adTracker, config)
    private val imageLoader = ImageLoader(coroutineScope, adTracker)
    private val videoLoader = VideoLoader(coroutineScope, adTracker)
    private var errorCallback: ErrorCallback? = null
    private lateinit var productListRecyclerView: RecyclerView
    private var currentIndex = -1
    private var dotsContainer: LinearLayout? = null

    /**
     * Set an error callback to handle errors that occur during PDA ad view creation.
     */
    fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
        baseNativeAdView.setErrorCallback(callback)
        imageLoader.setErrorCallback(callback)
        videoLoader.setErrorCallback(callback)
    }

    /**
     * Clear the error callback to prevent memory leaks.
     */
    fun clearErrorCallback() {
        this.errorCallback = null
        baseNativeAdView.clearErrorCallback()
        imageLoader.clearErrorCallback()
        videoLoader.clearErrorCallback()
    }

    /**
     * Sets up impression tracking for individual product items when they become visible.
     */
    private fun setupProductImpressionTracking(container: FrameLayout, productAd: NativeAd, cliUbid: String) {
        val viewTreeObserver = container.viewTreeObserver
        if (viewTreeObserver.isAlive) {
            lateinit var preDrawListener: ViewTreeObserver.OnPreDrawListener
            preDrawListener = ViewTreeObserver.OnPreDrawListener {
                try {
                    if (container.isVisibleInScreen()) {
                        // Track impression for individual product
                        productAd.uclid?.let { uclid ->
                            adTracker.trackImpression(uclid, cliUbid)
                        }

                        // Remove listener after first impression
                        container.viewTreeObserver.removeOnPreDrawListener(preDrawListener)
                    }
                } catch (e: Exception) {
                    errorLog("Error tracking product impression: ${e.message}", OsmosError.UNKNOWN, e)
                    container.viewTreeObserver.removeOnPreDrawListener(preDrawListener)
                }
                true
            }

            viewTreeObserver.addOnPreDrawListener(preDrawListener)
        }
    }

    /**
     * Logs an error message using structured error handling.
     */
    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    fun createVerticalAdView(
        context: Context,
        pdaAdData: PDAAd,
        width: Int,
        height: Int?,
        pdaAdCustomStyle: PDAAdCustomStyle?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        customBadgeView: View?,
        onViewLoadListener: ((adData: Map<String, Any>, String) -> Unit)? = null,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)? = null,
    ): View {
        return try {
            val (screenWidth, screenHeight) = ViewUtils.getScreenHeightWidth(0.97f, 0.8f)

            val finalWidth = minOf(ViewUtils.dpToPx(context, width), screenWidth)
            val isClamped = pdaAdCustomStyle?.let {
                (ViewUtils.dpToPx(context, it.bannerWidth ?: 0) > screenWidth ||
                        ViewUtils.dpToPx(context, it.bannerHeight ?: 0) > screenHeight)
            } ?: false

            val h: Int? = if (height != null) {
                ViewUtils.dpToPx(context, height)
            } else {
                null
            }
            val finalHeight = minOf(h ?: screenHeight, screenWidth)

            val mainContainer = FrameLayout(context).apply {
                layoutParams = ViewGroup.LayoutParams(
                    finalWidth,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                )
                setBackgroundColor(Color.WHITE)
            }

            val pdaContainer = LinearLayout(context).apply {
                orientation = LinearLayout.VERTICAL
                layoutParams = ViewGroup.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT,
                )
            }

            // Added a adTag and Badge
            val topTagRow =
                loadTagBudgeView(context, customBadgeView)

            val bannerView = when {
                !pdaAdData.elements.value.isNullOrEmpty() -> {
                    createBannerView(
                        context,
                        pdaAdData,
                        pdaAdData.elements.type!!,
                        isClamped,
                        true,
                        finalWidth,
                        finalHeight,
                        pdaAdCustomStyle?.bannerWidth,
                        pdaAdCustomStyle?.bannerHeight,
                        onViewLoadListener,
                        adClickedListener
                    )
                }

                !pdaAdData.elements.carouselCards.isNullOrEmpty() -> {
                    createBannerView(
                        context,
                        pdaAdData,
                        "CAROUSEL",
                        isClamped,
                        true,
                        finalWidth,
                        finalHeight,
                        pdaAdCustomStyle?.bannerWidth,
                        pdaAdCustomStyle?.bannerHeight,
                        onViewLoadListener,
                        adClickedListener
                    )
                }

                else -> null
            }

            bannerView?.let {
                val params = LinearLayout.LayoutParams(
                    LinearLayout.LayoutParams.WRAP_CONTENT,
                    LinearLayout.LayoutParams.WRAP_CONTENT
                ).apply {
                    gravity = Gravity.CENTER_HORIZONTAL
                }
                it.layoutParams = params
                pdaContainer.addView(it)
            }

            productListRecyclerView = RecyclerView(context).apply {
                if (pdaAdCustomStyle?.productScrollDirection == RecyclerView.HORIZONTAL) {
                    layoutManager =
                        LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false).apply {
                            layoutParams = ViewGroup.LayoutParams(
                                ViewGroup.LayoutParams.WRAP_CONTENT,
                                ViewGroup.LayoutParams.WRAP_CONTENT
                            )
                        }
                } else {
                    layoutManager =
                        LinearLayoutManager(context, LinearLayoutManager.VERTICAL, false).apply {
                            layoutParams = ViewGroup.LayoutParams(
                                ViewGroup.LayoutParams.MATCH_PARENT,
                                ViewGroup.LayoutParams.WRAP_CONTENT
                            )
                        }
                }

                adapter = ProductListAdapter(
                    context,
                    pdaAdData,
                    pdaAdData.elements.productList,
                    pdaAdCustomStyle,
                    baseNativeAdView,
                    adClickedListener,
                    onViewLoadListener
                )
            }

            pdaContainer.addView(productListRecyclerView)
            mainContainer.addView(pdaContainer)
            mainContainer.addView(topTagRow)
            mainContainer.addView(setAdLabel(context, adLabelText, adLabelAlignment))
            mainContainer
        } catch (e: Exception) {
            errorLog(
                "Error creating vertical PDA ad view: ${e.message}",
                OsmosError.PARSING_ERROR,
                e
            )
            FrameLayout(context) // Return empty container on error
        }
    }

    fun createHorizontalAdView(
        context: Context,
        pdaAdData: PDAAd,
        width: Int,
        height: Int?,
        pdaAdCustomStyle: PDAAdCustomStyle?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        customBadgeView: View?,
        onViewLoadListener: ((adData: Map<String, Any>, String) -> Unit)? = null,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)? = null,
    ): View {
        return try {
            val (screenWidth, screenHeight) = ViewUtils.getScreenHeightWidth(0.97f, 0.8f)

            val finalWidth = minOf(ViewUtils.dpToPx(context, width), screenWidth)
            val isClamped = pdaAdCustomStyle?.let {
                (it.bannerWidth ?: 0 > screenWidth || it.bannerHeight ?: 0 > screenHeight)
            } ?: false

            val h: Int? = if (height != null) {
                ViewUtils.dpToPx(context, height)
            } else {
                null
            }
            val finalHeight = minOf(h ?: screenHeight, screenWidth)

            val mainContainer = FrameLayout(context).apply {
                layoutParams = ViewGroup.LayoutParams(
                    finalWidth,
                    finalHeight
                )
                setBackgroundColor(Color.WHITE)
            }

            val pdaContainer = LinearLayout(context).apply {
                orientation = LinearLayout.HORIZONTAL
                layoutParams = ViewGroup.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT,
                )
            }

            // Added a adTag and Badge
            val topTagRow =
                loadTagBudgeView(context, customBadgeView)

            val bannerView = when {
                !pdaAdData.elements.value.isNullOrEmpty() -> {
                    createBannerView(
                        context,
                        pdaAdData,
                        pdaAdData.elements.type!!,
                        isClamped,
                        false,
                        finalWidth / 2,
                        finalHeight,
                        pdaAdCustomStyle?.bannerWidth,
                        pdaAdCustomStyle?.bannerHeight,
                        onViewLoadListener,
                        adClickedListener
                    )
                }

                !pdaAdData.elements.carouselCards.isNullOrEmpty() -> {
                    createBannerView(
                        context,
                        pdaAdData,
                        "CAROUSEL",
                        isClamped,
                        false,
                        finalWidth / 2,
                        finalHeight,
                        pdaAdCustomStyle?.bannerWidth,
                        pdaAdCustomStyle?.bannerHeight,
                        onViewLoadListener,
                        adClickedListener
                    )
                }

                else -> null
            }

            bannerView?.let {
                val params = LinearLayout.LayoutParams(
                    finalWidth / 2,
                    finalHeight
                ).apply {
                    gravity = Gravity.CENTER_VERTICAL
                }
                it.layoutParams = params
                pdaContainer.addView(it)
            }

            productListRecyclerView = RecyclerView(context).apply {
                if (pdaAdCustomStyle?.productScrollDirection == RecyclerView.HORIZONTAL) {
                    layoutManager =
                        LinearLayoutManager(context, LinearLayoutManager.HORIZONTAL, false).apply {
                            layoutParams = ViewGroup.LayoutParams(
                                ViewGroup.LayoutParams.WRAP_CONTENT,
                                ViewGroup.LayoutParams.MATCH_PARENT
                            )
                        }
                } else {
                    layoutManager =
                        LinearLayoutManager(context, LinearLayoutManager.VERTICAL, false).apply {
                            layoutParams = ViewGroup.LayoutParams(
                                ViewGroup.LayoutParams.MATCH_PARENT,
                                ViewGroup.LayoutParams.WRAP_CONTENT
                            )
                        }
                }

                adapter = ProductListAdapter(
                    context,
                    pdaAdData,
                    pdaAdData.elements.productList,
                    pdaAdCustomStyle,
                    baseNativeAdView,
                    adClickedListener,
                    onViewLoadListener
                )
            }

            pdaContainer.addView(productListRecyclerView)
            mainContainer.addView(pdaContainer)
            mainContainer.addView(topTagRow)
            mainContainer.addView(setAdLabel(context, adLabelText, adLabelAlignment))
            mainContainer
        } catch (e: Exception) {
            errorLog(
                "Error creating horizontal PDA ad view: ${e.message}",
                OsmosError.PARSING_ERROR,
                e
            )
            FrameLayout(context) // Return empty container on error
        }
    }

    private fun setAdLabel(context: Context, adLabelText: String?, adLabelAlignment: Int?): View {
        return try {
            val alignment = adLabelAlignment ?: (Gravity.TOP or Gravity.START)
            if (!adLabelText.isNullOrEmpty()) {
                ViewUtils.getAdLabelTextview(context, adLabelText, alignment, 5, 5)
            } else {
                ViewUtils.getEmptyView(context)
            }
        } catch (e: Exception) {
            errorLog("Error setting ad label: ${e.message}", OsmosError.PARSING_ERROR, e)
            ViewUtils.getEmptyView(context)
        }
    }

    private fun createBannerView(
        context: Context,
        ad: PDAAd,
        mediaType: String,
        isClamped: Boolean,
        isVerticalPDA: Boolean,
        finalWidth: Int,
        finalHeight: Int,
        bannerWidth: Int?,
        bannerHeight: Int?,
        onViewLoadListener: ((adData: Map<String, Any>, String) -> Unit)? = null,
        adClickedListener: ((adData: Map<String, Any>) -> Unit)? = null
    ): View {
        return try {
            val w = bannerWidth ?: ad.elements.width
            val h = bannerHeight ?: ad.elements.height

            val bannerFinalWidth = minOf(w ?: finalWidth, finalWidth)
            val bannerFinalHeight = minOf(h ?: finalHeight, finalHeight)

            val productAdContainer = FrameLayout(context).apply {
                layoutParams = ViewGroup.LayoutParams(
                    if (isVerticalPDA) bannerFinalWidth else finalWidth,
                    if (isVerticalPDA) h
                        ?: ViewGroup.LayoutParams.WRAP_CONTENT else ViewGroup.LayoutParams.WRAP_CONTENT
                )
                setPadding(10, 10, 10, 10)
            }

            when (mediaType) {
                "IMAGE" -> {
                    val imageAd = getImageAdObject(
                        pda = ad,
                        cardHeight = bannerFinalHeight,
                        cardWidth = if (isVerticalPDA) bannerFinalWidth else finalWidth,
                        carouselCards = null
                    )

                    val imageView = imageLoader.createImageView(
                        context,
                        if (isVerticalPDA) bannerFinalWidth else finalWidth,
                        bannerFinalHeight,
                        isClamped
                    ).apply {
                        layoutParams = FrameLayout.LayoutParams(
                            ViewGroup.LayoutParams.MATCH_PARENT,
                            ViewGroup.LayoutParams.MATCH_PARENT
                        )
                        scaleType = ImageView.ScaleType.FIT_CENTER
                    }

                    imageLoader.loadImage(
                        imageAd,
                        imageView,
                        ad.cliUbid ?: "",
                        false,
                        adClickListener = { adClickedListener?.invoke(imageAd.toMap()) }
                    )
                    productAdContainer.addView(imageView)
                }

                "VIDEO" -> {
                    val width = if (isVerticalPDA) bannerFinalWidth else finalWidth / 2
                    val videoAd = getVideoAdObject(ad, bannerFinalHeight, width, null)

                    val videoView = videoLoader.createVideoView(
                        context,
                        videoAd,
                        false,
                        width,
                        bannerFinalHeight,
                        ad.cliUbid ?: "",
                        adClickListener = { adClickedListener?.invoke(videoAd.toMap()) },
                        onViewLoadListener = onViewLoadListener
                    ).apply {
                        layoutParams = FrameLayout.LayoutParams(
                            ViewGroup.LayoutParams.MATCH_PARENT,
                            ViewGroup.LayoutParams.MATCH_PARENT
                        )
                    }
                    productAdContainer.addView(videoView)
                }

                "CAROUSEL" -> {
                    val recyclerView = RecyclerView(context).apply {
                        layoutManager = LinearLayoutManager(
                            context,
                            LinearLayoutManager.HORIZONTAL,
                            false
                        ).apply {
                            layoutParams = FrameLayout.LayoutParams(
                                ViewGroup.LayoutParams.WRAP_CONTENT,
                                ViewGroup.LayoutParams.WRAP_CONTENT,
                                Gravity.CENTER_VERTICAL
                            )
                        }
                        adapter = BannerCarouselAdapter(
                            context,
                            ad,
                            ad.elements.carouselCards ?: emptyList(),
                            ad.cliUbid,
                            bannerFinalHeight,
                            bannerWidth,
                        )
                    }

                    ViewUtils.setItemDecoration(context, recyclerView)
                    productAdContainer.addView(recyclerView)

                    val dotsContainer = ViewUtils.addDotIndicator(context, ad)
                    productAdContainer.addView(dotsContainer)

                    recyclerView.addOnScrollListener(object : RecyclerView.OnScrollListener() {
                        override fun onScrolled(rv: RecyclerView, dx: Int, dy: Int) {
                            ViewUtils.handleVisibleItem(recyclerView, videoLoader, dotsContainer)
                        }
                    })

                    recyclerView.post {
                        ViewUtils.handleVisibleItem(recyclerView, videoLoader, dotsContainer)
                    }
                }
            }
            productAdContainer
        } catch (e: Exception) {
            errorLog("Error creating banner view: ${e.message}", OsmosError.PARSING_ERROR, e)
            FrameLayout(context) // Return empty container on error
        }
    }

    private fun loadTagBudgeView(
        context: Context,
        customBadgeView: View?
    ): RelativeLayout {
        return try {
            val topTagRow = RelativeLayout(context).apply {
                layoutParams = LinearLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                )
                setPadding(10, 5, 10, 8)
            }

            customBadgeView?.let { badge ->
                if (badge is TextView) {
                    badge.layoutParams = RelativeLayout.LayoutParams(
                        ViewGroup.LayoutParams.WRAP_CONTENT,
                        ViewGroup.LayoutParams.WRAP_CONTENT,
                    ).apply {
                        addRule(RelativeLayout.ALIGN_PARENT_END)
                        addRule(RelativeLayout.ALIGN_PARENT_TOP)
                        addRule(RelativeLayout.ALIGN_PARENT_RIGHT)
                    }
                } else {
                    badge.layoutParams = RelativeLayout.LayoutParams(
                        ViewUtils.dpToPx(context, 50),
                        ViewUtils.dpToPx(context, 30)
                    ).apply {
                        addRule(RelativeLayout.ALIGN_PARENT_END)
                        addRule(RelativeLayout.ALIGN_PARENT_TOP)
                        addRule(RelativeLayout.ALIGN_PARENT_RIGHT)
                    }
                }
                topTagRow.addView(badge)
            }

            topTagRow
        } catch (e: Exception) {
            errorLog("Error loading tag badge view: ${e.message}", OsmosError.PARSING_ERROR, e)
            RelativeLayout(context)
        }
    }

    private fun getImageAdObject(
        pda: PDAAd,
        cardHeight: Int,
        cardWidth: Int,
        carouselCards: CarouselAdElement?
    ): ImageAd {
        return if (carouselCards == null) {
            ImageAd(
                clientId = pda.clientId,
                au = pda.au,
                rank = pda.rank,
                clickTrackingUrl = pda.clickTrackingUrl,
                impressionTrackingUrl = pda.impressionTrackingUrl,
                uclid = pda.uclid,
                crt = pda.crt,
                elements = AdElement(
                    value = pda.elements.value ?: "",
                    height = cardHeight,
                    width = cardWidth,
                    type = pda.elements.type ?: "IMAGE",
                    destinationUrl = null
                ),
                adMetadata = pda.adMetadata,
                cliUbid = pda.cliUbid
            )
        } else {
            ImageAd(
                clientId = pda.clientId,
                au = pda.au,
                rank = pda.rank,
                clickTrackingUrl = pda.clickTrackingUrl,
                impressionTrackingUrl = pda.impressionTrackingUrl,
                uclid = pda.uclid,
                crt = pda.crt,
                elements = AdElement(
                    value = carouselCards.media.value,
                    height = carouselCards.media.height,
                    width = carouselCards.media.width,
                    type = carouselCards.media.type,
                    destinationUrl = null
                ),
                adMetadata = pda.adMetadata,
                cliUbid = pda.cliUbid
            )
        }
    }

    private fun getVideoAdObject(
        pdaAdData: PDAAd,
        cardHeight: Int,
        cardWidth: Int,
        carouselCards: CarouselAdElement?
    ): VideoAd {
        return if (carouselCards == null) {
            VideoAd(
                clientId = pdaAdData.clientId,
                au = pdaAdData.au,
                rank = pdaAdData.rank,
                clickTrackingUrl = pdaAdData.clickTrackingUrl,
                impressionTrackingUrl = pdaAdData.impressionTrackingUrl,
                uclid = pdaAdData.uclid,
                crt = pdaAdData.crt,
                elements = AdElement(
                    value = pdaAdData.elements.value ?: "",
                    height = cardHeight,
                    width = cardWidth,
                    type = pdaAdData.elements.type ?: "VIDEO",
                    destinationUrl = null
                ),
                adMetadata = pdaAdData.adMetadata,
                cliUbid = pdaAdData.cliUbid
            )
        } else {
            VideoAd(
                clientId = pdaAdData.clientId,
                au = pdaAdData.au,
                rank = pdaAdData.rank,
                clickTrackingUrl = pdaAdData.clickTrackingUrl,
                impressionTrackingUrl = pdaAdData.impressionTrackingUrl,
                uclid = pdaAdData.uclid,
                crt = pdaAdData.crt,
                elements = AdElement(
                    value = carouselCards.media.value,
                    height = cardHeight,
                    width = cardWidth,
                    type = carouselCards.media.type,
                    destinationUrl = carouselCards.destinationUrl
                ),
                adMetadata = pdaAdData.adMetadata,
                cliUbid = pdaAdData.cliUbid
            )
        }
    }

    inner class BannerCarouselAdapter(
        private val context: Context,
        private val pdaAdData: PDAAd,
        private val carouselCards: List<CarouselAdElement>,
        private val cliUbid: String?,
        private val bannerHeight: Int?,
        private val bannerWidth: Int?
    ) : RecyclerView.Adapter<RecyclerView.ViewHolder>() {

        private val TYPE_VIDEO = 1
        private val TYPE_IMAGE = 0
        private var isFirstVisible = true

        override fun getItemViewType(position: Int): Int {
            val type = carouselCards[position].media.type
            return if (type.equals("video", ignoreCase = true)) TYPE_VIDEO else TYPE_IMAGE
        }

        override fun getItemCount(): Int = carouselCards.size

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): RecyclerView.ViewHolder {
            val frame = FrameLayout(context)
            return if (viewType == TYPE_VIDEO) VideoViewHolder(frame) else ImageViewHolder(frame)
        }

        override fun onBindViewHolder(holder: RecyclerView.ViewHolder, position: Int) {
            try {
                val item = carouselCards[position]
                val density = context.resources.displayMetrics.density
                val pxW = ((bannerWidth ?: 0) * density).toInt()
                val pxH = ((bannerHeight ?: 0) * density).toInt()

                when (holder) {
                    is VideoViewHolder -> {
                        holder.frame.removeAllViews()
                        val videoAd =
                            getVideoAdObject(pdaAdData, bannerHeight ?: 0, bannerWidth ?: 0, item)

                        val videoView = videoLoader.createCarouselVideoView(
                            context,
                            videoAd,
                            bannerWidth ?: 0,
                            bannerHeight ?: 0,
                            "$cliUbid-$position",
                            isCarousal = true
                        )
                        holder.playerView = videoView.getChildAt(0) as PlayerView
                        holder.frame.setOnClickListener {
                            adTracker.trackAdClick(videoAd.uclid, videoAd.cliUbid ?: "")
                        }
                        holder.frame.addView(videoView, FrameLayout.LayoutParams(pxW, pxH))
                    }

                    is ImageViewHolder -> {
                        holder.frame.removeAllViews()
                        val imageAd = getImageAdObject(pdaAdData, pxW, pxH, item)

                        val imageView = imageLoader.createImageView(
                            context,
                            bannerWidth ?: 0,
                            bannerHeight ?: 0,
                            isClamped = false
                        )
                        imageLoader.loadImage(
                            imageAd,
                            imageView,
                            "$cliUbid-$position",
                            true,
                        )
                        holder.frame.setOnClickListener {
                            adTracker.trackAdClick(imageAd.uclid, imageAd.cliUbid ?: "")
                        }
                        holder.frame.addView(imageView, FrameLayout.LayoutParams(pxW, pxH))
                    }
                }

                if (isFirstVisible && position == 0) {
                    isFirstVisible = false
                }
            } catch (e: Exception) {
                errorLog(
                    "Error binding carousel item at position $position: ${e.message}",
                    OsmosError.PARSING_ERROR,
                    e
                )
            }
        }

        inner class VideoViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame) {
            lateinit var playerView: PlayerView
        }

        inner class ImageViewHolder(val frame: FrameLayout) : RecyclerView.ViewHolder(frame)
    }

    inner class ProductListAdapter(
        private val context: Context,
        private val pdaAdData: PDAAd,
        private val productList: List<ProductItem>,
        private val pdaAdCustomStyle: PDAAdCustomStyle?,
        private val baseNativeAdView: BaseNativeAdView,
        private val adClickedListener: ((adData: Map<String, Any>) -> Unit)?,
        private val onViewLoadListener: ((adData: Map<String, Any>, String) -> Unit)?
    ) : RecyclerView.Adapter<ProductListAdapter.ProductAdViewHolder>() {

        private var isFirstVisible = true

        inner class ProductAdViewHolder(val container: FrameLayout) :
            RecyclerView.ViewHolder(container)

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ProductAdViewHolder {
            val marginInPx = ViewUtils.dpToPx(context, 5)
            val screenWidth = context.resources.displayMetrics.widthPixels

            val container = FrameLayout(context).apply {
                layoutParams = RecyclerView.LayoutParams(
                    screenWidth / 2,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                ).apply {
                    leftMargin = marginInPx
                    rightMargin = marginInPx
                    topMargin = marginInPx
                }
            }
            return ProductAdViewHolder(container)
        }

        override fun getItemCount(): Int = productList.size

        override fun onBindViewHolder(holder: ProductAdViewHolder, position: Int) {
            try {
                val item = productList[position]
                val productAd = getNativeAdObject(item)

                val nativeAdView: View = when (pdaAdCustomStyle?.pdaProductStyle) {
                    Vertical -> baseNativeAdView.createProductVerticalAdView(
                        context,
                        productAd,
                        pdaAdCustomStyle.nativeAdLayoutType,
                        pdaAdCustomStyle.nativeAdWidth ?: 200,
                        pdaAdCustomStyle.nativeAdHeight ?: 200,
                        pdaAdCustomStyle.nativeStyle,
                        pdaAdCustomStyle.ctaButton,
                    )

                    Horizontal -> baseNativeAdView.createProductHorizontalAdView(
                        context,
                        productAd,
                        pdaAdCustomStyle.nativeAdLayoutType,
                        pdaAdCustomStyle.nativeAdWidth ?: 200,
                        pdaAdCustomStyle.nativeAdHeight ?: 200,
                        pdaAdCustomStyle.nativeStyle,
                        pdaAdCustomStyle.ctaButton,
                    )

                    Custom -> {
                        errorLog(
                            "Custom PDA product style not implemented",
                            OsmosError.PARSING_ERROR
                        )
                        View(context)
                    }

                    else -> {
                        errorLog("PDA product style not specified", OsmosError.PARSING_ERROR)
                        View(context)
                    }
                }

                nativeAdView.setOnClickListener {
                    adClickedListener?.invoke(productAd.toMap())
                }

                nativeAdView.let {
                    (it.parent as? ViewGroup)?.removeView(it)
                    holder.container.removeAllViews()
                    holder.container.addView(it)
                }

                val divider = View(context).apply {
                    layoutParams = FrameLayout.LayoutParams(
                        FrameLayout.LayoutParams.MATCH_PARENT,
                        ViewUtils.dpToPx(context, 1)
                    ).apply {
                        topMargin = ViewUtils.dpToPx(context, 5)
                        bottomMargin = ViewUtils.dpToPx(context, 8)
                    }
                    setBackgroundColor(Color.LTGRAY)
                }

                if (pdaAdCustomStyle?.productScrollDirection == RecyclerView.VERTICAL && position != 0 && position != itemCount) {
                    holder.container.addView(divider)
                }

                // Track impression for individual product items in carousel
                setupProductImpressionTracking(holder.container, productAd, "${pdaAdData.cliUbid}-product-$position")

                if (isFirstVisible && position == 0) {
                    isFirstVisible = false
                    onViewLoadListener?.invoke(pdaAdData.toMap(), pdaAdData.cliUbid ?: "")
                }
            } catch (e: Exception) {
                errorLog(
                    "Error binding product item at position $position: ${e.message}",
                    OsmosError.PARSING_ERROR,
                    e
                )
            }
        }

        private fun getNativeAdObject(pda: ProductItem): NativeAd {
            return NativeAd(
                clientId = null,
                skuId = pda.skuId,
                name = pda.name,
                mrp = pda.mrp,
                productUrl = pda.productUrl,
                storeId = pda.storeId,
                imageUrl = pda.imageUrl,
                videoUrl = pda.videoUrl,
                salePrice = pda.salePrice,
                sellerId = pda.sellerId,
                itemGroupId = pda.itemGroupId,
                brand = pda.brand,
                description = pda.description,
                customLabel0 = pda.customLabel0,
                customLabel1 = pda.customLabel1,
                customLabel2 = pda.customLabel2,
                customLabel3 = pda.customLabel3,
                customLabel4 = pda.customLabel4,
                uclid = pda.uclid,
                categoryL1 = pda.categoryL1,
                categoryL2 = pda.categoryL2,
                categoryL3 = pda.categoryL3,
                adMetadata = null,
                cliUbid = pda.cliUbid,
                crt = ""
            )
        }
    }

    /**
     * Cleans up all resources to prevent memory leaks.
     * Follows the same approach as CarouselAdView.
     */
    fun cleanUp() {
        try {
            baseNativeAdView.cleanUp()

            // Release video players (essential for memory management)
            videoLoader.safeReleaseAllPlayers()

            // Clear error callbacks to prevent memory leaks
            imageLoader.clearErrorCallback()
            videoLoader.clearErrorCallback()
            clearErrorCallback()
        } catch (e: Exception) {
            errorLog("Error during cleanup: ${e.message}", OsmosError.UNKNOWN, e)
        }
    }
}