package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.renderer.loaders.CarouselLoader
import com.ai.osmos.ads.views.interfaces.CarouselAdViewInterface
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.CarouselAd
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: CarouselAdView
 *
 * CarouselAdView for displaying carousel-type advertisements.
 *
 * @param context Application context.
 * @param adData Carousel ad Data (defined data structure of carousel ad)
 * @param width(optional): Width of carousel ad view
 * @param height(optional): Height of carousel ad view
 * @param attrs(optional): Attrs for ad view
 * @param errorCallback(optional): Callback for handling errors
 * @return CarouselAdView or EmptyView
 * Example:
 * ```
 * val adView = CarouselAdView(
 *     context = this@MainActivity,
 *     adData = adData,
 *     width = 300,
 *     height = 200,
 *     errorCallback = object : ErrorCallback {
 *         override fun onError(errorCode: String, errorMessage: String, throwable: Throwable?) {
 *             Log.e("CarouselAd", "Error: $errorCode - $errorMessage")
 *         }
 *     }
 * )
 * ```
 */
@SuppressLint("ViewConstructor")
internal class CarouselAdView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private val width: Int? = null,
    private val height: Int? = null,
    private val elementWidth: Int? = null,
    private val elementHeight: Int? = null,
    private var adLabelText: String? = null,
    private var adLabelAlignment: Int? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config,
    attrs: AttributeSet? = null,
) : BaseAdView(context, attrs, config), CarouselAdViewInterface {

    private val persistentAdTracker by lazy {
        if (context.applicationContext != null) {
            PersistentAdTracker.create(context.applicationContext, coroutineScope, config)
        } else {
            PersistentAdTracker.create(context, coroutineScope, config)
        }
    }
    private val carouselLoader = CarouselLoader(coroutineScope, persistentAdTracker)

    init {
        errorCallback?.let {
            setErrorCallback(it)
            persistentAdTracker.setErrorCallback(it)
            // Pass error callback to CarouselLoader's internal loaders
            carouselLoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, context) else showEmptyView(context)
    }

    override fun loadAd(adData: Map<String, Any>, context: Context) {
        removeAllViews()
        coroutineScope.launch(Dispatchers.IO) {
            try {
                if (isRecyclerViewAvailable()) {
                    val ad = adData["ad"] as? BaseAd
                    if (ad != null) {
                        val cliUbid = getCliUbid(ad)
                        if (isSupportedCrtCarousel(ad.crt)) {
                            withContext(Dispatchers.Main) {
                                val carouselAd = ad as? CarouselAd
                                if (carouselAd != null) {

                                    val view = carouselLoader.renderCarouselView(
                                        context,
                                        width = width,
                                        height = height,
                                        elementWidth = elementWidth,
                                        elementHeight = elementHeight,
                                        adLabelText = adLabelText,
                                        adLabelAlignment = adLabelAlignment,
                                        cliUbid = cliUbid,
                                        adList = carouselAd.elements.carouselCards,
                                        carouselAd = carouselAd,
                                        adClickedListener = itemClickedListener,
                                        onViewLoadListener = {
                                            onViewLoadListener?.invoke(
                                                carouselAd.toMap(),
                                                cliUbid
                                            )
                                        }
                                    )
                                    addView(view)
                                } else showEmptyView(context)
                            }
                        } else showEmptyView(context)
                    } else showEmptyView(context)
                } else showEmptyView(context)
            } catch (e: Exception) {
                showEmptyView(context)
            }
        }
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {

    }


    override fun onDetachedFromWindow() {
        carouselLoader.releaseAllPlayers() // Release video players (essential for memory management)
        clearAllErrorCallbacks()
        super.onDetachedFromWindow()
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     */
    private fun clearAllErrorCallbacks() {
        carouselLoader.clearErrorCallback() // Clear both ImageLoader and VideoLoader error callbacks
        persistentAdTracker.setErrorCallback(null)
        clearErrorCallback()
    }

    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}
