package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.renderer.loaders.CarouselLoader
import com.ai.osmos.ads.views.interfaces.MultiAdCarouselViewInterface
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.MultiAd
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.error.ErrorCallback
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: MultiAdCarouselView
 *
 * MultiAdCarouselView for displaying multiple advertisements in a carousel format.
 *
 * @param context Application context.
 * @param adData Multi-ad carousel Data (defined data structure of multi-ad carousel)
 * @param width(optional): Width of multi-ad carousel view
 * @param height(optional): Height of multi-ad carousel view
 * @param attrs(optional): Attrs for ad view
 * @param errorCallback(optional): Callback for handling errors
 * @return MultiAdCarouselView or EmptyView
 * Example:
 * ```
 * val adView = MultiAdCarouselView(
 *     context = this@MainActivity,
 *     adData = adData,
 *     width = 300,
 *     height = 200,
 *     errorCallback = object : ErrorCallback {
 *         override fun onError(errorCode: String, errorMessage: String, throwable: Throwable?) {
 *             Log.e("MultiAdCarousel", "Error: $errorCode - $errorMessage")
 *         }
 *     }
 * )
 * ```
 */

@SuppressLint("ViewConstructor")
internal class MultiAdCarouselView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private val width: Int? = null,
    private val height: Int? = null,
    private val elementWidth: Int? = null,
    private val elementHeight: Int? = null,
    private var adLabelText: String? = null,
    private var adLabelAlignment: Int? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config,
    attrs: AttributeSet? = null,
) : BaseAdView(context, attrs, config), MultiAdCarouselViewInterface {

    private val persistentAdTracker by lazy {
        if (context.applicationContext != null) {
            PersistentAdTracker.create(context.applicationContext, coroutineScope, config)
        } else {
            PersistentAdTracker.create(context, coroutineScope, config)
        }
    }
    private val carouselLoader = CarouselLoader(coroutineScope, persistentAdTracker)

    init {
        errorCallback?.let {
            setErrorCallback(it)
            persistentAdTracker.setErrorCallback(it)
            // Pass error callback to CarouselLoader's internal loaders
            carouselLoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, context) else showEmptyView(context)
    }

    override fun loadAd(adData: Map<String, Any>, context: Context) {
        removeAllViews()
        coroutineScope.launch(Dispatchers.IO) {
            try {
                if (isRecyclerViewAvailable()) {
                    val ad = adData["ad"] as? List<MultiAd>

                    if (ad != null) {
                        withContext(Dispatchers.Main) {
                            val view = carouselLoader.renderMultiAdCarouselView(
                                context,
                                width = width,
                                height = height,
                                multiAd = ad,
                                elementWidth = elementWidth,
                                elementHeight = elementHeight,
                                adLabelText = adLabelText,
                                adLabelAlignment = adLabelAlignment,
                                adClickListener = adClickedListener,
                                onViewLoadListener = onViewLoadListener
                            )
                            addView(view)
                        }
                    } else showEmptyView(context)
                } else showEmptyView(context)

            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    showEmptyView(context)
                }
            }
        }
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {

    }

    override fun onDetachedFromWindow() {
        carouselLoader.releaseAllPlayers() // Release video players (essential for memory management)
        clearAllErrorCallbacks()
        super.onDetachedFromWindow()
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     */
    private fun clearAllErrorCallbacks() {
        carouselLoader.clearErrorCallback() // Clear both ImageLoader and VideoLoader error callbacks
        persistentAdTracker.setErrorCallback(null)
        clearErrorCallback()
    }

    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}

