package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.RecyclerView
import com.ai.osmos.ads.renderer.loaders.NativeAdLoader
import com.ai.osmos.ads.views.adapters.NativeAdCarouselAdapter
import com.ai.osmos.ads.views.interfaces.NativeAdCarouselViewInterface
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.error.ErrorCallback
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: NativeAdCarouselView
 */

/**
 * NativeAdCarouselView for displaying native advertisements in carousel layouts.
 *
 * @param context Application context
 * @param adData Map containing native ad data structure
 * @param adType Type of native ad (SMALL, MEDIUM, LARGE)
 * @param width Width of native ad view in pixels
 * @param height Optional height of native ad view in pixels
 * @param adLabelText Optional text for ad label
 * @param adLabelAlignment Optional alignment for ad label
 * @param customStyle Optional custom styling for the native ad
 * @param customCtaView Optional custom CTA button view
 * @param customBadgeView Optional custom badge view
 * @param attrs Optional attribute set for ad view styling
 * @param errorCallback Optional callback for handling errors
 * @param config SDK configuration object
 */
@SuppressLint("ViewConstructor")
internal class NativeAdCarouselView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private var adType: NativeAdStyle,
    private var width: Int,
    private var height: Int? = null,
    private var elementWidth: Int? = null,
    private var adLabelText: String? = null,
    private var adLabelAlignment: Int? = null,
    private var customStyle: NativeAdCustomStyle? = null,
    private var customCtaView: View? = null,
    private var customBadgeView: View? = null,
    attrs: AttributeSet? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config
) : BaseAdView(context, attrs, config), NativeAdCarouselViewInterface {

    private val nativeLoader = NativeAdLoader(coroutineScope, adTracker, config)
    private var loadAdJob: kotlinx.coroutines.Job? = null

    // Initialize the view by loading ad or showing an empty placeholder
    init {
        errorCallback?.let {
            setErrorCallback(it)
            (adTracker as? PersistentAdTracker)?.setErrorCallback(it)
            nativeLoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, context) else showEmptyView(context)
    }

    /**
     * Loads and displays a native carousel ad based on the given ad data.
     *
     * @param adData A map containing the ad object and its associated metadata.
     *               Expected to include a key `"ad"` with value of type `BaseAd`.
     */
    @SuppressLint("SuspiciousIndentation")
    override fun loadAd(adData: Map<String, Any>, context: Context) {
        // Cancel any existing job to prevent multiple concurrent loads
        loadAdJob?.cancel()

        loadAdJob = coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? List<NativeAd>

                if (ad != null) {
                    withContext(Dispatchers.Main) {
                        removeAllViews()
                        val nativeAdList = ad as? List<NativeAd>
                        if (nativeAdList != null) {
                            val (screenWidth, screenHeight) = getScreenHeightWidth()
                            val result = setMaxHeightWidth(
                                height ?: screenHeight,
                                width ?: screenWidth,
                                null
                            )

                            val view = nativeLoader.renderNativeAdCarouselView(
                                context = context,
                                nativeAd = nativeAdList,
                                adType = adType,
                                width = result.finalWidth,
                                height = result.finalHeight,
                                cardElementWidth = elementWidth,
                                customStyle = customStyle,
                                customCtaView = customCtaView,
                                customBadgeView = customBadgeView,
                                adLabelText = adLabelText,
                                adLabelAlignment = adLabelAlignment,
                                adClickedListener = adClickedListener,
                                onViewLoadListener = onViewLoadListener
                            )
                            addView(view)

                        } else {
                            showEmptyView(context)
                        }
                    }
                } else {
                    withContext(Dispatchers.Main) {
                        showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    showEmptyView(context)
                }
            }
        }
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {
    }

    override fun onDetachedFromWindow() {
        // Cancel any running coroutines to prevent memory leaks
        loadAdJob?.cancel()
        loadAdJob = null

        // Comprehensive cleanup to prevent memory leaks
        nativeLoader.cleanUp()
        nativeLoader.safeClearViewLoadListener()
        safeClearAllErrorCallbacks()

        // Clean up any RecyclerView adapters
        safeCleanupRecyclerViewAdapters()

        // Clear all views and listeners
        removeAllViews()

        // Clear callback references
        adClickedListener = null
        onViewLoadListener = null

        // Clear custom view references to prevent memory leaks
        customCtaView = null
        customBadgeView = null
        customStyle = null

        super.onDetachedFromWindow()
    }

    /**
     * Clean up all RecyclerView adapters to prevent memory leaks.
     * Made safe with null checks and bounds checking.
     */
    private fun safeCleanupRecyclerViewAdapters() {
        // Find all RecyclerViews in child views and clean up completely
        val count = childCount
        for (i in 0 until count) {
            val child = getChildAt(i) ?: continue
            if (child is RecyclerView) {
                // Complete RecyclerView cleanup
                val adapter = child.adapter
                if (adapter is NativeAdCarouselAdapter) {
                    adapter.cleanup()
                }
                child.adapter = null
                child.layoutManager = null  // Prevent LayoutManager memory leaks
                child.clearOnScrollListeners()  // Clear scroll listeners
                child.recycledViewPool.clear()  // Clear view pool
            }
            // Check nested ViewGroups recursively
            if (child is ViewGroup) {
                safeCleanupRecyclerViewInViewGroup(child)
            }
        }
    }

    /**
     * Recursively clean up RecyclerView adapters in nested ViewGroups.
     * Made safe with null checks and bounds checking.
     */
    private fun safeCleanupRecyclerViewInViewGroup(viewGroup: ViewGroup?) {
        viewGroup ?: return
        val count = viewGroup.childCount
        for (i in 0 until count) {
            val child = viewGroup.getChildAt(i) ?: continue
            if (child is RecyclerView) {
                // Complete RecyclerView cleanup
                val adapter = child.adapter
                if (adapter is NativeAdCarouselAdapter) {
                    adapter.cleanup()
                }
                child.adapter = null
                child.layoutManager = null  // Prevent LayoutManager memory leaks
                child.clearOnScrollListeners()  // Clear scroll listeners
                child.recycledViewPool?.clear()  // Clear view pool
            } else if (child is ViewGroup) {
                safeCleanupRecyclerViewInViewGroup(child)
            }
        }
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     * Made safe with null checks.
     */
    private fun safeClearAllErrorCallbacks() {
        nativeLoader.clearErrorCallback()
        (adTracker as? PersistentAdTracker)?.setErrorCallback(null)
        clearErrorCallback()
    }

    /**
     * Checks if the native ad format is supported.
     */
    private fun isSupportedNativeFormat(ad: BaseAd): Boolean {
        return ad is List<*> && ad.isNotEmpty() && ad.all { it is NativeAd && it.name.isNotEmpty() && !it.imageUrl.isNullOrEmpty() }
    }

    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}
