package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.renderer.loaders.NativeAdLoader
import com.ai.osmos.ads.views.interfaces.NativeAdHorizontalViewInterface
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.enums.NativeAdLayoutType
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: NativeAdHorizontalView
 */

/**
 * NativeAdHorizontalView for displaying native advertisements in horizontal layouts.
 *
 * @param context Application context
 * @param adData Map containing native ad data structure
 * @param adType Type of native ad (HORIZONTAL_SMALL, HORIZONTAL_LARGE, etc.)
 * @param width Width of native ad view in pixels
 * @param height Optional height of native ad view in pixels
 * @param adLabelText Optional text for ad label
 * @param adLabelAlignment Optional alignment for ad label
 * @param customStyle Optional custom styling for the native ad
 * @param customCtaView Optional custom CTA button view
 * @param customBadgeView Optional custom badge view
 * @param attrs Optional attribute set for ad view styling
 * @param errorCallback Optional callback for handling errors
 * @param config SDK configuration object
 *
 * Example:
 * ```
 * val adView = NativeAdHorizontalView(
 *     context = this@MainActivity,
 *     adData = adData,
 *     adType = NativeAdType.HORIZONTAL_SMALL,
 *     width = 400,
 *     height = 150,
 *     adLabelText = "Sponsored",
 *     adLabelAlignment = Gravity.TOP or Gravity.START,
 *     errorCallback = object : ErrorCallback {
 *         override fun onError(errorCode: String, errorMessage: String, throwable: Throwable?) {
 *             Log.e("NativeAd", "Error: $errorCode - $errorMessage")
 *         }
 *     },
 *     config = config
 * )
 * ```
 */
@SuppressLint("ViewConstructor")
internal class NativeAdHorizontalView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private var width: Int,
    private var height: Int? = null,
    private var adLabelText: String? = null,
    private var adLabelAlignment: Int? = null,
    private var customStyle: NativeAdCustomStyle? = null,
    private var customCtaView: View? = null,
    private var customBadgeView: View? = null,
    attrs: AttributeSet? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config
) : BaseAdView(context, attrs, config), NativeAdHorizontalViewInterface {

    private val nativeLoader = NativeAdLoader(coroutineScope, adTracker, config)

    // Initialize the view by loading ad or showing an empty placeholder
    init {
        errorCallback?.let {
            setErrorCallback(it)
            (adTracker as? PersistentAdTracker)?.setErrorCallback(it)
            nativeLoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, context) else showEmptyView(context)
    }

    /**
     * Loads and displays a native ad based on the given ad data.
     *
     * @param adData A map containing the ad object and its associated metadata.
     *               Expected to include a key `"ad"` with value of type `BaseAd`.
     */
    override fun loadAd(adData: Map<String, Any>, context: Context) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? BaseAd

                if (ad != null) {
                    val cliUbid = getCliUbid(ad)

                    if (isSupportedNativeFormat(ad)) {

                        withContext(Dispatchers.Main) {
                            removeAllViews()

                            val nativeAd = ad as? NativeAd
                            if (nativeAd != null) {
                                val (screenWidth, screenHeight) = getScreenHeightWidth()
                                val result = setMaxHeightWidth(
                                    height ?: screenHeight,
                                    width ?: screenWidth,
                                    null
                                )

                                val view = nativeLoader.renderNativeAdHorizontalView(
                                    context = context,
                                    cliUbid = cliUbid,
                                    nativeAd = nativeAd,
                                    width = result.finalWidth,
                                    height = result.finalHeight,
                                    customStyle = customStyle?: NativeAdCustomStyle(),
                                    customCtaView = customCtaView,
                                    customBadgeView = customBadgeView,
                                    adLabelText = adLabelText,
                                    adLabelAlignment = adLabelAlignment,
                                    adClickedListener = { adClickedListener?.invoke(nativeAd.toMap()) },
                                    onViewLoadListener = {
                                        onViewLoadListener?.invoke(
                                            nativeAd.toMap(),
                                            cliUbid
                                        )
                                    }
                                )
                                addView(view)

                                if (customStyle != null && customStyle?.adType == NativeAdLayoutType.Medium){
                                    showEmptyView(context)
                                }

                            } else {
                                showEmptyView(context)
                            }
                        }
                    } else {
                        withContext(Dispatchers.Main) {
                            showEmptyView(context)
                        }
                    }
                } else {
                    withContext(Dispatchers.Main) {
                        showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    showEmptyView(context)
                }
            }
        }
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {
    }

    override fun onDetachedFromWindow() {
        try {
            // Comprehensive cleanup to prevent memory leaks
            nativeLoader.cleanUp()
            nativeLoader.clearViewLoadListener()

            // Clear all views and listeners
            removeAllViews()

            // Clear callback references
            adClickedListener = null
            onViewLoadListener = null

            // Clear custom view references to prevent memory leaks
            customCtaView = null
            customBadgeView = null
            customStyle = null

            // Clear error callbacks
            clearAllErrorCallbacks()

        } catch (e: Exception) {
            // Log cleanup errors but don't throw to avoid crashing during view destruction
            // Note: errorCallback might not be accessible here, so we handle silently
        } finally {
            super.onDetachedFromWindow()
        }
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     */
    private fun clearAllErrorCallbacks() {
        nativeLoader.clearErrorCallback()
        (adTracker as? PersistentAdTracker)?.setErrorCallback(null)
        clearErrorCallback()
    }

    /**
     * Checks if the native ad format is supported.
     */
    private fun isSupportedNativeFormat(ad: BaseAd): Boolean {
        return ad is NativeAd && ad.name.isNotEmpty() && !ad.imageUrl.isNullOrEmpty()
    }

    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}