package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.View
import com.ai.osmos.ads.renderer.loaders.NativeAdLoader
import com.ai.osmos.ads.views.interfaces.NativeAdViewInterface
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: NativeAdView
 */

@SuppressLint("ViewConstructor")
internal class NativeAdView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private var contentView: View,
    attrs: AttributeSet? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config
) : BaseAdView(context, attrs, config), NativeAdViewInterface {

    private val nativeLoader = NativeAdLoader(coroutineScope, adTracker, config)

    // Initialize the view by loading ad or showing an empty placeholder
    init {
        errorCallback?.let {
            setErrorCallback(it)
            (adTracker as? PersistentAdTracker)?.setErrorCallback(it)
            nativeLoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, context) else showEmptyView(context)
    }

    /**
     * Loads and displays a native ad based on the given ad data.
     *
     * @param adData A map containing the ad object and its associated metadata.
     *               Expected to include a key `"ad"` with value of type `BaseAd`.
     */
    override fun loadAd(adData: Map<String, Any>, context: Context) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? BaseAd

                if (ad != null) {
                    val cliUbid = getCliUbid(ad)

                    if (isSupportedNativeFormat(ad)) {
                        withContext(Dispatchers.Main) {
                            removeAllViews()

                            val nativeAd = ad as? NativeAd
                            if (nativeAd != null) {

                                val view = nativeLoader.renderNativeAdView(
                                    context = context,
                                    cliUbid = cliUbid,
                                    nativeAd = nativeAd,
                                    contentView = contentView,
                                    adClickedListener = { adClickedListener?.invoke(nativeAd.toMap()) },
                                    onViewLoadListener = {
                                        onViewLoadListener?.invoke(
                                            nativeAd.toMap(),
                                            cliUbid
                                        )
                                    }
                                )
                                addView(view)
                            } else {
                                showEmptyView(context)
                            }
                        }
                    } else {
                        withContext(Dispatchers.Main) {
                            showEmptyView(context)
                        }
                    }
                } else {
                    withContext(Dispatchers.Main) {
                        showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    showEmptyView(context)
                }
            }
        }
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {
    }

    override fun onDetachedFromWindow() {
        try {
            // Comprehensive cleanup to prevent memory leaks
            nativeLoader.cleanUp()
            nativeLoader.clearViewLoadListener()
            clearAllErrorCallbacks()

            // Clear all views and listeners
            removeAllViews()

            // Clear callback references
            adClickedListener = null
            onViewLoadListener = null

        } catch (e: Exception) {
            // Log cleanup errors but don't throw to avoid crashing during view destruction
            // Note: errorCallback might not be accessible here, so we handle silently
        } finally {
            super.onDetachedFromWindow()
        }
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     */
    private fun clearAllErrorCallbacks() {
        nativeLoader.clearErrorCallback()
        (adTracker as? PersistentAdTracker)?.setErrorCallback(null)
        clearErrorCallback()
    }

    /**
     * Checks if the native ad format is supported.
     */
    private fun isSupportedNativeFormat(ad: BaseAd): Boolean {
        return ad is NativeAd && ad.name.isNotEmpty() && !ad.imageUrl.isNullOrEmpty()
    }

    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}