package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.View
import com.ai.osmos.ads.renderer.loaders.NativePDAAdLoader
import com.ai.osmos.ads.views.interfaces.PDAAdVerticalViewInterface
import com.ai.osmos.ads.views.style.PDAAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.PDAAd
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.ui.toMap
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: PDAAdVerticalView
 */
// Enum for Product list Ad Types
enum class PDAProductStyle {
    Horizontal, Vertical, Custom;
}

/**
 * PDAAdVerticalView for displaying PDA (Product Display Ad) advertisements in vertical layouts.
 *
 * @param context Application context
 * @param adData Map containing PDA ad data structure
 * @param adType Type of PDA ad (HORIZONTAL, VERTICAL)
 * @param width Width of PDA ad view in pixels
 * @param height Optional height of PDA ad view in pixels
 * @param pdaAdCustomStyle Custom styling for the PDA ad
 * @param adLabelText Optional text for ad label
 * @param adLabelAlignment Optional alignment for ad label
 * @param customBadgeView Optional custom badge view
 * @param attrs Optional attribute set for ad view styling
 * @param errorCallback Optional callback for handling errors
 * @param config SDK configuration object
 */
@SuppressLint("ViewConstructor")
internal class PDAAdVerticalView @JvmOverloads constructor(
    context: Context,
    adData: Map<String, Any>,
    private var adType: NativeAdStyle,
    private var width: Int,
    private var height: Int? = null,
    private var pdaAdCustomStyle: PDAAdCustomStyle,
    private var adLabelText: String? = null,
    private var adLabelAlignment: Int? = null,
    private var customBadgeView: View? = null,
    attrs: AttributeSet? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config
) : BaseAdView(context, attrs, config), PDAAdVerticalViewInterface {

    private val nativePDALoader = NativePDAAdLoader(coroutineScope, adTracker, config)

    // Initialize the view by loading ad or showing an empty placeholder
    init {
        errorCallback?.let {
            setErrorCallback(it)
            (adTracker as? PersistentAdTracker)?.setErrorCallback(it)
            nativePDALoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, context) else showEmptyView(context)
    }

    override fun onDetachedFromWindow() {
        // Comprehensive cleanup with defensive checks
        nativePDALoader.cleanUp()
        nativePDALoader.clearViewLoadListener()
        clearAllErrorCallbacks()

        // Clear views safely
        while (childCount > 0) {
            removeViewAt(0)
        }

        // Safe property cleanup
        adClickedListener = null
        onViewLoadListener = null
        customBadgeView = null
        // pdaAdCustomStyle = null // Cannot assign null to non-null type

        // Call super to handle coroutine cancellation and other base cleanup
        super.onDetachedFromWindow()
    }

    /**
     * Loads and displays a PDA ad based on the given ad data.
     *
     * @param adData A map containing the ad object and its associated metadata.
     *               Expected to include a key `"ad"` with value of type `BaseAd`.
     */
    override fun loadAd(adData: Map<String, Any>, context: Context) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? BaseAd

                if (ad != null) {
                    val cliUbid = getCliUbid(ad)

                    if (isSupportedFormatPDA(ad.crt)) {
                        withContext(Dispatchers.Main) {
                            removeAllViews()

                            val pdaAd = ad as? PDAAd
                            if (pdaAd != null) {
                                val (screenWidth, screenHeight) = getScreenHeightWidth()

                                val result = setMaxHeightWidth(
                                    height ?: screenHeight,
                                    width ?: screenWidth,
                                    null
                                )

                                val view = nativePDALoader.renderNativePDAAdView(
                                    context = context,
                                    cliUbid = cliUbid,
                                    pdaAd = pdaAd,
                                    adType = adType,
                                    width = result.finalWidth,
                                    height = result.finalHeight,
                                    pdaAdCustomStyle = pdaAdCustomStyle,
                                    adLabelText = adLabelText,
                                    adLabelAlignment = adLabelAlignment,
                                    customBadgeView = customBadgeView,
                                    adClickedListener = { adClickedListener?.invoke(pdaAd.toMap()) },
                                    onViewLoadListener = { adData, cliUbid ->
                                        onViewLoadListener?.invoke(
                                            adData,
                                            cliUbid
                                        )
                                    }
                                )
                                addView(view)
                            } else {
                                showEmptyView(context)
                            }
                        }
                    } else {
                        withContext(Dispatchers.Main) {
                            showEmptyView(context)
                        }
                    }
                } else {
                    withContext(Dispatchers.Main) {
                        showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    showEmptyView(context)
                }
            }
        }
    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     */
    private fun clearAllErrorCallbacks() {
        nativePDALoader.clearErrorCallback()
        (adTracker as? PersistentAdTracker)?.setErrorCallback(null)
        clearErrorCallback()
    }


    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}

