package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.renderer.loaders.PIPLoader
import com.ai.osmos.ads.views.interfaces.PIPAdViewInterface
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.error.ErrorCallback
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: PIPAdView
 *
 * PIPAdView for displaying Picture-in-Picture (PIP) video advertisements.
 *
 * @param activity Activity context for PIP functionality
 * @param adData PIP ad Data (defined data structure of PIP ad)
 * @param width(optional): Width of PIP ad view
 * @param height(optional): Height of PIP ad view
 * @param coordinates(optional): object containing the X and Y coordinates for absolute positioning of the PIP
 * @param attrs(optional): Attrs for ad view
 * @param errorCallback(optional): Callback for handling errors
 * @return PIPAdView or EmptyView
 * Example:
 * ```
 * val adView = PIPAdView(
 *     activity = this@MainActivity,
 *     adData = adData,
 *     width = 300,
 *     height = 200,
 *     coordinates = Coordinates(x = 10, y = 10),
 *     errorCallback = object : ErrorCallback {
 *         override fun onError(errorCode: String, errorMessage: String, throwable: Throwable?) {
 *             Log.e("PIPAd", "Error: $errorCode - $errorMessage")
 *         }
 *     }
 * )
 * ```
 */

@SuppressLint("ViewConstructor")
internal class PIPAdView @JvmOverloads constructor(
    activity: Activity,
    adData: Map<String, Any>,
    private var width: Int? = null,
    private var height: Int? = null,
    private var coordinates: Coordinates? = Coordinates(0, 0),
    private var adLabelText: String? = null,
    private var adLabelAlignment: Int? = null,
    attrs: AttributeSet? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config
) : BaseAdView(activity, attrs, config), PIPAdViewInterface {

    private val persistentAdTracker by lazy {
        if (activity.applicationContext != null) {
            PersistentAdTracker.create(activity.applicationContext, coroutineScope, config)
        } else {
            PersistentAdTracker.create(activity, coroutineScope, config)
        }
    }
    private val pipLoader = PIPLoader(coroutineScope, persistentAdTracker)

    init {
        errorCallback?.let {
            setErrorCallback(it)
            persistentAdTracker.setErrorCallback(it)
            // Pass error callback to PIPLoader's internal VideoLoader
            pipLoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, activity as Activity?) else showEmptyView(activity)
    }

    override fun onDetachedFromWindow() {
        pipLoader.cleanUp()
        clearAllErrorCallbacks()
        super.onDetachedFromWindow()
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     */
    private fun clearAllErrorCallbacks() {
        pipLoader.clearErrorCallback()
        persistentAdTracker.setErrorCallback(null)
        clearErrorCallback()
    }

    override fun loadAd(adData: Map<String, Any>, context: Context) {

    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? VideoAd
                if (ad != null && isSupportedVideoFormat(ad)) {
                    val cliUbid = getCliUbid(ad)

                    if (isSupportedCrtVideo(ad.crt)) {
                        val (screenWidth, screenHeight) = getScreenHeightWidth()

                        val h = height ?: (Constants.PIP_AD_HEIGHT ?: screenHeight)
                        val w = width ?: (Constants.PIP_AD_WIDTH ?: screenWidth)

                        val result = setMaxHeightWidth(h, w, coordinates)
                        coordinates = result.coordinates

                        val (resolvedWidth, resolvedHeight) = getAdDimensions(
                            result.finalWidth,
                            result.finalHeight,
                            Constants.PIP_AD_WIDTH ?: screenWidth,
                            Constants.PIP_AD_HEIGHT ?: screenHeight
                        )

                        withContext(Dispatchers.Main) {
                            if (activity != null) {
                                pipLoader.showPIP(
                                    activity = activity,
                                    ad = ad,
                                    cliUbid = cliUbid,
                                    width = resolvedWidth,
                                    height = resolvedHeight,
                                    coordinates = coordinates,
                                    adLabelText = adLabelText,
                                    adLabelAlignment = adLabelAlignment,
                                    result.isClamped,
                                    adClickListener = adClickedListener,
                                    onViewLoadListener = onViewLoadListener
                                )
                            }
                        }
                    } else {
                        withContext(Dispatchers.Main) {
                            activity?.let { showEmptyView(it) } ?: showEmptyView(context)
                        }
                    }

                } else {
                    withContext(Dispatchers.Main) {
                        activity?.let { showEmptyView(it) } ?: showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    activity?.let { showEmptyView(it) } ?: showEmptyView(context)
                }
            }
        }
    }

    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}