package com.ai.osmos.ads.views

import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import com.ai.osmos.ads.renderer.loaders.SliderLoader
import com.ai.osmos.ads.views.interfaces.SliderAdViewInterface
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.BaseAd
import com.ai.osmos.models.ads.ImageAd
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.tracking.tracker.PersistentAdTracker
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.error.ErrorCallback
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: SliderAdView
 *
 * SliderAdView for displaying sliding advertisements with positioning control.
 *
 * @param activity Activity context for slider functionality
 * @param adData Slider ad Data (defined data structure of slider ad)
 * @param width(optional): Width of slider ad view
 * @param height(optional): Height of slider ad view
 * @param coordinates(optional): object containing the X and Y coordinates for absolute positioning of the slider
 * @param attrs(optional): Attrs for ad view
 * @param errorCallback(optional): Callback for handling errors
 * @return SliderAdView or EmptyView
 * Example:
 * ```
 * val adView = SliderAdView(
 *     activity = this@MainActivity,
 *     adData = adData,
 *     width = 300,
 *     height = 200,
 *     coordinates = Coordinates(x = 10, y = 10),
 *     errorCallback = object : ErrorCallback {
 *         override fun onError(errorCode: String, errorMessage: String, throwable: Throwable?) {
 *             Log.e("SliderAd", "Error: $errorCode - $errorMessage")
 *         }
 *     }
 * )
 * ```
 */

@SuppressLint("ViewConstructor")
internal class SliderAdView @JvmOverloads constructor(
    activity: Activity,
    adData: Map<String, Any>,
    private var width: Int? = null,
    private var height: Int? = null,
    private var coordinates: Coordinates? = Coordinates(0, 0),
    private var adLabelText: String? = null,
    private var adLabelAlignment: Int? = null,
    attrs: AttributeSet? = null,
    errorCallback: ErrorCallback? = null,
    private val config: Config
) : BaseAdView(activity, attrs, config), SliderAdViewInterface {

    private val persistentAdTracker by lazy {
        if (activity.applicationContext != null) {
            PersistentAdTracker.create(activity.applicationContext, coroutineScope, config)
        } else {
            PersistentAdTracker.create(activity, coroutineScope, config)
        }
    }
    private val sliderLoader = SliderLoader(coroutineScope, persistentAdTracker)

    init {
        errorCallback?.let {
            setErrorCallback(it)
            persistentAdTracker.setErrorCallback(it)
            // Pass error callback to SliderLoader's internal loaders
            sliderLoader.setErrorCallback(it)
        }
        if (adData.isNotEmpty()) loadAd(adData, activity as Activity?) else showEmptyView(activity)
    }

    override fun onDetachedFromWindow() {
        sliderLoader.cleanUp()
        clearAllErrorCallbacks()
        super.onDetachedFromWindow()
    }

    /**
     * Clears all error callbacks to prevent memory leaks.
     */
    private fun clearAllErrorCallbacks() {
        sliderLoader.clearErrorCallback()
        persistentAdTracker.setErrorCallback(null)
        clearErrorCallback()
    }

    override fun loadAd(adData: Map<String, Any>, context: Context) {

    }

    override fun loadAd(adData: Map<String, Any>, activity: Activity?) {
        coroutineScope.launch(Dispatchers.IO) {
            try {
                val ad = adData["ad"] as? BaseAd
                if (ad != null) {
                    val cliUbid = getCliUbid(ad)

                    when {
                        ad is VideoAd && isSupportedVideoFormat(ad) -> {
                            if (isSupportedCrtVideo(ad.crt)) {
                                val (screenWidth, screenHeight) = getScreenHeightWidth()

                                val h = height ?: (Constants.SLIDER_AD_HEIGHT ?: screenHeight)
                                val w = width ?: (Constants.SLIDER_AD_WIDTH ?: screenWidth)

                                val result = setMaxHeightWidth(h, w, coordinates)
                                coordinates = result.coordinates

                                val (resolvedWidth, resolvedHeight) = getAdDimensions(
                                    result.finalWidth,
                                    result.finalHeight,
                                    Constants.SLIDER_AD_WIDTH ?: screenWidth,
                                    Constants.SLIDER_AD_HEIGHT ?: screenHeight
                                )

                                withContext(Dispatchers.Main) {
                                    if (activity != null) {
                                        sliderLoader.showVideoSlider(
                                            activity = activity,
                                            videoAd = ad,
                                            cliUbid = cliUbid,
                                            width = resolvedWidth,
                                            height = resolvedHeight,
                                            coordinates = coordinates,
                                            adLabelText = adLabelText,
                                            adLabelAlignment = adLabelAlignment,
                                            result.isClamped,
                                            adClickListener = adClickedListener,
                                            onViewLoadListener = onViewLoadListener
                                        )
                                    }
                                }
                            } else {
                                withContext(Dispatchers.Main) {
                                    activity?.let { showEmptyView(it) } ?: showEmptyView(context)
                                }
                            }
                        }

                        ad is ImageAd && isSupportedImageFormat(ad) -> {
                            if (isSupportedCrtImage(ad.crt)) {
                                val (screenWidth, screenHeight) = getScreenHeightWidth()

                                val h = height ?: (Constants.SLIDER_AD_HEIGHT ?: screenHeight)
                                val w = width ?: (Constants.SLIDER_AD_WIDTH ?: screenWidth)

                                val result = setMaxHeightWidth(h, w, coordinates)
                                coordinates = result.coordinates

                                val (resolvedWidth, resolvedHeight) = getAdDimensions(
                                    result.finalWidth,
                                    result.finalHeight,
                                    Constants.SLIDER_AD_WIDTH ?: screenWidth,
                                    Constants.SLIDER_AD_HEIGHT ?: screenHeight
                                )

                                withContext(Dispatchers.Main) {
                                    if (activity != null) {
                                        sliderLoader.showImageSlider(
                                            activity = activity,
                                            imageAd = ad,
                                            cliUbid = cliUbid,
                                            width = resolvedWidth,
                                            height = resolvedHeight,
                                            coordinates = coordinates,
                                            adLabelText = adLabelText,
                                            adLabelAlignment = adLabelAlignment,
                                            result.isClamped,
                                            adClickListener = adClickedListener
                                        )
                                    }
                                }
                            } else {
                                withContext(Dispatchers.Main) {
                                    activity?.let { showEmptyView(it) } ?: showEmptyView(context)
                                }
                            }
                        }

                        else -> {
                            withContext(Dispatchers.Main) {
                                activity?.let { showEmptyView(it) } ?: showEmptyView(context)
                            }
                        }
                    }
                } else {
                    withContext(Dispatchers.Main) {
                        activity?.let { showEmptyView(it) } ?: showEmptyView(context)
                    }
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    activity?.let { showEmptyView(it) } ?: showEmptyView(context)
                }
            }
        }
    }

    /**
     * Returns the underlying View for adding to view hierarchy
     */
    override fun getView(): View = this
}