package com.ai.osmos.ads.views.adapters

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.ViewTreeObserver
import android.widget.FrameLayout
import android.widget.LinearLayout
import android.widget.TextView
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.ai.osmos.ads.renderer.BaseNativeAdView
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.enums.NativeAdLayoutType
import com.ai.osmos.models.enums.NativeAdStyle
import com.ai.osmos.utils.ui.ViewUtils
import com.ai.osmos.utils.ui.isVisibleInScreen
import com.ai.osmos.utils.ui.toMap
import kotlin.math.min

/**
 * Project Name: OSMOS-Android-SDK
 * File Name: PLAProductListAdapter
 */
internal class NativeAdCarouselAdapter(
    private val context: Context,
    private val adType: NativeAdStyle,
    private val width: Int,
    private val height: Int,
    private var cardElementWidth: Int? = null,
    private val nativeAdList: List<NativeAd>,
    private val adLabelText: String?,
    private val adLabelAlignment: Int?,
    private val customStyle: NativeAdCustomStyle?,
    private var customCtaView: View?,
    private var customBadgeView: View?,
    private var adClickedListener: ((adData: Map<String, Any>) -> Unit)?,
    private var onViewLoadListener: ((adData: Map<String, Any>, cliUbid: String) -> Unit)?,
    private var baseNativeAdView: BaseNativeAdView?,
    private val dotIndicatorContainer: LinearLayout,
    private val trackImpression: ((uclid: String, cliUbid: String) -> Unit)?
) : RecyclerView.Adapter<NativeAdCarouselAdapter.NativeAdViewHolder>() {

    private var currentDotIndex = 0
    private val impressionTrackedSet = mutableSetOf<Int>() // Track which positions have fired impressions
    private val preDrawListenerMap: MutableMap<View, ViewTreeObserver.OnPreDrawListener> = mutableMapOf() // Track PreDrawListeners for cleanup

    init {
        setupDotIndicators()
    }

    inner class NativeAdViewHolder(val container: FrameLayout) :
        RecyclerView.ViewHolder(container)

    /**
     * Calculate card width so max 2 cards fit in carousel.
     * Width is already constrained by BaseNativeAdView.
     */
    private fun calculateCardWidth(): Pair<Int, Int> {
        // Use the width directly as it's already constrained in BaseNativeAdView
        val carouselWidth = width

        // Always 2 cards on screen
        val cardsPerScreen = 2
        val spacing = ViewUtils.dpToPx(context, 8)

        // Only 1 gap between 2 cards (not left + right margins)
        val totalSpacing = spacing * (cardsPerScreen - 1)
        val cardWidth = (carouselWidth - totalSpacing) / cardsPerScreen

        return Pair(carouselWidth, cardWidth)
    }

    /**
     * Setup dot indicators based on the number of ads
     */
    private fun setupDotIndicators() {
        dotIndicatorContainer.removeAllViews()

        for (i in nativeAdList.indices) {
            val dot = TextView(context).apply {
                text = "•"
                textSize = 20f
                setTextColor(if (i == currentDotIndex) Color.WHITE else Color.GRAY)
                layoutParams = LinearLayout.LayoutParams(
                    LinearLayout.LayoutParams.WRAP_CONTENT,
                    LinearLayout.LayoutParams.WRAP_CONTENT
                ).apply {
                    setMargins(ViewUtils.dpToPx(context, 4), 0, ViewUtils.dpToPx(context, 4), 0)
                }
            }
            dotIndicatorContainer.addView(dot)
        }
    }

    /**
     * Update dot indicators to show current position
     */
    fun updateDotIndicator(position: Int) {
        if (position == currentDotIndex) return

        currentDotIndex = position
        for (i in 0 until dotIndicatorContainer.childCount) {
            val dot = dotIndicatorContainer.getChildAt(i) as? TextView
            dot?.setTextColor(if (i == currentDotIndex) Color.WHITE else Color.GRAY)
        }
    }

    /**
     * Setup scroll listener for RecyclerView to update dot indicator
     */
    fun setupScrollListener(recyclerView: RecyclerView) {
        recyclerView.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrolled(recyclerView: RecyclerView, dx: Int, dy: Int) {
                super.onScrolled(recyclerView, dx, dy)

                val layoutManager = recyclerView.layoutManager as? LinearLayoutManager
                layoutManager?.let {
                    // Update dot indicator for completely visible items
                    val visiblePosition = it.findFirstCompletelyVisibleItemPosition()
                    if (visiblePosition != RecyclerView.NO_POSITION) {
                        updateDotIndicator(visiblePosition)
                    }
                }
            }
        })
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): NativeAdViewHolder {
        val spacing = ViewUtils.dpToPx(context, 8)
        val (carouselWidth, cardWidth) = calculateCardWidth()
        val container = FrameLayout(context).apply {
            layoutParams = RecyclerView.LayoutParams(
                cardWidth,
                ViewGroup.LayoutParams.WRAP_CONTENT
            ).apply {
                // Only apply margin to the right side (gap between cards)
                rightMargin = spacing
                topMargin = spacing
                bottomMargin = spacing
            }
        }

        return NativeAdViewHolder(container)
    }

    override fun getItemCount(): Int {
        return nativeAdList.size
    }

    override fun onBindViewHolder(holder: NativeAdViewHolder, position: Int) {
        val currentPosition = holder.bindingAdapterPosition
        if (currentPosition == RecyclerView.NO_POSITION) return

        val item = nativeAdList[currentPosition]
        val (_, cardWidth) = calculateCardWidth()

        // Convert cardWidth from pixels back to DP for the renderer
        val cardWidthInDp = (cardWidth / context.resources.displayMetrics.density).toInt()

        val nativeAdView: View = when (adType) {
            NativeAdStyle.Vertical -> {
                baseNativeAdView?.createVerticalAdView(
                    context = context,
                    adData = item,
                    width = cardWidthInDp,
                    height = height,
                    adLabelText = adLabelText,
                    adLabelAlignment = adLabelAlignment,
                    style = customStyle,
                    customCtaView = customCtaView,
                    customBadgeView = customBadgeView,
                    isCarouselView = true
                )
            }

            NativeAdStyle.Horizontal -> {
                baseNativeAdView?.createHorizontalAdView(
                    context = context,
                    adData = item,
                    width = cardWidthInDp,
                    height = height,
                    adLabelText = adLabelText,
                    adLabelAlignment = adLabelAlignment,
                    style = customStyle,
                    customCtaView = customCtaView,
                    customBadgeView = customBadgeView,
                    isCarouselView = true
                )
            }
        } ?: return // Return early if baseNativeAdView is null

        nativeAdView.setOnClickListener {
            adClickedListener?.invoke(item.toMap())
        }
        nativeAdView.let {
            (it.parent as? ViewGroup)?.removeView(it)
            holder.container.removeAllViews()
            holder.container.addView(it)
        }

        // Add impression tracking using PreDrawListener (like VideoLoader)
        if (!impressionTrackedSet.contains(currentPosition)) {
            val preDrawListener = object : ViewTreeObserver.OnPreDrawListener {
                override fun onPreDraw(): Boolean {
                    if (nativeAdView.isVisibleInScreen()) {
                        // Mark as tracked and fire impression
                        impressionTrackedSet.add(currentPosition)
                        val adItem = nativeAdList[currentPosition]
                        adItem.cliUbid?.let { cliUbid ->
                            // Fire impression tracking (same as VideoLoader)
                            adItem.uclid?.let { uclid ->
                                trackImpression?.invoke(uclid, cliUbid)
                            }
                            // Fire view load listener callback
                            onViewLoadListener?.invoke(adItem.toMap(), cliUbid)
                        }

                        // Remove listener after firing (one-time only)
                        removePreDrawListener(nativeAdView)
                    }
                    return true
                }
            }
            nativeAdView.viewTreeObserver.addOnPreDrawListener(preDrawListener)
            preDrawListenerMap[nativeAdView] = preDrawListener // Track for cleanup
        }
    }

    override fun onViewRecycled(holder: NativeAdViewHolder) {
        super.onViewRecycled(holder)
        // Clear PreDrawListeners from recycled views to prevent memory leaks
        for (i in 0 until holder.container.childCount) {
            val childView = holder.container.getChildAt(i)
            removePreDrawListener(childView)
        }
        // Clear any click listeners to prevent memory leaks
        holder.container.removeAllViews()
        holder.container.setOnClickListener(null)
    }

    override fun onDetachedFromRecyclerView(recyclerView: RecyclerView) {
        super.onDetachedFromRecyclerView(recyclerView)
        // Clear all references when adapter is detached
        cleanup()
    }

    /**
     * Remove PreDrawListener from view and tracking map
     */
    private fun removePreDrawListener(view: View) {
        preDrawListenerMap.remove(view)?.let { listener ->
            try {
                view.viewTreeObserver.removeOnPreDrawListener(listener)
            } catch (e: Exception) {
                // ViewTreeObserver might be dead, ignore
            }
        }
    }

    /**
     * Clean up all references to prevent memory leaks
     */
    fun cleanup() {
        // Clean up all PreDrawListeners
        preDrawListenerMap.keys.forEach { view ->
            removePreDrawListener(view)
        }
        preDrawListenerMap.clear()

        customCtaView = null
        customBadgeView = null
        adClickedListener = null
        onViewLoadListener = null
        baseNativeAdView = null
    }
}