package com.ai.osmos.ads.views.helpers

import android.content.Context
import android.graphics.Color
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.LinearLayout
import com.ai.osmos.ads.renderer.loaders.ImageLoader
import com.ai.osmos.ads.renderer.loaders.VideoLoader
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.AdElement
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.models.enums.NativeAdLayoutType
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import kotlinx.coroutines.CoroutineScope

/**
 * Internal renderer for horizontal native ad layouts.
 * Contains the extracted horizontal rendering logic from BaseNativeAdView.
 */
internal class HorizontalNativeAdViewRenderer(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface,
    private val config: Config
) : NativeAdViewRenderer {

    private val imageLoader = ImageLoader(coroutineScope, adTracker)
    private val videoLoader = VideoLoader(coroutineScope, adTracker)
    private val viewHelper = NativeAdViewHelper()
    private var errorCallback: ErrorCallback? = null

    override fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
        imageLoader.setErrorCallback(callback)
        videoLoader.setErrorCallback(callback)
    }

    override fun clearErrorCallback() {
        this.errorCallback = null
        imageLoader.clearErrorCallback()
        videoLoader.clearErrorCallback()
    }

    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    override fun renderVerticalView(
        context: Context,
        adData: NativeAd,
        width: Int,
        height: Int?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        isCarouselView: Boolean
    ): View {
        // This renderer only handles horizontal views
        throw UnsupportedOperationException("HorizontalNativeAdViewRenderer does not support vertical rendering")
    }

    override fun renderHorizontalView(
        context: Context,
        adData: NativeAd,
        width: Int,
        height: Int?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        isCarouselView: Boolean
    ): View {
        val (screenWidth, screenHeight) = ViewUtils.getScreenHeightWidth(0.8f, 0.8f)
        val finalWidth = minOf(ViewUtils.dpToPx(context, width), screenWidth)

        val h: Int? = if (height != null) {
            ViewUtils.dpToPx(context, height)
        } else if (style?.mediaHeight != null) {
            ViewUtils.dpToPx(context, style.mediaHeight)
        } else {
            null
        }
        val finalHeight = minOf(h ?: screenHeight, screenWidth)

        val mainContainer = FrameLayout(context).apply {
            layoutParams = ViewGroup.LayoutParams(
                finalWidth,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
        }
        val container = LinearLayout(context).apply {
            orientation = LinearLayout.HORIZONTAL
            setBackgroundColor(Color.WHITE)
            setPadding(12, 12, 12, 12)
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT,
                FrameLayout.LayoutParams.MATCH_PARENT
            )
        }

        // Right content container
        val imageContentLayout = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                0,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                1.6f
            )
            gravity = Gravity.CENTER
            setPadding(0, 0, 10, 0)
        }

        if (adData.imageUrl != null) {
            val imageView = viewHelper.loadImageView(context, adData, imageLoader)
            imageContentLayout.addView(imageView)
        } else {
            val videoAd = VideoAd(
                au = "",
                rank = 3,
                clickTrackingUrl = "",
                impressionTrackingUrl = "",
                uclid = adData.uclid ?: "",
                crt = "",
                elements = AdElement(
                    type = "VIDEO",
                    height = ViewGroup.LayoutParams.WRAP_CONTENT,
                    width = ViewGroup.LayoutParams.WRAP_CONTENT,
                    value = adData.videoUrl ?: "",
                ),
                adMetadata = emptyMap(),
                clientId = (config.clientId).toLong()
            )

            val videoView = videoLoader.createVideoViewForNativeAd(
                context,
                videoAd,
                finalWidth,
                finalHeight,
                adData.cliUbid.toString()
            )
            imageContentLayout.addView(videoView)
        }

        container.addView(imageContentLayout)

        val topTagRow = viewHelper.loadTagBudgeView(context, style, customBadgeView)

        // Create view components
        val titleView = viewHelper.loadTitleView(context, adData, style)
        val skuDescriptionView = viewHelper.loadSkuDescriptionView(context, adData, style)
        val deliveryDetailsView = viewHelper.loadDeliveryDetailView(context, adData, style)
        val priceRow = viewHelper.loadPriceRowView(context, adData, style)
        val ratingRow = viewHelper.loadRatingView(context, adData, style)
        val membershipPrice = viewHelper.loadMembershipView(context, adData, style)

        val contentWrapper = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                0,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                1.9f
            )
        }
        val estimatedButtonHeight = ViewUtils.dpToPx(context, 48 + 8) // button + margin
        val availableHeightForContent = finalHeight - estimatedButtonHeight

        val contentLayout = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                2f // take all space above button
            )
        }

        val buttonLayout = LinearLayout(context).apply {
            orientation = LinearLayout.HORIZONTAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                estimatedButtonHeight
            ).apply {
                topMargin = ViewUtils.dpToPx(context, 8)
            }
            gravity = Gravity.CENTER_HORIZONTAL
        }

        // Add views on container
        val nativeStyle = style ?: NativeAdCustomStyle()
        when (nativeStyle.adType) {
            NativeAdLayoutType.Small -> {
                contentLayout.addViewsInOrder(
                    titleView,
                    priceRow,
                    ratingRow
                )

                viewHelper.addCTAButton(context, adData, customCtaView, style, buttonLayout, isCarouselView)

                contentWrapper.addView(contentLayout)
                contentWrapper.addView(buttonLayout)
                container.addView(contentWrapper)
                mainContainer.addView(container)
                mainContainer.addView(topTagRow)
                mainContainer.addView(viewHelper.setAdLabel(context, adLabelText, adLabelAlignment))
            }

            NativeAdLayoutType.Large -> {
                contentLayout.addViewsInOrder(
                    titleView,
                    skuDescriptionView,
                    deliveryDetailsView,
                    priceRow,
                    ratingRow,
                    membershipPrice
                )

                viewHelper.addCTAButton(context, adData, customCtaView, style, buttonLayout, isCarouselView)

                contentWrapper.addView(contentLayout)
                contentWrapper.addView(buttonLayout)
                container.addView(contentWrapper)
                mainContainer.addView(container)
                mainContainer.addView(topTagRow)
                mainContainer.addView(viewHelper.setAdLabel(context, adLabelText, adLabelAlignment))
            }
            NativeAdLayoutType.Medium -> {
            }
        }

        return mainContainer
    }

    override fun renderProductVerticalView(
        context: Context,
        adData: NativeAd,
        adType: NativeAdLayoutType,
        width: Int,
        height: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?
    ): View {
        // This renderer only handles horizontal views
        throw UnsupportedOperationException("HorizontalNativeAdViewRenderer does not support vertical rendering")
    }

    override fun renderProductHorizontalView(
        context: Context,
        adData: NativeAd,
        adType: NativeAdLayoutType,
        width: Int,
        height: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?
    ): View {
        val (screenWidth, screenHeight) = ViewUtils.getScreenHeightWidth(0.8f, 0.8f)
        val finalWidth = minOf(ViewUtils.dpToPx(context, width), screenWidth)

        val h: Int? = if (height != null) {
            ViewUtils.dpToPx(context, height)
        } else if (style?.mediaHeight != null) {
            ViewUtils.dpToPx(context, style.mediaHeight)
        } else {
            null
        }
        val finalHeight = minOf(h ?: screenHeight, screenWidth)

        val mainContainer = FrameLayout(context).apply {
            layoutParams = ViewGroup.LayoutParams(
                finalWidth,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
        }
        val container = LinearLayout(context).apply {
            orientation = LinearLayout.HORIZONTAL
            setBackgroundColor(Color.WHITE)
            setPadding(12, 12, 12, 12)
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT,
                FrameLayout.LayoutParams.MATCH_PARENT
            )
        }

        // Right content container
        val imageContentLayout = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                0,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                1.6f
            )
            gravity = Gravity.CENTER
            setPadding(0, 0, 10, 0)
        }

        if (adData.imageUrl != null) {
            val imageView = imageLoader.createImageView(
                context,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                isClamped = false
            ).apply {
                layoutParams = LinearLayout.LayoutParams(
                    ViewGroup.LayoutParams.WRAP_CONTENT,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                ).apply {
                    gravity = Gravity.CENTER
                }
            }
            imageLoader.loadNativeImage(adData.imageUrl.toString(), imageView)
            imageContentLayout.addView(imageView)
        } else {
            val videoAd = VideoAd(
                au = "",
                rank = 0,
                clickTrackingUrl = "",
                impressionTrackingUrl = "",
                uclid = adData.uclid ?: "",
                crt = "",
                elements = AdElement(
                    type = "VIDEO",
                    height = ViewGroup.LayoutParams.WRAP_CONTENT,
                    width = ViewGroup.LayoutParams.WRAP_CONTENT,
                    value = adData.videoUrl ?: "",
                ),
                adMetadata = emptyMap(),
                clientId = (config.clientId).toLong()
            )

            val videoView = videoLoader.createVideoViewForNativeAd(
                context,
                videoAd,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                adData.cliUbid.toString()
            )
            imageContentLayout.addView(videoView)
        }

        container.addView(imageContentLayout)

        // Create view components
        val titleView = viewHelper.loadTitleView(context, adData, style)
        val skuDescriptionView = viewHelper.loadSkuDescriptionView(context, adData, style)
        val deliveryDetailsView = viewHelper.loadDeliveryDetailView(context, adData, style)
        val priceRow = viewHelper.loadPriceRowView(context, adData, style)
        val ratingRow = viewHelper.loadRatingView(context, adData, style)
        val membershipPrice = viewHelper.loadMembershipView(context, adData, style)

        val contentWrapper = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                0,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                1.9f
            )
        }
        val estimatedButtonHeight = ViewUtils.dpToPx(context, 48 + 8) // button + margin
        val availableHeightForContent = finalHeight - estimatedButtonHeight

        val contentLayout = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                2f // take all space above button
            )
        }

        val buttonLayout = LinearLayout(context).apply {
            orientation = LinearLayout.HORIZONTAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                estimatedButtonHeight
            ).apply {
                topMargin = ViewUtils.dpToPx(context, 8)
            }
            gravity = Gravity.CENTER_HORIZONTAL
        }

        // Add views on container based on adType parameter
        when (adType) {
            NativeAdLayoutType.Small -> {
                contentLayout.addViewsInOrder(
                    titleView,
                    priceRow,
                    ratingRow
                )

                viewHelper.addCTAButton(context, adData, customCtaView, style, buttonLayout, true)

                contentWrapper.addView(contentLayout)
                contentWrapper.addView(buttonLayout)
                container.addView(contentWrapper)
                mainContainer.addView(container)

                return mainContainer
            }

            NativeAdLayoutType.Large -> {
                contentLayout.addViewsInOrder(
                    titleView,
                    skuDescriptionView,
                    deliveryDetailsView,
                    priceRow,
                    ratingRow,
                    membershipPrice
                )

                viewHelper.addCTAButton(context, adData, customCtaView, style, buttonLayout, true)

                contentWrapper.addView(contentLayout)
                contentWrapper.addView(buttonLayout)
                container.addView(contentWrapper)
                mainContainer.addView(container)

                return mainContainer
            }

            NativeAdLayoutType.Medium -> {
            }
        }
        return mainContainer
    }

    // Extension function to add views in order, skipping null views
    private fun LinearLayout.addViewsInOrder(vararg views: View?) {
        views.forEach { view ->
            view?.let { addView(it) }
        }
    }
}