package com.ai.osmos.ads.views.helpers

import com.ai.osmos.core.Config
import com.ai.osmos.models.enums.NativeAdLayoutType
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import kotlinx.coroutines.CoroutineScope

/**
 * Factory class for creating native ad view renderers based on layout type.
 * Uses the Strategy pattern to provide the appropriate renderer.
 */
internal class NativeAdViewFactory(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface,
    private val config: Config
) {

    // Cache renderers to avoid recreating them unnecessarily
    private val verticalRenderer by lazy {
        VerticalNativeAdViewRenderer(coroutineScope, adTracker, config)
    }

    private val horizontalRenderer by lazy {
        HorizontalNativeAdViewRenderer(coroutineScope, adTracker, config)
    }

    /**
     * Creates and returns the appropriate renderer based on layout type.
     * For now, this always returns vertical renderer but can be enhanced
     * to determine layout type based on requirements.
     *
     * @param layoutType The layout type (currently not used but ready for future enhancement)
     * @return NativeAdViewRenderer instance for the specified layout
     */
    fun createRenderer(layoutType: NativeAdLayoutType): NativeAdViewRenderer {
        // For now, always return vertical renderer since the original implementation
        // used createVerticalAdView and createHorizontalAdView as separate methods
        // This can be enhanced later to actually switch based on layoutType
        return verticalRenderer
    }

    /**
     * Get vertical renderer specifically.
     */
    fun getVerticalRenderer(): NativeAdViewRenderer {
        return verticalRenderer
    }

    /**
     * Get horizontal renderer specifically.
     */
    fun getHorizontalRenderer(): NativeAdViewRenderer {
        return horizontalRenderer
    }

    /**
     * Set error callback for all renderers.
     */
    fun setErrorCallback(callback: com.ai.osmos.utils.error.ErrorCallback?) {
        verticalRenderer.setErrorCallback(callback)
        horizontalRenderer.setErrorCallback(callback)
    }

    /**
     * Clear error callbacks for all renderers.
     */
    fun clearErrorCallbacks() {
        verticalRenderer.clearErrorCallback()
        horizontalRenderer.clearErrorCallback()
    }
}