package com.ai.osmos.ads.views.helpers

import android.content.Context
import android.graphics.Paint
import android.graphics.drawable.GradientDrawable
import android.text.TextUtils
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.Button
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.RelativeLayout
import android.widget.TextView
import com.ai.osmos.ads.renderer.loaders.ImageLoader
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.ads.views.style.TextStyle
import com.ai.osmos.ads.views.style.defaultDeliveryDetailStyle
import com.ai.osmos.ads.views.style.defaultDiscountedPriceStyle
import com.ai.osmos.ads.views.style.defaultMembershipStyle
import com.ai.osmos.ads.views.style.defaultOfferStyle
import com.ai.osmos.ads.views.style.defaultOriginalPriceStyle
import com.ai.osmos.ads.views.style.defaultRatingCountStyle
import com.ai.osmos.ads.views.style.defaultRatingStyle
import com.ai.osmos.ads.views.style.defaultSkuDescriptionStyle
import com.ai.osmos.ads.views.style.defaultTitleStyle
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.enums.NativeAdLayoutType
import com.ai.osmos.utils.common.Constants
import com.ai.osmos.utils.ui.ViewUtils

/**
 * Helper class for creating native ad view components.
 * Contains all the UI element creation functions extracted from BaseNativeAdView.
 */
internal class NativeAdViewHelper {

    fun loadImageView(context: Context, adData: NativeAd, imageLoader: ImageLoader): ImageView {
        val imageView = imageLoader.createImageView(
            context,
            ViewGroup.LayoutParams.MATCH_PARENT,
            ViewGroup.LayoutParams.WRAP_CONTENT,
            isClamped = false
        )
        adData.imageUrl?.let { imageUrl ->
            imageLoader.loadNativeImage(imageUrl, imageView)
        }
        return imageView
    }

    private fun cloneButtonView(original: Button): Button {
        return Button(original.context).apply {
            text = original.text
            textSize = original.textSize / resources.displayMetrics.scaledDensity
            setTextColor(original.currentTextColor)
            background = original.background
            setPadding(
                original.paddingLeft,
                original.paddingTop,
                original.paddingRight,
                original.paddingBottom
            )
            layoutParams = original.layoutParams ?: LinearLayout.LayoutParams(
                LinearLayout.LayoutParams.WRAP_CONTENT,
                LinearLayout.LayoutParams.WRAP_CONTENT
            )
            setOnClickListener { original.performClick() } // Delegate click
        }
    }

    fun addCTAButton(
        context: Context,
        adData: NativeAd,
        customCtaView: View?,
        style: NativeAdCustomStyle?,
        container: LinearLayout,
        isPDAAds: Boolean
    ) {
        if (customCtaView != null && customCtaView is Button) {
            val layoutParams = LinearLayout.LayoutParams(
                LinearLayout.LayoutParams.WRAP_CONTENT,
                LinearLayout.LayoutParams.MATCH_PARENT
            ).apply {
                gravity = Gravity.CENTER_HORIZONTAL
            }
            customCtaView.layoutParams = layoutParams

            // Remove from parent if already attached to prevent "child already has parent" error
            (customCtaView.parent as? ViewGroup)?.removeView(customCtaView)

            if (isPDAAds) {
                val copyButton = cloneButtonView(customCtaView)
                copyButton.layoutParams = layoutParams
                container.addView(copyButton)
            } else {
                (customCtaView.parent as? ViewGroup)?.removeView(customCtaView)
                customCtaView.layoutParams = layoutParams
                container.addView(customCtaView)
            }
        } else {
            if (style?.ctaTitle != null && style.ctaAction != null) {
                val ctaButton = Button(context).apply {
                    text = style.ctaTitle
                    setPadding(20, 10, 20, 10)
                    setOnClickListener { style.ctaAction.invoke() }
                }
                container.addView(ctaButton)
            }
        }
    }


    fun loadTagBudgeView(
        context: Context,
        customStyle: NativeAdCustomStyle?,
        customBadgeView: View?
    ): RelativeLayout {
        val topTagRow = RelativeLayout(context).apply {
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
            setPadding(10, 5, 10, 8)
        }

        if (customBadgeView != null) {
            if (customBadgeView is TextView) {
                customBadgeView.layoutParams = RelativeLayout.LayoutParams(
                    ViewGroup.LayoutParams.WRAP_CONTENT,
                    ViewGroup.LayoutParams.WRAP_CONTENT,
                ).apply {
                    addRule(RelativeLayout.ALIGN_PARENT_END)
                    addRule(RelativeLayout.ALIGN_PARENT_TOP)
                    addRule(RelativeLayout.ALIGN_PARENT_RIGHT)
                }
            } else {
                customBadgeView.layoutParams = RelativeLayout.LayoutParams(
                    ViewUtils.dpToPx(context, 50),
                    ViewUtils.dpToPx(context, 30)
                ).apply {
                    addRule(RelativeLayout.ALIGN_PARENT_END)
                    addRule(RelativeLayout.ALIGN_PARENT_TOP)
                    addRule(RelativeLayout.ALIGN_PARENT_RIGHT)
                }
            }

            if (customStyle?.adType != NativeAdLayoutType.Small) topTagRow.addView(customBadgeView)
        }
        return topTagRow
    }

    fun loadTitleView(context: Context, adData: NativeAd, style: NativeAdCustomStyle?): TextView {
        return TextView(context).apply {
            text = adData.name
            applyTextStyle(style?.titleTextStyle, defaultTitleStyle)
            maxLines = 2
            ellipsize = TextUtils.TruncateAt.END
        }
    }

    fun loadSkuDescriptionView(
        context: Context,
        adData: NativeAd,
        style: NativeAdCustomStyle?
    ): TextView {
        return TextView(context).apply {
            text = adData.description
            applyTextStyle(style?.skuDescriptionTextStyle, defaultSkuDescriptionStyle)
            maxLines = 2
            ellipsize = TextUtils.TruncateAt.END
        }
    }

    fun loadDeliveryDetailView(
        context: Context,
        adData: NativeAd,
        style: NativeAdCustomStyle?
    ): TextView {
        return TextView(context).apply {
            text = adData.customLabel2
            applyTextStyle(style?.deliveryDetailTextStyle, defaultDeliveryDetailStyle)
        }
    }

    fun loadPriceRowView(context: Context, adData: NativeAd, style: NativeAdCustomStyle?): View {
        val originalPriceView = TextView(context).apply {
            text = "${Constants.CURRENCY_SYMBOL}${adData.mrp}"
            paintFlags = paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
            maxLines = 1
            isSingleLine = true
            ellipsize = null
            setHorizontallyScrolling(false)
            applyTextStyle(style?.originalPriceTextStyle, defaultOriginalPriceStyle)
        }

        val discountedPriceView = TextView(context).apply {
            text = "${Constants.CURRENCY_SYMBOL}${adData.salePrice}"
            maxLines = 1
            isSingleLine = true
            ellipsize = null
            setHorizontallyScrolling(false)
            applyTextStyle(style?.discountedPriceTextStyle, defaultDiscountedPriceStyle)
        }

        val offerPercentage = if (adData.mrp > 0 && adData.salePrice!! < adData.mrp) {
            val discount = ((adData.mrp - adData.salePrice) / adData.mrp) * 100
            "${discount.toInt()}%"
        } else {
            ""
        }

        val offerView = TextView(context).apply {
            text = offerPercentage
            maxLines = 1
            isSingleLine = true
            ellipsize = null
            setHorizontallyScrolling(false)
            applyTextStyle(style?.offerTextStyle, defaultOfferStyle)
        }

        val lp = LinearLayout.LayoutParams(
            LinearLayout.LayoutParams.WRAP_CONTENT,
            LinearLayout.LayoutParams.WRAP_CONTENT
        )

        originalPriceView.layoutParams = lp
        discountedPriceView.layoutParams = lp
        offerView.layoutParams = lp
        return createHorizontalRowView(context, discountedPriceView, originalPriceView, offerView)
    }

    fun loadRatingView(context: Context, adData: NativeAd, style: NativeAdCustomStyle?): View {
        val ratingView = TextView(context).apply {
            text = "${adData.customLabel0} ★"
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )

            applyTextStyle(style?.ratingTextStyle, defaultRatingStyle)
            setPadding(16, 8, 16, 8)

            val bgColor =
                style?.ratingTextStyle?.backgroundColor ?: defaultRatingStyle.backgroundColor
            bgColor?.let {
                background = GradientDrawable().apply {
                    cornerRadius = 10f
                    setColor(it)
                }
            }
        }

        val ratingCount = TextView(context).apply {
            text = "(${adData.customLabel1})"
            applyTextStyle(style?.ratingCountTextStyle, defaultRatingCountStyle)
        }

        return createHorizontalRowView(context, ratingView, ratingCount, null)
    }

    fun loadMembershipView(
        context: Context,
        adData: NativeAd,
        style: NativeAdCustomStyle?
    ): View {
        return TextView(context).apply {
            text = adData.customLabel3
            applyTextStyle(style?.membershipPriceTextStyle, defaultMembershipStyle)
        }
    }

    fun setAdLabel(context: Context, adLabelText: String?, adLabelAlignment: Int?): View {
        val alignment = adLabelAlignment ?: (Gravity.TOP or Gravity.START)
        if (!adLabelText.isNullOrEmpty()) {
            return ViewUtils.getAdLabelTextview(context, adLabelText, alignment, 5, 5)
        }
        return ViewUtils.getEmptyView(context)
    }

    private fun createHorizontalRowView(
        context: Context,
        view1: View?,
        view2: View?,
        view3: View?
    ): View {
        return try {
            LinearLayout(context).apply {
                orientation = LinearLayout.HORIZONTAL
                layoutParams = LinearLayout.LayoutParams(
                    LinearLayout.LayoutParams.WRAP_CONTENT,
                    LinearLayout.LayoutParams.WRAP_CONTENT
                ).apply {
                    topMargin = 10
                    bottomMargin = 10
                }

                listOf(view1, view2, view3).forEachIndexed { index, view ->
                    view?.let {
                        it.layoutParams = LinearLayout.LayoutParams(
                            LinearLayout.LayoutParams.WRAP_CONTENT,
                            LinearLayout.LayoutParams.WRAP_CONTENT
                        )
                        it.setPadding(5, 0, if (index < 2) 10 else 0, 0)
                        addView(it)
                    }
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
            View(context)
        }
    }

    private fun TextView.applyTextStyle(style: TextStyle?, fallback: TextStyle?) {
        val finalStyle = style ?: fallback
        finalStyle?.let {
            it.font?.let { typeface -> this.typeface = typeface }
            it.color?.let { color -> setTextColor(color) }
            it.backgroundColor?.let { backgroundColor -> setBackgroundColor(backgroundColor) }
            it.textSizeSp?.let { size -> textSize = size }
        }
    }
}