package com.ai.osmos.ads.views.helpers

import android.content.Context
import android.graphics.Color
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.LinearLayout
import com.ai.osmos.ads.renderer.loaders.ImageLoader
import com.ai.osmos.ads.renderer.loaders.VideoLoader
import com.ai.osmos.ads.views.style.NativeAdCustomStyle
import com.ai.osmos.core.Config
import com.ai.osmos.models.ads.AdElement
import com.ai.osmos.models.ads.NativeAd
import com.ai.osmos.models.ads.VideoAd
import com.ai.osmos.models.enums.NativeAdLayoutType
import com.ai.osmos.tracking.tracker.AdTrackerInterface
import com.ai.osmos.utils.error.ErrorCallback
import com.ai.osmos.utils.error.ExceptionHandler
import com.ai.osmos.utils.error.OsmosError
import com.ai.osmos.utils.ui.ViewUtils
import kotlinx.coroutines.CoroutineScope

/**
 * Internal renderer for vertical native ad layouts.
 * Contains the extracted vertical rendering logic from BaseNativeAdView.
 */
internal class VerticalNativeAdViewRenderer(
    private val coroutineScope: CoroutineScope,
    private val adTracker: AdTrackerInterface,
    private val config: Config
) : NativeAdViewRenderer {

    private val imageLoader = ImageLoader(coroutineScope, adTracker)
    private val videoLoader = VideoLoader(coroutineScope, adTracker)
    private val viewHelper = NativeAdViewHelper()
    private var errorCallback: ErrorCallback? = null

    override fun setErrorCallback(callback: ErrorCallback?) {
        this.errorCallback = callback
        imageLoader.setErrorCallback(callback)
        videoLoader.setErrorCallback(callback)
    }

    override fun clearErrorCallback() {
        this.errorCallback = null
        imageLoader.clearErrorCallback()
        videoLoader.clearErrorCallback()
    }

    private fun errorLog(message: String, errorType: OsmosError, throwable: Throwable? = null) {
        val exception = ExceptionHandler(errorType, message, throwable)
        errorCallback?.onError(exception.errorCode, exception.errorMessage, exception)
    }

    override fun renderVerticalView(
        context: Context,
        adData: NativeAd,
        width: Int,
        height: Int?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        isCarouselView: Boolean
    ): View {
        val (screenWidth, screenHeight) = ViewUtils.getScreenHeightWidth(0.8f, 0.8f)
        val finalWidth = minOf(ViewUtils.dpToPx(context, width), screenWidth)

        val h: Int? = if (height != null) {
            ViewUtils.dpToPx(context, height)
        } else if (style?.mediaHeight != null) {
            ViewUtils.dpToPx(context, style.mediaHeight)
        } else {
            null
        }

        val finalHeight = minOf(h ?: screenHeight, screenWidth)
        val container = FrameLayout(context).apply {
            layoutParams = ViewGroup.LayoutParams(
                finalWidth,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
        }

        val contentLayout = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            setBackgroundColor(Color.WHITE)
            setPadding(12, 12, 12, 12)
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT,
                FrameLayout.LayoutParams.MATCH_PARENT
            )
        }

        // Tag layout view
        val topTagRow = viewHelper.loadTagBudgeView(context, style, customBadgeView)

        val mediaContentLayout = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                finalHeight / 2,
            )
            gravity = Gravity.CENTER
            setPadding(0, 0, 10, 0)
        }

        if (adData.imageUrl != null) {
            val imageView = imageLoader.createImageView(
                context,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                isClamped = false
            ).apply {
                layoutParams = LinearLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT
                )
                this.layoutParams = layoutParams
                Gravity.CENTER
            }
            imageLoader.loadNativeImage(adData.imageUrl.toString(), imageView)
            mediaContentLayout.addView(imageView)
        } else {
            val videoAd = VideoAd(
                au = "",
                rank = 3,
                clickTrackingUrl = "",
                impressionTrackingUrl = "",
                uclid = adData.uclid ?: "",
                crt = "",
                elements = AdElement(
                    type = "VIDEO",
                    height = finalHeight / 2,
                    width = ViewGroup.LayoutParams.MATCH_PARENT,
                    value = adData.videoUrl ?: "",
                ),
                adMetadata = emptyMap(),
                clientId = (config.clientId).toLong()
            )

            val videoView = videoLoader.createVideoViewForNativeAd(
                context,
                videoAd,
                finalWidth,
                finalHeight / 2,
                adData.cliUbid.toString()
            )
            mediaContentLayout.addView(videoView)
        }

        // Create view components
        val titleView = viewHelper.loadTitleView(context, adData, style)
        val skuDescriptionView = viewHelper.loadSkuDescriptionView(context, adData, style)
        val deliveryDetailsView = viewHelper.loadDeliveryDetailView(context, adData, style)
        val priceRow = viewHelper.loadPriceRowView(context, adData, style)
        val ratingRow = viewHelper.loadRatingView(context, adData, style)
        val membershipPrice = viewHelper.loadMembershipView(context, adData, style)

        // Add views on container
        val nativeStyle = style ?: NativeAdCustomStyle()
        when (nativeStyle.adType) {
            NativeAdLayoutType.Small ->
                contentLayout.addViewsInOrder(
                    mediaContentLayout,
                    titleView,
                    priceRow,
                )

            NativeAdLayoutType.Medium ->
                contentLayout.addViewsInOrder(
                    mediaContentLayout,
                    titleView,
                    ratingRow,
                    priceRow
                )

            NativeAdLayoutType.Large ->
                contentLayout.addViewsInOrder(
                    mediaContentLayout,
                    titleView,
                    skuDescriptionView,
                    deliveryDetailsView,
                    priceRow,
                    membershipPrice,
                    ratingRow,
                )
        }

        viewHelper.addCTAButton(
            context,
            adData,
            customCtaView,
            style,
            contentLayout,
            isCarouselView
        )

        container.addView(contentLayout)
        container.addView(topTagRow)
        container.addView(viewHelper.setAdLabel(context, adLabelText, adLabelAlignment))

        return container
    }

    override fun renderHorizontalView(
        context: Context,
        adData: NativeAd,
        width: Int,
        height: Int?,
        adLabelText: String?,
        adLabelAlignment: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?,
        customBadgeView: View?,
        isCarouselView: Boolean
    ): View {
        // This renderer only handles vertical views
        // Horizontal views should be handled by HorizontalNativeAdViewRenderer
        throw UnsupportedOperationException("VerticalNativeAdViewRenderer does not support horizontal rendering")
    }

    override fun renderProductVerticalView(
        context: Context,
        adData: NativeAd,
        adType: NativeAdLayoutType,
        width: Int,
        height: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?
    ): View {
        val (screenWidth, screenHeight) = ViewUtils.getScreenHeightWidth(0.8f, 0.8f)
        val finalWidth = minOf(ViewUtils.dpToPx(context, width), screenWidth)

        val h: Int? = if (height != null) {
            ViewUtils.dpToPx(context, height)
        } else if (style?.mediaHeight != null) {
            ViewUtils.dpToPx(context, style.mediaHeight)
        } else {
            null
        }

        val finalHeight = minOf(h ?: screenHeight, screenWidth)
        val container = FrameLayout(context).apply {
            layoutParams = ViewGroup.LayoutParams(
                finalWidth,
                ViewGroup.LayoutParams.WRAP_CONTENT
            )
        }
        val contentLayout = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            setBackgroundColor(Color.WHITE)
            setPadding(12, 12, 12, 12)
            layoutParams = FrameLayout.LayoutParams(
                FrameLayout.LayoutParams.MATCH_PARENT,
                FrameLayout.LayoutParams.MATCH_PARENT
            )
        }

        val mediaContentLayout = LinearLayout(context).apply {
            orientation = LinearLayout.VERTICAL
            layoutParams = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                finalHeight / 2,
            )
            gravity = Gravity.CENTER
            setPadding(0, 0, 10, 0)
        }

        if (adData.imageUrl != null) {
            val imageView = imageLoader.createImageView(
                context,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                isClamped = false
            ).apply {
                layoutParams = LinearLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT
                )
                this.layoutParams = layoutParams
                Gravity.CENTER
            }
            imageLoader.loadNativeImage(adData.imageUrl.toString(), imageView)
            mediaContentLayout.addView(imageView)
        } else {
            val videoAd = VideoAd(
                au = "",
                rank = 0,
                clickTrackingUrl = "",
                impressionTrackingUrl = "",
                uclid = adData.uclid ?: "",
                crt = "",
                elements = AdElement(
                    type = "VIDEO",
                    height = finalHeight / 2,
                    width = ViewGroup.LayoutParams.MATCH_PARENT,
                    value = adData.videoUrl ?: ""
                ),
                adMetadata = emptyMap(),
                clientId = (config.clientId).toLong()
            )

            val videoView = videoLoader.createVideoViewForNativeAd(
                context,
                videoAd,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
                adData.cliUbid.toString()
            )
            mediaContentLayout.addView(videoView)
        }

        // Create view components
        val titleView = viewHelper.loadTitleView(context, adData, style)
        val skuDescriptionView = viewHelper.loadSkuDescriptionView(context, adData, style)
        val deliveryDetailsView = viewHelper.loadDeliveryDetailView(context, adData, style)
        val priceRow = viewHelper.loadPriceRowView(context, adData, style)
        val ratingRow = viewHelper.loadRatingView(context, adData, style)
        val membershipPrice = viewHelper.loadMembershipView(context, adData, style)

        // Add views on container based on adType parameter
        when (adType) {
            NativeAdLayoutType.Small ->
                contentLayout.addViewsInOrder(
                    mediaContentLayout,
                    titleView,
                    priceRow,
                )

            NativeAdLayoutType.Medium ->
                contentLayout.addViewsInOrder(
                    mediaContentLayout,
                    titleView,
                    ratingRow,
                    priceRow
                )

            NativeAdLayoutType.Large ->
                contentLayout.addViewsInOrder(
                    mediaContentLayout,
                    titleView,
                    skuDescriptionView,
                    deliveryDetailsView,
                    priceRow,
                    membershipPrice,
                    ratingRow,
                )
        }

        viewHelper.addCTAButton(context, adData, customCtaView, style, contentLayout, true)
        container.addView(contentLayout)
        return container
    }

    override fun renderProductHorizontalView(
        context: Context,
        adData: NativeAd,
        adType: NativeAdLayoutType,
        width: Int,
        height: Int?,
        style: NativeAdCustomStyle?,
        customCtaView: View?
    ): View {
        // This renderer only handles vertical views
        throw UnsupportedOperationException("VerticalNativeAdViewRenderer does not support horizontal rendering")
    }

    // Extension function to add views in order, skipping null views
    private fun LinearLayout.addViewsInOrder(vararg views: View?) {
        views.forEach { view ->
            view?.let { addView(it) }
        }
    }
}